-- =============================================
-- Author:		Jeremy Formby
-- Create date: 3/05/2016
-- Version:		1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON

GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_ResponseHeader_StockCodeIdByResponseId')
	EXEC('CREATE FUNCTION [dbo].[FNC_ResponseHeader_StockCodeIdByResponseId]() RETURNS bit AS BEGIN RETURN 0 END')
GO
-- =============================================
-- Author:		Markp
-- Create date: 25/03/2009
-- Description:	Gets stockcodeid for individual response
-- Version:		6
-- =============================================
-- JT TFS 7830 - Also link through raw stockcode datalookup
-- JE TFS 8684 - added filter so deleted Header items are ignored
-- MP TFS 12720: Added check on ResponseID to mitigate duplicates
-- JF TFS 15314 - timeout issues - changed duplicate check to an outer join (Greencell had 100k records in @ResponseHeader)
-- JT TFS 78355 - Get StockcodeID by reponseID
-- Jf TFS 142691 -  added final product and ordered by listindex
-- =============================================
ALTER FUNCTION [dbo].[FNC_ResponseHeader_StockCodeIdByResponseId] 
(
	@ResponseId int
)
RETURNS int
AS
BEGIN
	--Add header where stock code selected
	
	RETURN (
		SELECT TOP 1 StockCodeId FROM (
			SELECT     1 AS Source, QMS_Response.ResponseID, QMS_ResponseHeader.Header_SourceID AS StockCodeId, QMS_AssessmentHeader.ListIndex
			FROM        QMS_Response 
			INNER JOIN	QMS_ResponseHeader ON QMS_Response.ResponseID = QMS_ResponseHeader.ResponseID 
			INNER JOIN	QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
			WHERE     (QMS_Response.RecordStatus < 201) 
			AND		(QMS_AssessmentHeader.RecordStatus < 201)
			AND		(QMS_ResponseHeader.Header_SourceID IS NOT NULL) 
			AND		(QMS_AssessmentHeader.DataTypeID = 9) 
			AND		(QMS_AssessmentHeader.DataLookupID IN (2, 35, 69)) 
			AND QMS_Response.ResponseID = @ResponseId

			--Add consignment item headers
			UNION ALL
			SELECT  2,  QMS_Response.ResponseID, PM_VW_ConsignmentItems.StockCodeID, QMS_AssessmentHeader.ListIndex
			FROM         QMS_Response INNER JOIN
							  QMS_ResponseHeader ON QMS_Response.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							  QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
							  PM_VW_ConsignmentItems ON QMS_ResponseHeader.Header_SourceID = PM_VW_ConsignmentItems.ConsignmentItemID
			WHERE	(QMS_Response.RecordStatus < 201) 
			AND		(QMS_AssessmentHeader.RecordStatus < 201) 
			AND		(QMS_AssessmentHeader.DataTypeID = 9) 
			AND		(QMS_AssessmentHeader.DataLookupID = 6) 
			AND QMS_Response.ResponseID = @ResponseId
	
			--Add delivery item headers
			UNION ALL
			SELECT  3,   QMS_Response.ResponseID, PM_VW_ConsignmentItems.StockCodeID, QMS_AssessmentHeader.ListIndex
			FROM         QMS_Response INNER JOIN
							  QMS_ResponseHeader ON QMS_Response.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							  QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
							  PM_VW_GoodsIntake ON QMS_ResponseHeader.Header_SourceID = PM_VW_GoodsIntake.GoodsIntakeID INNER JOIN
							  PM_VW_ConsignmentItems ON PM_VW_GoodsIntake.ConsignmentItemID = PM_VW_ConsignmentItems.ConsignmentItemID
			WHERE   (QMS_Response.RecordStatus < 201) 
			AND		(QMS_AssessmentHeader.RecordStatus < 201) 
			AND		(QMS_AssessmentHeader.DataTypeID = 9) 
			AND		(QMS_AssessmentHeader.DataLookupID = 8) 
			AND QMS_Response.ResponseID = @ResponseId
		) as _
		WHERE _.StockCodeId > 0
		ORDER BY Source, ListIndex
	)
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_User_ResponseList')
	EXEC('CREATE FUNCTION [dbo].[FNC_User_ResponseList]() RETURNS @x TABLE (y bit) AS BEGIN RETURN END')
GO
-- =============================================
-- Author:		Markp
-- Create date: 17/08/2009
-- Description:	Returns list of responses that user can access
-- Version: 	21
-- =============================================
-- JT TFS 4194 - 14/08/2012 - When stock code is restricted, this now returns checks where no stock code is specified in the header.
-- JE TFS 4622 - returns QAFlagID based on ReportQAFlag config setting
-- JT TFS 5801 - performance improvements
-- JE TFS 5905 - updated to make use of Reporting Warehouse table
-- MP TFS 6031 - changed non-admin filter to ensure that temp checks are not included
-- MP TFS 6152 - fix to handle supplier site permissions
-- JT TFS 6689 - FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7612 - Void checks visibility is controlled by UserSetting for specific reports.
-- JE TFS 13-3570: added PRIMARY KEY to @Table_StockCode to improve performance
-- MP TFS 13-12783: fixed issue with ambiguous Location permission linking
-- jf tfs 16880 - UBE Check view performance for user with 2000+ stock code permissions - added lots of filters for updates like ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)
-- MP TFS 23409 - Added conditions in the CanUseWarehouse block so that Suppliers (userlevel=9) use supplier site code and non supplier (userlevel<>9) use locations
-- SS TFS 13-49452: Added logic so that when a client is using datawarehousing, if they are filtering on today's date, the warehouse is used for all dates up to today's date AND then today's date data
-- is worked out without the warehouse being used. This prevents timeouts FROM happening.
-- SS TFS 13-49897: Added ISNULL(Uploaded, EndDate) so that EndDate will be used if Uploaded contains a NULL Value
-- PH TSF 13-61027 : v16 18/11/2016 - Bug Fix.  When including checks from the datawarehouse for todays date, an assumption was made that the datawarehouse had been updated before @DateFrom.  
--                   However, if this was not the case then checks were included from the past (but after the warehouse had been updated).  The greater of the datawarehouse update time and @DateFrom is now used.
-- JT TFS 13-82627 : Fix issue of Uploaded results appearing on QC Checks screen not in the date range specified.
-- JT TFS 13-88313 : Only merge results from warehouse within the date range specified.
-- JF TFS 108475 : enddate is local time so calculate the offset before doing date comparrisons
-- TS TFS 125176 : Update checks filter to include checjs that haven't had DW run on them yet 
-- JT TFS 142691 : replace stockcode permissions with single function
-- =============================================
ALTER FUNCTION [dbo].[FNC_User_ResponseList]
(
	@UserID int,
	@DateFrom datetime, 
	@DateTo datetime
)
RETURNS 
@Response TABLE 
(
	ResponseID int,
	ResponseStatus tinyint,
	QAFlagID int
)
AS
BEGIN

--declare @UserID int, @DateFrom datetime,	@DateTo datetime
--select @UserID = 836, @DateFrom = '1 Jun 2013 00:00:00',  @DateTo = '21 May 2014 23:59:59'

	DECLARE @UnfilteredResponse TABLE (
		ResponseID int PRIMARY KEY,
		ResponseStatus tinyint,
		LocationID int NULL,
		CustomerID int NULL,
		SupplierID int NULL,
		StockCodeID int NULL,
		QAFlagID int NULL
	)

	-- This will contain responses which are not in the warehouse, typically those with today's date.
	DECLARE @UnfilteredResponseNotInWarehouse TABLE (
		ResponseID int PRIMARY KEY,
		ResponseStatus tinyint,
		LocationID int NULL,
		CustomerID int NULL,
		SupplierID int NULL,
		StockCodeID int NULL,
		QAFlagID int NULL
	)

	--get date reporting warehouse table last built
	DECLARE @WarehouseBuildDate datetime
	SET @WarehouseBuildDate = (SELECT Updated FROM APP_Config WHERE OptionID = 78)

	DECLARE @UserSettingDisplayVoids bit 
	SET @UserSettingDisplayVoids = ISNULL(cast(dbo.FNC_GetUserSetting(@UserID, 'DisplayVoidChecks') as bit), 0)

	-- Lookup user level
	DECLARE @UserLevel int, @RecordStatus tinyint
	SELECT     @UserLevel = ISNULL(UserLevel,0)
	FROM         QMS_User
	WHERE     (UserID = @UserID)
	IF @UserSettingDisplayVoids = 1 -- Include all recordstatuses when we set a report to do the same by setting the User Setting.
		SET @RecordStatus = 255 -- maximum possible. 
	ELSE IF @UserLevel = 1  --Admin can always see hidden/void checks.
		SET @RecordStatus = 201
	ELSE
		SET @RecordStatus = 100

	-- Permissions that this user has.
	DECLARE @PermissionLink TABLE (
		UserID int, SupplierID int, CustomerID int, LocationID int, StockCodeID int, 
		StockCodeSubGroup nvarchar(100), BusinessGroup nvarchar(100)
	)
	INSERT INTO	@PermissionLink (UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup)
	SELECT	UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup
	FROM	VW_PermissionLink PermissionLink
	WHERE	PermissionLink.PermissionAllow = 1 AND PermissionLink.PermissionType IN (101,102,103,104,105,106) AND UserID = @UserID

	DECLARE @Table_StockCode TABLE (
		StockCodeID int PRIMARY KEY
	)
	
	--check for permissions in three levels of hierarchy: Business Group, Sub Group, Stock Code, and translate to StockCodeID
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		BusinessGroup IN (SELECT BusinessGroup FROM @PermissionLink WHERE BusinessGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeSubGroup IN (SELECT StockCodeSubGroup FROM @PermissionLink WHERE StockCodeSubGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeID IN (SELECT StockCodeID FROM @PermissionLink WHERE StockCodeID IS NOT NULL)

	-- The datawarehouse can still be used when the user filters after the last warehouse build, but it will not contain much (if any) useful checks.
	-- We insert all the recent checks into @UnfilteredResponse so that the datawarehouse can still be used when the user filters on today's date.
	-- Just because the date range in entirely contained within the warehouse, doesn't mean that we only search in the warehouse. 
	-- Checks uploaded after the last warehouse build, but whose EndDate is within that range need to be included, as do incomplete checks (ResponseStatus 0)
	IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1
	BEGIN
		DECLARE @override bit 
		SET @override = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)
		
		--get the min offset to catch all. Alternative method to calc each response UTC using UTCOffset would be too slow
		declare @MinOffset int 
		select @MinOffset = ISNULL(MIN(utcoffset),0) from qms_response where EndDate > DATEADD(MONTH,-3,GETDATE())

		DECLARE @ResponseFromDate datetime
		SELECT @ResponseFromDate = CASE WHEN @WarehouseBuildDate > @DateFrom THEN @WarehouseBuildDate ELSE @DateFrom END
		SELECT @ResponseFromDate = DATEADD(Hour,@MinOffset,@ResponseFromDate)
			
		-- Get the majority of checks which will not be in the warehouse - the ones added after the warehouse has been built.
		INSERT INTO @UnfilteredResponseNotInWarehouse (ResponseID, ResponseStatus, QAFlagID) 
		SELECT ResponseID, ResponseStatus,
				--check config value to use Actual or Override QAFlagID
				CASE WHEN @override = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
		FROM	QMS_Response
		WHERE	(RecordStatus < @RecordStatus) 
		AND EndDate BETWEEN @ResponseFromDate AND @DateTo


		-- Make sure we include the checks uploaded after the warehouse was last built, but within the entire date range. They will not yet be in the warehouse.
		-- Also, we have to make sure that incomplete checks get included, since they are never put in the warehouse.
		INSERT INTO @UnfilteredResponseNotInWarehouse (ResponseID, ResponseStatus, QAFlagID) 
		SELECT ResponseID, ResponseStatus,
				--check config value to use Actual or Override QAFlagID
				CASE WHEN @override = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
		FROM QMS_Response
		WHERE	(RecordStatus < @RecordStatus) 
		AND (Uploaded > @WarehouseBuildDate or Updated > @WarehouseBuildDate OR ResponseStatus = 0)
		AND EndDate BETWEEN @DateFrom AND @DateTo
		AND ResponseID NOT IN (SELECT ResponseID FROM @UnfilteredResponseNotInWarehouse)


		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
			UPDATE	UnfilteredResponseTodaysDate
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponseTodaysDate.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 53) and
			ISNULL(UnfilteredResponseTodaysDate.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
			UPDATE	UnfilteredResponseTodaysDate
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponseTodaysDate.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 4) and 
			ISNULL(UnfilteredResponseTodaysDate.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
			UPDATE	UnfilteredResponseTodaysDate
			SET		SupplierID = QMS_ResponseHeader.Header_SourceID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponseTodaysDate.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1) AND
			ISNULL(UnfilteredResponseTodaysDate.SupplierID,0) <> ISNULL(QMS_ResponseHeader.Header_SourceID,0)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
			UPDATE	UnfilteredResponseTodaysDate
			SET		CustomerID = PM_VW_StockCode.CustomerID
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponseTodaysDate.ResponseID = HeaderStockCode.ResponseID INNER JOIN
						PM_VW_StockCode ON PM_VW_StockCode.StockCodeID = HeaderStockCode.StockCodeID
			WHERE ISNULL(UnfilteredResponseTodaysDate.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

		IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
		BEGIN
			UPDATE	UnfilteredResponseTodaysDate
			SET		StockCodeID = dbo.FNC_ResponseHeader_StockCodeIdByResponseId(UnfilteredResponseTodaysDate.ResponseID)
			FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate 
			WHERE ISNULL(UnfilteredResponseTodaysDate.StockCodeID,0) <> ISNULL(dbo.FNC_ResponseHeader_StockCodeIdByResponseId(UnfilteredResponseTodaysDate.ResponseID),0)

			--UPDATE	UnfilteredResponseTodaysDate
			--SET		StockCodeID = HeaderStockCode.StockCodeID
			--FROM	@UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate INNER JOIN
			--			dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponseTodaysDate.ResponseID = HeaderStockCode.ResponseID
			--WHERE ISNULL(UnfilteredResponseTodaysDate.StockCodeID,0) <> ISNULL(HeaderStockCode.StockCodeID,0)
		END

		-- Include checks from the warehouse.
		INSERT INTO	@UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
		SELECT		RPT_Warehouse_Response.ResponseID, QMS_Response.ResponseStatus, RPT_Warehouse_Response.QAFlagID --Actual or Override QAFlag setting checked when building warehouse table
		FROM		RPT_Warehouse_Response 
		INNER JOIN  QMS_Response ON RPT_Warehouse_Response.ResponseID = QMS_Response.ResponseID
		WHERE		(QMS_Response.RecordStatus < @RecordStatus) 
		AND (RPT_Warehouse_Response.EndDate BETWEEN @DateFrom AND @DateTo)
		AND NOT EXISTS (SELECT UnfilteredResponseTodaysDate.ResponseID FROM @UnfilteredResponseNotInWarehouse UnfilteredResponseTodaysDate
						WHERE UnfilteredResponseTodaysDate.ResponseID = RPT_Warehouse_Response.ResponseID)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
			UPDATE	UnfilteredResponse
			SET	LocationID = RPT_Warehouse_Response.SupplierSiteID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
			WHERE ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(RPT_Warehouse_Response.SupplierSiteID,0)
 
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
			UPDATE	UnfilteredResponse
			SET	LocationID = RPT_Warehouse_Response.LocationID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
			WHERE ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(RPT_Warehouse_Response.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
			UPDATE	UnfilteredResponse
			SET		SupplierID = RPT_Warehouse_Response.SupplierID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
			WHERE ISNULL(UnfilteredResponse.SupplierID,0) <> ISNULL(RPT_Warehouse_Response.SupplierID,0)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
			UPDATE	UnfilteredResponse
			SET		CustomerID = PM_VW_StockCode.CustomerID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID INNER JOIN
						PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID
			WHERE ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

		IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
			UPDATE	UnfilteredResponse
			SET		StockCodeID = RPT_Warehouse_Response.StockCodeID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID

		INSERT INTO @UnfilteredResponse(ResponseID, ResponseStatus, LocationID, CustomerID, SupplierID, StockCodeID, QAFlagID)
		SELECT ResponseID, ResponseStatus, LocationID, CustomerID, SupplierID, StockCodeID, QAFlagID FROM @UnfilteredResponseNotInWarehouse
	END
	ELSE --don't use warehouse table
	BEGIN
		DECLARE @OverrideOption bit 
		SET @OverrideOption = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)
			
		INSERT INTO @UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
		SELECT ResponseID, ResponseStatus,
				--check config value to use Actual or Override QAFlagID
				CASE WHEN @OverrideOption = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
		FROM	QMS_Response
		WHERE	(RecordStatus < @RecordStatus) AND (EndDate BETWEEN @DateFrom AND @DateTo)
			
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
			UPDATE	UnfilteredResponse
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 53) and
			ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
			UPDATE	UnfilteredResponse
			SET		LocationID = Location.LocationID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
						PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 4) and 
			ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(Location.LocationID,0)

		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
			UPDATE	UnfilteredResponse
			SET		SupplierID = QMS_ResponseHeader.Header_SourceID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
						QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
			WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1) AND
			ISNULL(UnfilteredResponse.SupplierID,0) <> ISNULL(QMS_ResponseHeader.Header_SourceID,0)
				
		IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
			UPDATE	UnfilteredResponse
			SET		CustomerID = PM_VW_StockCode.CustomerID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID INNER JOIN
						PM_VW_StockCode ON PM_VW_StockCode.StockCodeID = HeaderStockCode.StockCodeID
			WHERE ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

		IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
			UPDATE	UnfilteredResponse
			SET		StockCodeID = HeaderStockCode.StockCodeID
			FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
						dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID
			WHERE ISNULL(UnfilteredResponse.StockCodeID,0) <> ISNULL(HeaderStockCode.StockCodeID,0)
	END

	-- Only return responses which match the filter specified, or for which we do not have a filter.
	-- This means that admin users (any user with no permissions set) can see all responses.
	INSERT INTO @Response (ResponseID, ResponseStatus, QAFlagID)
	SELECT	ResponseID, ResponseStatus, QAFlagID
	FROM	@UnfilteredResponse UnfilteredResponse
	WHERE (	NOT EXISTS (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL)
		OR	LocationID IN (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
		OR	SupplierID IN (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL)
		OR	CustomerID IN (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL) )
	AND (	StockCodeID IS NULL -- Allow unlinked stockcode responses
		OR	NOT EXISTS (SELECT StockCodeID FROM @Table_StockCode)
		OR	StockCodeID IN (SELECT StockCodeID FROM @Table_StockCode) )
	
	RETURN
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_CorrectiveActionDetail')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_CorrectiveActionDetail] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 23/06/2009
-- Description:	Gets data for Corrective Action Detail Report
-- Version: 10
-- =============================================
-- JE TFS 2745 - updated to look up CAStatus resources
-- JE TFS 2769 - added filtering by AssignSupplierID
-- JE TFS 6004 - now uses FNC_User_CorrectiveActionList for user CA filtering
-- MP TFS 6004 - now uses FNC_User_CorrectiveActionList for report by supplier & group
-- MP TFS 6749 - fix to ensure that closed CAs are also picked up
-- JT TFS 7586 - Stop non-outstanding only query filtered by user from returning all corrective actions.
-- JT TFS 7586 - Supplier users shows AssignSupplierID links.
-- JT TFS 7586 - Use FilterID user specified in supplier link for FilterBy User.
-- JE TFS 13-12643: replaced use of FNC_User_CorrectiveActionList with QMS_CorrectiveActionUser
-- CJP - HOTFIX - Report not returning data where datelogged is null so using ISNULL to substitute in DateOccured which seems to always be populated
-- =============================================
ALTER PROCEDURE [dbo].[RPT_CorrectiveActionDetail]
	@UserID int,
	@DateFrom datetime, 
	@DateTo datetime,
	@FilterBy tinyint, --1=user, 2=group
	@FilterID int,
	@OutstandingOnly bit
AS

SET NOCOUNT ON
--Fill temp table with selected CAs
DELETE FROM TEMP_Selected
WHERE     (UserID = @UserID)

DECLARE @User_CA TABLE (CorrectiveActionID INT)

INSERT INTO @User_CA ( CorrectiveActionID )
SELECT	QMS_CorrectiveActionUser.CorrectiveActionID
FROM	QMS_CorrectiveActionUser
		INNER JOIN QMS_CorrectiveAction ON QMS_CorrectiveActionUser.CorrectiveActionID = QMS_CorrectiveAction.CorrectiveActionID
WHERE	(QMS_CorrectiveActionUser.UserID = CASE WHEN @FilterBy = 1 THEN @FilterID ELSE @UserID END)
		AND (QMS_CorrectiveAction.CAStatusID IN (1,2,3,6))

IF @OutstandingOnly = 0 --Add closed CAs too
	INSERT INTO @User_CA ( CorrectiveActionID )
	SELECT	QMS_CorrectiveActionUser.CorrectiveActionID
	FROM	QMS_CorrectiveActionUser
			INNER JOIN QMS_CorrectiveAction ON QMS_CorrectiveActionUser.CorrectiveActionID = QMS_CorrectiveAction.CorrectiveActionID
	WHERE	(QMS_CorrectiveActionUser.UserID = CASE WHEN @FilterBy = 1 THEN @FilterID ELSE @UserID END)
			AND (QMS_CorrectiveAction.CAStatusID NOT IN (1,2,3,6))

IF @FilterBy=1 --User
BEGIN
--1. if CA assigned to that user
--2. If CA assigned to a group that the user is in
--3. If CA assigned to a supplier that the user has permissions to see and they have a supplier user level (this should be handled in "permissions to see - FNC_User_CorrectiveActionList")

	DECLARE @IsSupplierUser bit
	SELECT @IsSupplierUser = CASE WHEN [dbo].[FNC_GetUserOrderLevel](@FilterID) = 9 THEN 1 ELSE 0 END

	INSERT INTO TEMP_Selected
                      (UserID, SelectedID)
	SELECT     @UserID AS UserID, QMS_CorrectiveAction.CorrectiveActionID
	FROM        QMS_CorrectiveAction 
	INNER JOIN	@User_CA AS UserCAs ON QMS_CorrectiveAction.CorrectiveActionID = UserCAs.CorrectiveActionID
	WHERE   (ISNULL(QMS_CorrectiveAction.DateLogged,QMS_CorrectiveAction.DateOccurred) BETWEEN @DateFrom AND @DateTo) 
	AND		(QMS_CorrectiveAction.RecordStatus < 201) 
	
	AND (	QMS_CorrectiveAction.AssignUserID = @FilterID
		OR 
			QMS_CorrectiveAction.NotificationGroupID IN (SELECT NotificationGroupID FROM QMS_NotificationGroupUser where UserID = @FilterID)
		OR 
			(@IsSupplierUser = 1 AND QMS_CorrectiveAction.AssignSupplierID IN (SELECT SupplierID FROM VW_PermissionLink WHERE SupplierID IS NOT NULL AND UserID = @FilterID))
	)
	AND (	(@OutstandingOnly = 0)
			OR
			((@OutstandingOnly = 1) AND (QMS_CorrectiveAction.CAStatusID < 4))
	)
END
	
IF @FilterBy=2 --Group
	INSERT INTO TEMP_Selected
                      (UserID, SelectedID)
	SELECT     @UserID AS UserID, QMS_CorrectiveAction.CorrectiveActionID
	FROM         QMS_CorrectiveAction INNER JOIN
						  @User_CA AS UserCAs ON QMS_CorrectiveAction.CorrectiveActionID = UserCAs.CorrectiveActionID
	WHERE   (ISNULL(QMS_CorrectiveAction.DateLogged,QMS_CorrectiveAction.DateOccurred) BETWEEN @DateFrom AND @DateTo) 
	AND		(RecordStatus < 201)  
	AND		(NotificationGroupID = @FilterID)
	AND	(	(@OutstandingOnly = 0) 
			OR
			((@OutstandingOnly = 1) AND (CAStatusID < 4))
	)

IF @FilterBy=3 --Supplier
	INSERT INTO TEMP_Selected
						(UserID, SelectedID)
	SELECT		@UserID AS UserID, QMS_CorrectiveAction.CorrectiveActionID
	FROM		QMS_CorrectiveAction INNER JOIN
						  @User_CA AS UserCAs ON QMS_CorrectiveAction.CorrectiveActionID = UserCAs.CorrectiveActionID
	WHERE	(ISNULL(QMS_CorrectiveAction.DateLogged,QMS_CorrectiveAction.DateOccurred) BETWEEN @DateFrom AND @DateTo) 
	AND		(RecordStatus < 201) 
	AND		(AssignSupplierID = @FilterID) 
	AND	(	(@OutstandingOnly = 0) 
			OR
            ((@OutstandingOnly = 1) AND (CAStatusID < 4))
    )

SET NOCOUNT OFF

SELECT     @DateFrom AS DateFrom, @DateTo AS DateTo, QMS_CorrectiveAction.CorrectiveActionID, QMS_CorrectiveAction.CATitle, 
                  QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, QMS_CorrectiveAction.CARequired,
					  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS CAStatus, 
                      APP_PriorityLevel.PriorityLevel, ISNULL(QMS_CorrectiveAction.DateLogged,QMS_CorrectiveAction.DateOccurred) as DateLogged, QMS_CorrectiveAction.DateDeadline, 
                      QMS_CorrectiveActionType.CorrectiveActionType, QMS_CorrectiveActionEscalation.ListIndex AS EscalationLevel, 
                      ISNULL(QMS_NotificationGroup.NotificationGroup, QMS_User_Assign.FullName) AS AssignTo, QMS_User_Logged.FullName AS LoggedBy, 
                      QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, QMS_CorrectiveAction.DateCompleted, 
                      QMS_User_Completed.FullName AS CompletedBy, APP_PriorityLevel.ListIndex AS PriorityOrder, QMS_CorrectiveAction.DateSignedOff, 
                      QMS_User.FullName AS SignedOffBy, QMS_CorrectiveAction.DateFailed
FROM         QMS_CorrectiveAction INNER JOIN
                      APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID INNER JOIN
                      TEMP_Selected ON QMS_CorrectiveAction.CorrectiveActionID = TEMP_Selected.SelectedID LEFT OUTER JOIN
                      QMS_User ON QMS_CorrectiveAction.SignOff_UserID = QMS_User.UserID LEFT OUTER JOIN
                      QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
                      QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
                      QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
                      QMS_CorrectiveActionEscalation ON 
                      QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
                      QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
                      QMS_User AS QMS_User_Completed ON QMS_CorrectiveAction.Completed_UserID = QMS_User_Completed.UserID LEFT OUTER JOIN
                      APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID
WHERE     (TEMP_Selected.UserID = @UserID)
ORDER BY QMS_CorrectiveAction.DateLogged, PriorityOrder
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'PM_QRY_Consignment_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[PM_QRY_Consignment_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 16/07/2008
-- Description:	Gets list of consignments
-- Version:	5
-- =============================================
-- JF TFS 78132 -can use barcode
-- JF output integrated data
-- JF TFS 92518 - filter on Location if the trigger is enabled
-- JF TFS 92518 - format Consignment to cater for older, non integarated PO data using orderref or ConsignmentNo
-- JF TFS 104066 GYFD - Allow Header To Work with Barcode Prefix
-- JF TFS can pre filter using Text4 - controlled using triggers
-- =============================================
ALTER PROCEDURE [dbo].[PM_QRY_Consignment_Combo]
	@SupplierID int = null,
	@Barcode nvarchar(50) =null,
	@LocationID INT=null,
	@Text4 nvarchar(50)=null
AS

	DECLARE @DownloadConsignmentDays int, @DateFrom datetime
	SET @DownloadConsignmentDays = ISNULL(dbo.FNC_GetOptionValue(101),0)
	
	IF @DownloadConsignmentDays = 0 
		SELECT @DateFrom = NULL
	ELSE
		SET @DateFrom = DATEADD(day,-@DownloadConsignmentDays,GETDATE())

	
	if exists(select * from APP_DataLookupTrigger where datalookupid_Trigger = 4 and DataLookupID_Update = 5 and enabled = 1)
		--filter PO based on Location
		SELECT  PM_VW_Consignment.ConsignmentID, E_PO AS Consignment
		FROM         PM_VW_Consignment 
		WHERE     (LocationID= @LocationID) and (DateRequired > @DateFrom OR DateRequired IS NULL or @DateFrom IS NULL)
		ORDER BY E_PO DESC

	if exists(select * from APP_DataLookupTrigger where datalookupid_Trigger = 93 and DataLookupID_Update = 5 and enabled = 1)
		--filter PO based on Consignmentitem.Text4
		SELECT  distinct PM_VW_Consignment.ConsignmentID, E_PO AS Consignment
		FROM         PM_VW_Consignment 
		INNER JOIN PM_VW_ConsignmentItems on PM_VW_Consignment.consignmentID = PM_VW_ConsignmentItems.consignmentID
		WHERE     
		(PM_VW_ConsignmentItems.Text4 = @Text4 and @Text4 IS NOT NULL)
		or
		((PM_VW_Consignment.DateRequired > @DateFrom OR @DateFrom IS NULL) and @Text4 IS NULL )
		
		ORDER BY E_PO DESC

	ELSE if @Barcode IS NOT NULL
		BEGIN
			SELECT  DISTINCT   PM_VW_Consignment.ConsignmentID, ISNULL(E_PO,ISNULL(OrderRef, ConsignmentNo)) AS Consignment
			FROM         PM_VW_Consignment 
			INNER JOIN PM_VW_ConsignmentItems on PM_VW_Consignment.consignmentID = PM_VW_ConsignmentItems.consignmentID
			INNER JOIN PM_VW_GoodsIntake ON PM_VW_ConsignmentItems.consignmentitemID=PM_VW_GoodsIntake.consignmentitemid
			WHERE     (PM_VW_GoodsIntake.Barcode = dbo.FNC_BarcodeFormat(@Barcode))
			ORDER BY ISNULL(E_PO,ISNULL(OrderRef, ConsignmentNo)) DESC
		END

	ELSE --if exists(select * from APP_DataLookupTrigger where datalookupid_Trigger = 1 and DataLookupID_Update = 5 and enabled = 1)
		--filter PO based on supplier
		SELECT  ConsignmentID, ISNULL(E_PO,ISNULL(OrderRef, ConsignmentNo)) AS Consignment
		FROM         PM_VW_Consignment
		WHERE     
			(SupplierID = @SupplierID OR @SupplierID IS NULL) AND
			(DateRequired > @DateFrom OR DateRequired IS NULL or @DateFrom IS NULL)
		ORDER BY E_PO DESC
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'PM_QRY_ConsignmentItem_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[PM_QRY_ConsignmentItem_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 21/07/2008
-- Description:	Get items for selected consignment
-- version: 3
-- =============================================
-- jf tfs 145518 - added delivery parameter
-- =============================================
ALTER PROCEDURE [dbo].[PM_QRY_ConsignmentItem_Combo] 
	@ConsignmentID int = null,
	@DeliveryID int = null

AS

IF exists(select 1 from APP_DataLookupTrigger where DataLookupID_Trigger =7 and DataLookupID_Update = 6 and enabled = 1  ) and @DeliveryID IS NOT NULL
	SELECT     DISTINCT PM_VW_ConsignmentItems.ConsignmentItemID, PM_VW_StockCode.DisplayText
	FROM         PM_VW_ConsignmentItems 
				INNER JOIN PM_GoodsIntake on PM_GoodsIntake.ConsignmentItemID =PM_VW_ConsignmentItems.ConsignmentItemID 
				inner join PM_VW_StockCode ON PM_VW_ConsignmentItems.StockCodeID = PM_VW_StockCode.StockCodeID
	WHERE     (PM_GoodsIntake.DeliveryID = @DeliveryID)
	ORDER BY PM_VW_StockCode.DisplayText
else
	SELECT     PM_VW_ConsignmentItems.ConsignmentItemID, PM_VW_StockCode.DisplayText
	FROM         PM_VW_ConsignmentItems INNER JOIN
						  PM_VW_StockCode ON PM_VW_ConsignmentItems.StockCodeID = PM_VW_StockCode.StockCodeID
	WHERE     (PM_VW_ConsignmentItems.ConsignmentID = @ConsignmentID)
	ORDER BY PM_VW_StockCode.DisplayText

GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'PM_QRY_ConsignmentItem_Text4_Lookup')
	EXEC ('CREATE PROCEDURE [dbo].[PM_QRY_ConsignmentItem_Text4_Lookup] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JMF
-- Create date: 11/12/18
-- Description:	Data Lookup
-- Version:		3
-- =============================================
-- JF TFS 116450 -
-- added a no param filter option if the trigger isnt enabled
-- =============================================
ALTER PROCEDURE [dbo].[PM_QRY_ConsignmentItem_Text4_Lookup] 
	@ConsignmentItemID int = NULL
AS

if exists(select * from APP_DataLookupTrigger where DataLookupID_Update  =93 and DataLookupID_Trigger = 6 and enabled = 1)
	SELECT    
		Text4
	FROM         
		PM_VW_ConsignmentItems
	WHERE     
		ConsignmentItemID = @ConsignmentItemID
else
BEGIN
	DECLARE @DownloadConsignmentDays int, @DateFrom datetime
	SET @DownloadConsignmentDays = ISNULL(dbo.FNC_GetOptionValue(101),0)
	
	IF @DownloadConsignmentDays = 0 
		SET @DateFrom = NULL
	ELSE
		SET @DateFrom = DATEADD(day,-@DownloadConsignmentDays,GETDATE())

	SELECT    
		distinct Text4
	FROM         
		PM_VW_ConsignmentItems
	WHERE     
		(DateRequired > @DateFrom OR DateRequired IS NULL or @DateFrom IS NULL)
END


GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'PM_QRY_Delivery_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[PM_QRY_Delivery_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 11/09/2008
-- Description:	Delivery details for combo
-- Version: 3
-- =============================================
-- JF TFS 145518 - allow delivery (lot number) to be pulled from consignmentID and text4 if trigger setup
-- =============================================
ALTER PROCEDURE [dbo].[PM_QRY_Delivery_Combo] 
	@SupplierID int = 0,
	@DateArrived datetime = NULL,
	@ConsignmentID int = null,
	@Text4 nvarchar(50)=null

AS

if	exists(select 1 from APP_DataLookupTrigger where datalookupid_Trigger = 5 and DataLookupID_Update = 7 and enabled = 1) and 
	exists(select 1 from APP_DataLookupTrigger where datalookupid_Trigger = 93 and DataLookupID_Update = 7 and enabled = 1) and 
	@ConsignmentID IS NOT NULL
		SELECT     distinct PM_Delivery.DeliveryID, LorryNo
		FROM         PM_Delivery 
		inner join PM_GoodsIntake on PM_GoodsIntake.DeliveryID = PM_Delivery.DeliveryID 
		inner join PM_ConsignmentItems on PM_GoodsIntake.ConsignmentItemID =PM_ConsignmentItems.ConsignmentItemID 
		WHERE PM_ConsignmentItems.ConsignmentID = @ConsignmentID and (PM_ConsignmentItems.Text4 = @Text4 OR @Text4 IS NULL)
		ORDER BY LorryNo DESC
	
else IF @DateArrived IS NULL
	SELECT     DeliveryID, LorryNo
	FROM         PM_VW_Delivery
	WHERE     (SupplierID = @SupplierID OR
						  SupplierID IS NULL) OR
						  (@SupplierID = 0)
	ORDER BY LorryNo DESC

ELSE
	SELECT     PM_VW_Delivery.DeliveryID, PM_VW_Delivery.LorryNo
	FROM         PM_VW_Delivery INNER JOIN
						  PM_VW_GoodsIntake ON PM_VW_Delivery.DeliveryID = PM_VW_GoodsIntake.DeliveryID
	WHERE     (PM_VW_Delivery.SupplierID = @SupplierID) AND (dbo.FNC_TrimDate(PM_VW_GoodsIntake.DateArrived) = dbo.FNC_TrimDate(@DateArrived)) OR
						  (PM_VW_Delivery.SupplierID IS NULL) AND (dbo.FNC_TrimDate(PM_VW_GoodsIntake.DateArrived) = dbo.FNC_TrimDate(@DateArrived)) OR
						  (dbo.FNC_TrimDate(PM_VW_GoodsIntake.DateArrived) = dbo.FNC_TrimDate(@DateArrived)) AND (@SupplierID = 0)
	ORDER BY PM_VW_Delivery.LorryNo DESC
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'PM_QRY_IntakeBarcode_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[PM_QRY_IntakeBarcode_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		jmf
-- Create date: 3/2/2020
-- Description:	Looks up the arrival barcode
-- version: 3
-- =============================================
-- jf TFS 145518 created - dont show barcodes with checks against them
-- =============================================
ALTER PROCEDURE [dbo].[PM_QRY_IntakeBarcode_Combo]
	@ConsignmentItemID int = null
AS

	SELECT     Barcode 
	FROM        PM_VW_GoodsIntake
	WHERE     (ConsignmentItemID =@ConsignmentItemID and @ConsignmentItemID IS NOT NULL)
	and GoodsIntakeID not in
	(
		select PM_VW_GoodsIntake.GoodsIntakeID from QMS_ResponseHeader inner join QMS_AssessmentHeader on  QMS_AssessmentHeader.AssessmentHeaderID = QMS_ResponseHeader.AssessmentHeaderID 
		inner join PM_VW_GoodsIntake on PM_VW_GoodsIntake.Barcode = QMS_ResponseHeader.Header_Text 
		where QMS_AssessmentHeader.DataLookupID =11
		and QMS_ResponseHeader.Inserted>DATEADD(Day,-28,GETDATE())
		and Header_Text IS NOT NULL 
	)

GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'PM_QRY_Location_IntakeCombo')
	EXEC ('CREATE PROCEDURE [dbo].[PM_QRY_Location_IntakeCombo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		jmf
-- Create date: 3/2/2020
-- Description:	Looks up a list of intake locations for the user or the location of a goodsintake record
-- version: 3
-- =============================================
-- jf TFS 110206 Update Vitacress test with Integration config
-- =============================================
ALTER PROCEDURE [dbo].[PM_QRY_Location_IntakeCombo]
	@UserID int = 0,
	@Barcode nvarchar(100) = null
AS


if @Barcode IS NOT NULL and (SELECT Enabled FROM APP_DataLookupTrigger WHERE DataLookupID_Trigger = 11 AND DataLookupID_Update = 30) = 1
	select 
		PM_VW_Location.LocationID  ,DisplayText
	from 
		PM_VW_Location inner join PM_VW_GoodsIntake on PM_VW_GoodsIntake.LocationID = PM_VW_Location.LocationID 
	where PM_VW_GoodsIntake.Barcode  = @Barcode
else
	BEGIN
	--Get table of available locations
	DECLARE @Table_Location TABLE (LocationID int)
	INSERT INTO @Table_Location (LocationID)
	SELECT     CAST(PermissionTitle AS int) AS LocationID
	FROM         QMS_UserPermissions
	WHERE     (PermissionType = 103) AND (PermissionAllow = 1) AND (UserID = @UserID)

	IF (SELECT COUNT (LocationID) FROM @Table_Location)>0 --Location restriction
		SELECT     PM_VW_Location.LocationID, PM_VW_Location.LocationName
		FROM         PM_VW_Location INNER JOIN
							  @Table_Location AS User_Location ON PM_VW_Location.LocationID = User_Location.LocationID
		WHERE     (PM_VW_Location.IntakeLocation = 1)
		ORDER BY PM_VW_Location.LocationName

	ELSE
		SELECT     LocationID, LocationName
		FROM         PM_VW_Location
		WHERE     (IntakeLocation = 1)
		ORDER BY LocationName
	END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'PM_QRY_StockCode_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[PM_QRY_StockCode_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 16/07/2008
-- Description:	Gets a stock code list
-- Version:		12 
-- =============================================
-- JE updated to filter list by Customer
-- JT TFS 3657 - updated to only return one query result.
-- MP TFS 3869 - updated to ensure customer filter only used when trigger enabled
-- JE TFS 4178 - added AssessmentTypeID param so we know when to filter out hidden Stock Codes
-- JT TFS 5902 - 13-Feb-2012 - Added searchText
-- JE TFS 13-11942: exclude stock type of Seed/Variety when accessed from Mobile
-- JF TFS 74073 - allow stockcode to be picked up from goods intake without PO data ie PalletStockCodeID. Also check APP_DataLookupTrigger table
-- JF TFS 80479 - allow stockcode list to be filtered from PO
-- JF TFS 104066 GYFD - Allow Header To Work with Barcode Prefix
-- JF TFS 145518  - new consignmentitems parameter
-- =============================================
ALTER PROCEDURE [dbo].[PM_QRY_StockCode_Combo]
	@CaseEndBarcode nvarchar(100) = NULL,
	@UserID int = NULL,
	@CustomerID int = NULL,
	@AssessmentTypeID int = NULL,
	@SearchText nvarchar(100) = NULL,
	@Barcode nvarchar(50) = null,
	@ConsignmentID int = null,
	@ConsignmentItemID int = null
AS


IF @ConsignmentID IS NOT NULL and (SELECT Enabled FROM APP_DataLookupTrigger WHERE DataLookupID_Trigger = 5 AND DataLookupID_Update = 2) = 1
	SELECT DISTINCT PM_VW_StockCode.StockCodeID, PM_VW_StockCode.DisplayText 
	FROM PM_VW_StockCode 
	INNER JOIN PM_VW_ConsignmentItems on  PM_VW_ConsignmentItems.StockCodeID = PM_VW_StockCode.StockCodeID 
	INNER JOIN	dbo.FNC_User_StockCodeList(@UserID) AS User_StockCode ON PM_VW_StockCode.StockCodeID = User_StockCode.StockCodeID
	WHERE PM_VW_ConsignmentItems.ConsignmentID = @ConsignmentID

else IF @ConsignmentItemID IS NOT NULL and (SELECT Enabled FROM APP_DataLookupTrigger WHERE DataLookupID_Trigger = 6 AND DataLookupID_Update = 2) = 1
	SELECT DISTINCT PM_VW_StockCode.StockCodeID, PM_VW_StockCode.DisplayText 
	FROM PM_VW_StockCode 
	INNER JOIN PM_VW_ConsignmentItems on  PM_VW_ConsignmentItems.StockCodeID = PM_VW_StockCode.StockCodeID 
	INNER JOIN	dbo.FNC_User_StockCodeList(@UserID) AS User_StockCode ON PM_VW_StockCode.StockCodeID = User_StockCode.StockCodeID
	WHERE PM_VW_ConsignmentItems.ConsignmentItemID = @ConsignmentItemID

else IF @CaseEndBarcode IS NOT NULL --combo being accessed from an assessment, so hide hidden stock codes
BEGIN
	DECLARE @FilterStockCodeID int
	IF @CaseEndBarcode IS NOT NULL
		SET @FilterStockCodeID = dbo.FNC_CaseEndToStockCode(@CaseEndBarcode)
	ELSE
		SET @FilterStockCodeID = NULL

	SELECT     StockCodeID, DisplayText
	FROM         PM_VW_StockCode
	WHERE   ((StockCodeID = @FilterStockCodeID) AND (@FilterStockCodeID IS NOT NULL) 	OR		(@FilterStockCodeID IS NULL) AND (Hide = 0))
	AND (@SearchText IS NULL OR [Description] LIKE @SearchText OR StockCode LIKE @SearchText)
	AND (PM_VW_StockCode.StockTypeID <> 4)
	ORDER BY DisplayText
END
ELSE
BEGIN
	DECLARE @EnableCustomerFilter bit
	IF @CustomerID IS NOT NULL
		SELECT     @EnableCustomerFilter = Enabled
		FROM         dbo.APP_DataLookupTrigger
		WHERE     (DataLookupID_Update = 2) AND (DataLookupID_Trigger = 3)
	ELSE
		SET @EnableCustomerFilter = 0
		
	IF @UserID IS NULL AND @EnableCustomerFilter = 1 --combo not being accessed from an assessment, so show all stock codes
		SELECT		StockCodeID, DisplayText
		FROM         PM_VW_StockCode
		WHERE		CustomerID = @CustomerID
		AND (@SearchText IS NULL OR [Description] LIKE @SearchText OR StockCode LIKE @SearchText)
		ORDER BY DisplayText
		
	ELSE IF @UserID IS NOT NULL AND @EnableCustomerFilter = 0 AND @AssessmentTypeID IS NOT NULL --combo being accessed from an assessment, so hide hidden stock codes
		--Where is stockcode stored
		IF @Barcode is NOT NULL and (SELECT Enabled FROM dbo.APP_DataLookupTrigger WHERE DataLookupID_Trigger = 11 AND DataLookupID_Update = 2) = 1
			BEGIN
				SET @Barcode = dbo.FNC_BarcodeFormat(@Barcode)

				IF EXISTS(SELECT consignmentitemid FROM PM_VW_GoodsIntake WHERE Barcode = @Barcode and consignmentitemid IS NOT NULL)
					SELECT     DISTINCT PM_VW_StockCode.StockCodeID, DisplayText
					FROM         PM_VW_StockCode 
					INNER JOIN PM_VW_ConsignmentItems on PM_VW_ConsignmentItems.StockCodeID = PM_VW_StockCode.StockCodeID 
					INNER JOIN PM_vw_GoodsIntake ON PM_vw_GoodsIntake.ConsignmentItemID  = PM_VW_ConsignmentItems.ConsignmentItemID 
					WHERE   
					PM_vw_GoodsIntake.Barcode = @Barcode 
					ORDER BY DisplayText
				ELSE IF EXISTS(SELECT PalletStockCodeID FROM PM_VW_GoodsIntake WHERE Barcode = @Barcode and PalletStockCodeID IS NOT NULL)
					SELECT     DISTINCT StockCodeID, DisplayText
					FROM         PM_VW_StockCode INNER JOIN PM_vw_GoodsIntake ON PM_vw_GoodsIntake.PalletStockCodeID = PM_VW_StockCode.StockCodeID 
					WHERE   
					PM_vw_GoodsIntake.Barcode = @Barcode 
					ORDER BY DisplayText
				ELSE IF EXISTS(SELECT StockBatchID  FROM PM_VW_StockBatch WHERE Barcode = @Barcode)
					SELECT PM_VW_StockCode.StockCodeID, PM_VW_StockCode.DisplayText FROM PM_VW_StockBatch 
					INNER JOIN PM_VW_StockBatchItems ON PM_VW_StockBatchItems.StockBatchID  = PM_VW_StockBatch.StockBatchID
					INNER JOIN PM_VW_StockCode  ON PM_VW_StockCode.StockCodeID  = PM_VW_StockBatchItems.StockCodeID 
					INNER JOIN	dbo.FNC_User_StockCodeList(@UserID) AS User_StockCode ON PM_VW_StockCode.StockCodeID = User_StockCode.StockCodeID
					WHERE PM_VW_StockBatch.Barcode = @Barcode
			END
		ELSE

			SELECT		PM_VW_StockCode.StockCodeID, PM_VW_StockCode.DisplayText
			FROM         PM_VW_StockCode INNER JOIN
							dbo.FNC_User_StockCodeList(@UserID) AS User_StockCode ON PM_VW_StockCode.StockCodeID = User_StockCode.StockCodeID
			WHERE (Hide = 0)
			AND (@SearchText IS NULL OR [Description] LIKE @SearchText OR StockCode LIKE @SearchText)
			AND (PM_VW_StockCode.StockTypeID <> 4)
			ORDER BY PM_VW_StockCode.DisplayText

	ELSE IF @UserID IS NOT NULL AND @EnableCustomerFilter = 0 AND @AssessmentTypeID IS NULL --combo not being accessed from an assessment, so show all stock codes
		SELECT		PM_VW_StockCode.StockCodeID, PM_VW_StockCode.DisplayText
		FROM         PM_VW_StockCode INNER JOIN
						dbo.FNC_User_StockCodeList(@UserID) AS User_StockCode ON PM_VW_StockCode.StockCodeID = User_StockCode.StockCodeID
		AND (@SearchText IS NULL OR [Description] LIKE @SearchText OR StockCode LIKE @SearchText)
		ORDER BY PM_VW_StockCode.DisplayText
		
	ELSE IF @UserID IS NOT NULL AND @EnableCustomerFilter = 1 AND @AssessmentTypeID IS NOT NULL --combo being accessed from an assessment, so hide hidden stock codes
		SELECT		PM_VW_StockCode.StockCodeID, PM_VW_StockCode.DisplayText
		FROM         PM_VW_StockCode INNER JOIN
						dbo.FNC_User_StockCodeList(@UserID) AS User_StockCode ON PM_VW_StockCode.StockCodeID = User_StockCode.StockCodeID
		WHERE		(CustomerID = @CustomerID) AND (Hide = 0)
		AND (@SearchText IS NULL OR [Description] LIKE @SearchText OR StockCode LIKE @SearchText)
		AND (PM_VW_StockCode.StockTypeID <> 4)
		ORDER BY PM_VW_StockCode.DisplayText
	
	ELSE IF @UserID IS NOT NULL AND @EnableCustomerFilter = 1 AND @AssessmentTypeID IS NULL --combo not being accessed from an assessment, so show all stock codes
		SELECT		PM_VW_StockCode.StockCodeID, PM_VW_StockCode.DisplayText
		FROM         PM_VW_StockCode INNER JOIN
						dbo.FNC_User_StockCodeList(@UserID) AS User_StockCode ON PM_VW_StockCode.StockCodeID = User_StockCode.StockCodeID
		WHERE		(CustomerID = @CustomerID)
		AND (@SearchText IS NULL OR [Description] LIKE @SearchText OR StockCode LIKE @SearchText)
		ORDER BY PM_VW_StockCode.DisplayText	
		
	ELSE
		SELECT     StockCodeID, DisplayText
		FROM         PM_VW_StockCode
		WHERE (@SearchText IS NULL OR [Description] LIKE @SearchText OR StockCode LIKE @SearchText)
		ORDER BY DisplayText
END
GO




PRINT 'Database updated successfully!'
GO