-- =============================================
-- Author:		Jeremy Formby
-- Create date: 3/05/2016
-- Version:		1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON

GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_Device_List')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_Device_List] AS SELECT ''stub version, to be replaced''')
GO
-- ===================================================
-- Author:		Jonathan Toolan
-- Create date:	13/5/2014
-- Description:	List all devices 
-- Version:		5
-- ===================================================
-- JT TFS 13-2931 - Created
-- MP TFS 4704: Update to make sure hidden versions still included
-- JT TFS 13-11262: Added column warning.
-- NM TFS 96065: Expanded to include devices running qc2/native-ios, added App & DeviceType columns, and updated order
-- ===================================================
ALTER PROCEDURE [dbo].[QRY_Device_List]
	@UserID int = NULL
AS
BEGIN
	SET NOCOUNT ON

		DECLARE @LatestVersion nvarchar(50)
		SELECT @LatestVersion = ApplicationVersion
		FROM APP_ApplicationUpdate
		WHERE ApplicationUpdateID = (	SELECT max(ApplicationUpdateID) 
										FROM APP_ApplicationUpdate
										WHERE Offline = 1 
										AND ApplicationEXE = 'Greenlight.QC.Mobile.exe')

		DECLARE @workstations TABLE (
			WorkStation nvarchar(50),
			LastLogin datetime NULL,
			AppVersion nvarchar(50) NULL
		)

		INSERT @workstations (WorkStation, AppVersion, LastLogin)
		SELECT WorkStation, AppVersion, LoggedIn
		FROM (
			SELECT	WorkStation, 
					AppVersion, 
					LoggedIn, 
					row_number() OVER (PARTITION BY WorkStation ORDER BY LoggedIn DESC) as row
			FROM QMS_UserLog
			WHERE AppName = 'Greenlight QC Mobile'
		) AS workstations
		WHERE row = 1

	SET NOCOUNT OFF

	-- WARNING: If you change the columns type or order, please update APP_Notification_OutOfDateDevices
	SELECT	list.ClientGUID, 
			CASE WHEN list.[Enabled] = 0				THEN 'Disabled'
				 WHEN work.AppVersion = @LatestVersion	THEN 'Up-to-date'
				 WHEN list.Application IS NOT NULL		THEN 'N/A'
														ELSE 'Update required' END StatusText,
			CASE WHEN list.[Enabled] = 0				THEN 180
				 WHEN work.AppVersion = @LatestVersion	THEN 20
				 WHEN list.Application IS NOT NULL		THEN null
														ELSE 2 END StatusID,
			list.MachineName,
			coalesce(list.Version, work.AppVersion, list.iosNativeVersion, null) as AppVersion,	-- Use QC2 version if available, windows native, ios-native, or null 
			list.App,
			list.DeviceType,
			list.LastSync,
			list.[Enabled]			
	FROM (
		SELECT	ClientGUID, 
				MachineName,
				WorkStation = substring(MachineName, CASE WHEN MachineName LIKE '%\%' THEN charindex('\', MachineName) + 1 ELSE 0 END, 50),
				LastSync,
				Application,
				Version,

				CASE WHEN charindex('GreenlightQC', Application) > 0	THEN 'QC2'
				WHEN charindex('QualityControl', Application) > 0		THEN 'Native iOS'
																		ELSE 'Mobile for Windows' END App,

								   -- check for presence of qualifying characters, and that substring can't generate a negative number
				iosNativeVersion = iif((charindex('QualityControl', Application) > 0) AND (charindex(' ', Application) > 0) AND (charindex('/', Application) > 0)
								   AND (charindex(' ', Application) - charindex('/', Application) > 0),

										substring(Application,	-- fetch version located between '/' and ' ' 
											charindex('/', Application) + 1, 
											charindex(' ', Application) - charindex('/', Application)), 

								   null),	-- where we didn't pattern match a version, return null

							 -- check for presence of qualifying characters, and that substring can't generate a negative number
				DeviceType = iif((charindex('GreenlightQC', Application) > 0) AND (charindex('|', Application) > 0) 
							 AND (charindex('|', Application, (charindex('|', Application))+1) > 0), 

								substring(Application,	-- fetch version located between first '|' and second '|' 
									charindex('|', Application) + 1, 
									(charindex('|', Application, (charindex('|', Application))+1) - charindex('|', Application)-1)), 

							 null),	-- where we didn't pattern match a device type, return null

				[Enabled]
		FROM SYNC_Client
		WHERE ClientType IN (1, 4, 5, 8, 9)	-- Check windows, ios and android
	) AS list
	LEFT OUTER JOIN @workstations work ON list.WorkStation = work.WorkStation
	ORDER BY App, case when DeviceType is null then 2 else 1 end, DeviceType, LastLogin DESC, MachineName
END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_Notification_OutOfDateDevices')
	EXEC ('CREATE PROCEDURE [dbo].[APP_Notification_OutOfDateDevices] AS SELECT ''stub version, to be replaced''')
GO
-- =====================================================================
-- Author:		Jonathan Toolan
-- Create date: 20/11/2014
-- Description:	Sends emails to admins when devices are out of date.
-- Version:		6
-- =====================================================================
-- JT TFS 13-11262 - Created with link to QRY_Device_List
-- JT TFS 13-11657 - Alert will not go out if ApplicationVersion is not specified in APP_ApplicationUpdate
-- JT TFS 13-11262 - Changed text of email.
-- JT TFS 13-11657 - Do not send out emails for machines for which we do not know the version.
-- JT TFS 13-11657 - Do not send out emails if the messagebody is null (for any reason)
-- NM TFS 96065	   - Match structure updates to QRY_Device_List
-- =====================================================================
ALTER PROCEDURE [dbo].[APP_Notification_OutOfDateDevices]
AS
BEGIN

	-- Latest Version
	DECLARE @LatestVersion nvarchar(50)
	SELECT @LatestVersion = ApplicationVersion
	FROM APP_ApplicationUpdate
	WHERE ApplicationUpdateID = (	SELECT max(ApplicationUpdateID) 
									FROM APP_ApplicationUpdate
									WHERE Offline = 1 
									AND ApplicationEXE = 'Greenlight.QC.Mobile.exe')

	--Link device log screen
	DECLARE @BaseURL nvarchar(1000)
	SELECT @BaseURL = dbo.FNC_GetOptionValue(65)

	DECLARE @LogDeviceURL nvarchar(max)
	SELECT @LogDeviceURL =  @BaseURL  + CASE WHEN @BaseURL LIKE '%/' THEN '' ELSE '/' END + MenuAction 
	FROM APP_Menu 
	WHERE ResourceName = 'MNU_MobileDeviceLog' 
	AND AppID = 3

	-- Which devices need updating?
	DECLARE @Devices TABLE (
		ClientGUID uniqueidentifier, 
		StatusText nvarchar(50),
		StatusID int,
		MachineName nvarchar(50),
		AppVersion nvarchar(50),
		App nvarchar(50),
		DeviceType nvarchar(50),
		LastSync datetime,
		[Enabled] bit
	)
	INSERT @Devices EXEC QRY_Device_List 1



	-- Build message body in two steps.
	DECLARE @DeviceList nvarchar(max)
	SELECT @DeviceList = substring(
				(	SELECT nchar(13) + nchar(10) + MachineName + N' (v' + AppVersion + N')'
					FROM @Devices WHERE StatusID = 2
					AND AppVersion IS NOT NULL
					AND MachineName IS NOT NULL
					FOR XML PATH (''), type
				).value('.', 'nvarchar(max)'),
			3,
			2147483647)

	--If there no devices needing an update, no email is sent
	IF NULLIF(@DeviceList, '') IS NULL
	BEGIN
		DECLARE @MessageBody nvarchar(max)
SELECT @MessageBody = 
N'The latest version of the mobile device software is ' + ISNULL(@LatestVersion, 'available.') + N'

The following mobile devices are out of date, and require an update:

' + @DeviceList + N'

Please visit ' + coalesce(@LogDeviceURL, @BaseURL, 'your Greenlight QC website')  + ' to download the latest version to install on those devices.

This is an automatically created report for your information. You will only receive this report if you have any devices that still need updating.'

		-- Send to all users in new IT support group
		DECLARE @Admins TABLE (Fullname nvarchar(255), Email nvarchar(100))
		INSERT @Admins EXEC QRY_User_ListOfAdmins

		--Enter QMS_Notification record listing all devices that are out of date, including Device name, device version
		INSERT QMS_Notification(SendTo, Subject, MessageBody)
		SELECT	[sendto] = admins.Email,
				[subject] = 'Mobile devices which require an update.',
				[message] = @MessageBody
		FROM @Admins admins
		WHERE @MessageBody IS NOT NULL
	END
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_Response_MultiCheckQuestion')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_Response_MultiCheckQuestion] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author		: Jon Edwards
-- Create date	: 10/01/2014
-- Description	: Returns data for Multi Check parent and child questions
-- Version		: 8
-- =============================================
-- JE TFS 8717		: amended filter to show archived questions if they have a response against them
-- JE TFS 13-1588	: added RecordStatus column
-- JE TFS 13-2555	: display Score after response when Score > 0 and return NoOfQuestionsToHide
-- JE TFS 13-43069	: Changed functionality when applying 'Suppress Blanks' setting.  The process now excludes questions 
--                    where there are null responses, but does not exclude null responses from included questions. The 
--                    score and qa flags are duplicated in every question response of a check and exlcusion may result in
--                    the score/qa flag being shown on the report.
-- JT TFS 125510	: Make sure response data is shown even if the related assessment is deleted or archived (by recordstatus).
-- NM TFS 83743		: Make sure child checks are correctly ordered in association with their parent check.
-- =============================================
ALTER PROCEDURE [dbo].[RPT_Response_MultiCheckQuestion]
	@UserID int
AS
BEGIN

--declare @UserID int
--select @UserID = 1

	DECLARE @SupressBlanks bit;
	SET @SupressBlanks = ISNULL(CAST(dbo.FNC_GetOptionValue(36) AS bit),0);

	IF NULLIF(object_id('tempdb..#tmpAllAssesmentQuestions'), 0) IS NOT NULL DROP TABLE #tmpAllAssesmentQuestions
	SELECT
		question.AssessmentID,
		sort.AssessmentQuestionID,
		sort.SortIndex,
		sort.QuestionNumber
	INTO #tmpAllAssesmentQuestions
	FROM dbo.FNC_AssessmentQuestion_Sort_Multi(@UserID) sort -- This is the most expensive bit of code.
	INNER JOIN QMS_AssessmentQuestion question
		ON sort.AssessmentQuestionID = question.AssessmentQuestionID;

	IF NULLIF(object_id('tempdb..#tmpShownAssessmentQuestions'), 0) IS NOT NULL DROP TABLE #tmpShownAssessmentQuestions
	SELECT responseQuestion.ResponseID, includedAssessmentQuestions.AssessmentQuestionID 
	INTO #tmpShownAssessmentQuestions
	FROM
	(
		SELECT assessmentQuestions.AssessmentQuestionID
		FROM #tmpAllAssesmentQuestions AS assessmentQuestions
		INNER JOIN QMS_ResponseQuestion
			ON QMS_ResponseQuestion.AssessmentQuestionID = assessmentQuestions.AssessmentQuestionID
		INNER JOIN TEMP_Selected
			ON QMS_ResponseQuestion.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID
		INNER JOIN QMS_Response AS response
			ON response.ResponseID = TEMP_Selected.SelectedID
		WHERE
			response.ParentResponseGUID IS NOT NULL
			AND
			(
				(
					@SupressBlanks = 1
					AND
					(
						QMS_ResponseQuestion.Response_Text IS NOT NULL
						OR LEN(LTRIM(RTRIM(QMS_ResponseQuestion.Response_Text))) > 0
					)
				)
				OR
				(
					@SupressBlanks = 0
				)
			)
		GROUP BY assessmentQuestions.AssessmentQuestionID
	) AS includedAssessmentQuestions
	INNER JOIN QMS_ResponseQuestion AS responseQuestion
		ON responseQuestion.AssessmentQuestionID = includedAssessmentQuestions.AssessmentQuestionID
	INNER JOIN TEMP_Selected
		ON responseQuestion.ResponseID = TEMP_Selected.SelectedID
		AND TEMP_Selected.UserID = @UserID;

	-- Add headers for shown items if suppressing blanks.
	IF (@SupressBlanks = 1)
		INSERT #tmpShownAssessmentQuestions
		(
			ResponseID,
			AssessmentQuestionID
		)
		SELECT DISTINCT
			shown.ResponseID,
			questionChildren.AssessmentQuestionID
		FROM
		(
			SELECT	questions.AssessmentQuestionID,
			(
				SELECT '|' + convert(varchar(10), AssessmentQuestionID
			)
			FROM #tmpAllAssesmentQuestions
			WHERE
				QuestionNumber LIKE questions.QuestionNumber + '.%'
				AND AssessmentID = questions.AssessmentID FOR XML PATH('')) + '|' Children
			FROM #tmpAllAssesmentQuestions questions
		) AS questionChildren
		INNER JOIN #tmpShownAssessmentQuestions AS shown
			ON questionChildren.Children like '%|' + convert(varchar(10), shown.AssessmentQuestionID) + '|%' 
		WHERE
			NOT EXISTS
			(
				SELECT *
				FROM #tmpShownAssessmentQuestions
				WHERE
					ResponseID = shown.ResponseID
					AND AssessmentQuestionID = questionChildren.AssessmentQuestionID
			);

	--get Question data for parent and child checks
	SELECT
		QMS_ResponseQuestion.ResponseID,
		QMS_AssessmentQuestion.AssessmentQuestionID,
		AssessmentQuestion_Sort.QuestionNumber,
		QMS_QuestionPool.Question,
		APP_QuestionLevel.QuestionLevel,
		APP_QuestionLevel.ListIndex AS QuestionLevelSort,
		QMS_ResponseQuestion.Response_Text +
						CASE WHEN ISNULL(QMS_ResponseQuestion.Score, 0) = 0 THEN ''
								ELSE ' (' + CAST(QMS_ResponseQuestion.Score AS varchar(5)) + ')' END AS Response_Text, --include score after response when it is > 0
		QMS_ResponseQuestion.Recommended_QAFlagID,
		QMS_ResponseQuestion.QuestionComments,
		QMS_QuestionPool_Parent.Question AS ParentQuestion,
		APP_QAFlag.QAFlagBackColour,
		APP_QAFlag.QAFlagForeColour,
		QMS_ResponseQuestion.ResponseQuestionID,
		IsSectionHeaderRow = convert(bit, CASE WHEN NULLIF(QMS_AssessmentQuestion.DataTypeID, 0) IS NULL THEN 1 ELSE 0 END),
		TEMP_Selected.SelectedValue, --this indicates if question row is belongs to a parent or child check
		QMS_Response.Actual_Score,
		OverallFlag.QAFlag AS OverallFlag,
		OverallFlag.QAFlagBackColour AS OverallBackColour,
		QMS_Response.RecordStatus,
		--questions are in blocks of 5, the below calculates how many we need to hide
		CASE WHEN
		(
			ROUND(NumberOfChildChecks.NoOfChildChecks / 5, 0) * 5) < NumberOfChildChecks.NoOfChildChecks
			THEN (((ROUND(NumberOfChildChecks.NoOfChildChecks / 5, 0) * 5) + 5) - NumberOfChildChecks.NoOfChildChecks)
			ELSE ((ROUND(NumberOfChildChecks.NoOfChildChecks / 5, 0) * 5) - NumberOfChildChecks.NoOfChildChecks
		) END AS NoOfQuestionsToHide
	FROM	TEMP_Selected
	INNER JOIN QMS_Response
		ON TEMP_Selected.SelectedID = QMS_Response.ResponseID
	INNER JOIN APP_QAFlag AS OverallFlag
		ON QMS_Response.Actual_QAFlagID = OverallFlag.QAFlagID
	INNER JOIN QMS_ResponseQuestion
		ON TEMP_Selected.SelectedID = QMS_ResponseQuestion.ResponseID
	INNER JOIN QMS_AssessmentQuestion
		ON QMS_ResponseQuestion.AssessmentQuestionID = QMS_AssessmentQuestion.AssessmentQuestionID
	INNER JOIN #tmpShownAssessmentQuestions AS visibleQuestions
		ON visibleQuestions.AssessmentQuestionID = QMS_ResponseQuestion.AssessmentQuestionID
	INNER JOIN #tmpAllAssesmentQuestions AS AssessmentQuestion_Sort
		ON visibleQuestions.AssessmentQuestionID = AssessmentQuestion_Sort.AssessmentQuestionID
	INNER JOIN QMS_QuestionPool
		ON QMS_AssessmentQuestion.QuestionID = QMS_QuestionPool.QuestionID
	LEFT OUTER JOIN APP_QAFlag
		ON QMS_ResponseQuestion.Recommended_QAFlagID = APP_QAFlag.QAFlagID
	LEFT OUTER JOIN QMS_AssessmentQuestion AS QMS_AssessmentQuestion_Parent
		ON QMS_AssessmentQuestion.ParentQuestionID = QMS_AssessmentQuestion_Parent.AssessmentQuestionID
	LEFT OUTER JOIN APP_QuestionLevel
		ON QMS_AssessmentQuestion.QuestionLevelID = APP_QuestionLevel.QuestionLevelID
	LEFT OUTER JOIN QMS_QuestionPool AS QMS_QuestionPool_Parent
		ON QMS_AssessmentQuestion_Parent.QuestionID = QMS_QuestionPool_Parent.QuestionID
	LEFT OUTER JOIN APP_SampleSummary
		ON QMS_AssessmentQuestion.SampleSummaryID = APP_SampleSummary.SampleSummaryID
	LEFT OUTER JOIN
	(
		SELECT
			SelectedValue AS ParentCheckID,
			COUNT(SelectedID) As NoOfChildChecks
		FROM TEMP_Selected
		WHERE UserID = @UserID
			AND SelectedValue IS NOT NULL
		GROUP BY SelectedValue
	) AS NumberOfChildChecks
		ON TEMP_Selected.SelectedValue = NumberOfChildChecks.ParentCheckID
WHERE
	TEMP_Selected.UserID = @UserID
	AND QMS_ResponseQuestion.RecordStatus < 201
	AND QMS_AssessmentQuestion.HideFromReports = 0
	AND	EXISTS
	(
		SELECT ResponseID
		FROM #tmpShownAssessmentQuestions
		WHERE
			ResponseID = QMS_ResponseQuestion.ResponseID
			AND AssessmentQuestionID = QMS_AssessmentQuestion.AssessmentQuestionID
	)
ORDER BY ISNULL(TEMP_Selected.SelectedValue, QMS_ResponseQuestion.ResponseID), QMS_ResponseQuestion.ResponseID, AssessmentQuestion_Sort.SortIndex;

END
GO



PRINT 'Database updated successfully!'
GO