-- =============================================
-- Author:		Jeremy Formby
-- Create date: 29/04/2016
-- Version:		1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 
-- =============================================

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO


PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'UP_User_Password')
	EXEC ('CREATE PROCEDURE [dbo].[UP_User_Password] AS SELECT ''stub version, to be replaced''')
GO
-- ===================================================
-- Author:		Jon Edwards
-- Create date: 23/05/2011
-- Description: Updates password when changed at login
-- Version:		5
-- ===================================================
-- J.Toolan 8 Mar 2012: Added SHA2 password field to update
-- JT TFS 6629 - Expire old userpassword entries before adding a new one.
-- JT TFS 6791 - Handle cases where there are more than one username in the QMS_User table (and exclude deleted people)
-- Martin P TFS 13 - 42414 - Changed any GTEDATE() to GETUTCDATE()
-- ===================================================
ALTER PROCEDURE [dbo].[UP_User_Password]
	@UserName nvarchar(255),
	@Password nvarchar(255),
	@Password_MD5 nvarchar(255) = NULL,
	@Password_SHA2 nvarchar(255) = NULL
AS
BEGIN

	SET NOCOUNT ON;

	-- #tmpUser should only contain one username, but historically it sometimes has more.
	IF NULLIF(object_id('tempdb..#tmpUser'), 0) IS NOT NULL DROP TABLE #tmpUser
	SELECT UserID
	INTO #tmpUser
	FROM QMS_User
	WHERE UserName = @UserName 
	AND RecordStatus < 201
	
	-- update User table
	UPDATE QMS_User
	SET Password = @Password, 
		Password_MD5 = ISNULL(@Password_MD5, Password_MD5), 
		Password_SHA2 = ISNULL(@Password_SHA2, Password_SHA2),
        Updated = GETUTCDATE()
	WHERE UserID IN (SELECT UserID FROM #tmpUser)
	
	UPDATE QMS_UserPassword 
	SET ExpiryDate = getutcdate()
	WHERE UserID IN (SELECT UserID FROM #tmpUser)
	AND ISNULL(ExpiryDate, '31 Dec 9999') > getutcdate()
	
	-- add new row to passwords table
	INSERT INTO QMS_UserPassword (UserID, Password, CreationDate, ExpiryDate)
	SELECT UserID, @Password, GETUTCDATE(), DATEADD(day, CAST(dbo.FNC_GetOptionValue(46) AS int), GETUTCDATE())
	FROM QMS_User
	WHERE UserID IN (SELECT UserID FROM #tmpUser)
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_Response_AttachmentImages')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_Response_AttachmentImages] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 12/03/2009
-- Description:	Gets attachment data for check images
-- Version:		3
-- =============================================
-- MP TFS 5658 - added images linked to corrective actions
-- MartinP TFS 13-45992 - Removed CROSS JOIN by splitting out into temporary tables
-- =============================================
ALTER PROCEDURE [dbo].[RPT_Response_AttachmentImages] 
	@UserID int 
AS
BEGIN

	SET NOCOUNT ON

	DECLARE @ResponseCATable TABLE (ResponseID INT, CorrectiveActionID INT)

	INSERT INTO @ResponseCATable (ResponseID, CorrectiveActionID)
		SELECT SelectedID,NULL 
		FROM TEMP_Selected
		WHERE UserID = @UserID

	UPDATE    ResponseCATable
	SET          CorrectiveActionID = QMS_CorrectiveAction.CorrectiveActionID
	FROM         QMS_CorrectiveAction 
				 INNER JOIN @ResponseCATable AS ResponseCATable ON QMS_CorrectiveAction.SourceObjectID = ResponseCATable.ResponseID
	WHERE     (QMS_CorrectiveAction.CASourceID = 1) 

	DECLARE @ResponseAttachTable TABLE (ResponseID INT, AttachmentID INT)
	INSERT INTO @ResponseAttachTable (ResponseID,AttachmentID)
	SELECT ResponseCATable.ResponseID, AttachmentID
	FROM @ResponseCATable AS ResponseCATable
	INNER JOIN QMS_Attachment ON ResponseCATable.ResponseID = QMS_Attachment.SourceObjectID
	WHERE      (QMS_Attachment.AttachmentSourceID = 1) 
				AND (QMS_Attachment.RecordStatus < 201) 
				AND (QMS_Attachment.AttachmentFileName IS NOT NULL) 
				AND (LOWER(RIGHT(QMS_Attachment.AttachmentFileName, 3)) IN ('jpg', 'bmp', 'gif', 'png')) 
				AND (QMS_Attachment.SourceObjectID = ResponseCATable.ResponseID)
			
	INSERT INTO @ResponseAttachTable (ResponseID,AttachmentID)
	SELECT ResponseCATable.ResponseID, AttachmentID
	FROM @ResponseCATable AS ResponseCATable
	INNER JOIN QMS_Attachment ON ResponseCATable.ResponseID = QMS_Attachment.SourceObjectID
				AND (QMS_Attachment.AttachmentSourceID = 2) 
				AND (QMS_Attachment.RecordStatus < 201)
				AND (QMS_Attachment.AttachmentFileName IS NOT NULL) 
				AND (LOWER(RIGHT(QMS_Attachment.AttachmentFileName, 3)) IN ('jpg', 'bmp', 'gif', 'png')) 
				AND (QMS_Attachment.SourceObjectID = ResponseCATable.CorrectiveActionID)

	SET NOCOUNT OFF

	SELECT     ResponseAttachTable.ResponseID,
			   QMS_Attachment.AttachmentTitle, 
			   QMS_Attachment.AttachmentFileName, 
			   QMS_Attachment.Attachment,  
			   QMS_Attachment.AttachmentURL
	FROM @ResponseAttachTable AS ResponseAttachTable
	INNER JOIN QMS_Attachment ON ResponseAttachTable.AttachmentID = QMS_Attachment.AttachmentID


END




GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_ResponseQuestion_PreviousList')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_ResponseQuestion_PreviousList] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 13/10/2009
-- Description:	Gets list of previous question responses
-- Version:		8
-- =============================================
-- Jonathan Toolan: 25-Aug-2011 AssessmentTypeID linked from Response, not Assessment.
-- Jonathan Toolan: 26-Mar-2012 Location Added.
-- Jonathan Toolan: TFS 3211 Multiple AssessmentTypeID_LIST
-- JT TFS 6363: Added overridable to Overall flag as per option 74 in APP_Config.
-- JF TFS3004 - only show child checks in this screen
-- JT TFS 13-46301: 26/04/2016 - Split assessmenttypeid 2 into two chunks for performance reasons.
-- =============================================
ALTER PROCEDURE [dbo].[QRY_ResponseQuestion_PreviousList] 
	@AssessmentTypeID int = NULL,
	@AssessmentTypeID_LIST nvarchar(255) = NULL,
	@DateFrom datetime,
	@DateTo datetime,
	@UserID int
AS
BEGIN
	IF (@AssessmentTypeID_LIST IS NULL)
		SET @AssessmentTypeID_LIST = convert(nvarchar, @AssessmentTypeID)

	--set Override QAFlag option
	DECLARE @OverrideQAFlag bit
	SET @OverrideQAFlag = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)
	
	--create temp table of QA Flag values to join on in main query
	IF nullif(object_id('tempdb..#tmpQAFlag'), 0) IS NOT NULL DROP TABLE #tmpQAFlag
	SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
	INTO #tmpQAFlag
	FROM         APP_QAFlag

	IF @AssessmentTypeID_LIST = '2' --Arrival Check
	BEGIN
		--These types require temp table populated
		SET NOCOUNT ON
		--Clear temp table
		DELETE FROM TEMP_Selected WHERE (UserID = @UserID)
		--Fill temp table
		INSERT INTO TEMP_Selected
							  (SelectedID, UserID)
		SELECT     QMS_Response.ResponseID, @UserID AS UserID
		FROM         QMS_Response INNER JOIN
							  dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
							  QMS_Response.ResponseID = User_ResponseList.ResponseID
		WHERE     (QMS_Response.AssessmentTypeID = convert(int, @AssessmentTypeID_LIST))


		-- split main section into two - inner joins, then outer joins (to get performance benefits on Flamingo)
		DECLARE @Halfway TABLE (
			ResponseID int,
			AssessmentName nvarchar(100),
			EndDate datetime,
			Question nvarchar(255),
			Response_Text nvarchar(255),
			Response_Numeric float,
			Score float,
			QuestionComments nvarchar(255),
			Question_QAFlagID int,
			QuestionLevelID int,
			Response_QAFlagID int
		)

		INSERT @Halfway (	ResponseID,
							AssessmentName,
							EndDate,
							Question,
							Response_Text,
							Response_Numeric,
							Score,
							QuestionComments,
							Question_QAFlagID,
							QuestionLevelID,
							Response_QAFlagID)
		SELECT  QMS_Response.ResponseID, 
				QMS_Assessment.AssessmentName, 
				QMS_Response.EndDate, 
				QMS_QuestionPool.Question, 
				QMS_ResponseQuestion.Response_Text, 
				QMS_ResponseQuestion.Response_Numeric, 
				ROUND(QMS_ResponseQuestion.Score, 2) AS Score, 
				QMS_ResponseQuestion.QuestionComments, 
				QMS_ResponseQuestion.Recommended_QAFlagID AS Question_QAFlagID,
				QMS_AssessmentQuestion.QuestionLevelID,
				CASE WHEN @OverrideQAFlag = 0 THEN QMS_Response.Actual_QAFlagID ELSE ISNULL(QMS_Response.Override_QAFlagID, QMS_Response.Actual_QAFlagID) END AS Response_QAFlagID
		FROM        QMS_QuestionPool 
		INNER JOIN	QMS_AssessmentQuestion ON QMS_QuestionPool.QuestionID = QMS_AssessmentQuestion.QuestionID 
		INNER JOIN QMS_Response 
					INNER JOIN	QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID 
					INNER JOIN	QMS_ResponseQuestion ON QMS_Response.ResponseID = QMS_ResponseQuestion.ResponseID ON QMS_AssessmentQuestion.AssessmentQuestionID = QMS_ResponseQuestion.AssessmentQuestionID 
		INNER JOIN	TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID 
		WHERE     (TEMP_Selected.UserID = @UserID)
		
		SET NOCOUNT OFF


		SELECT main.ResponseID, 
				main.AssessmentName, 
				ResponseHeader_Selected.Supplier, 
				ResponseHeader_Selected.StockCode,
				ResponseHeader.Location,  
				main.EndDate, 
				QAFlag_Response.QAFlagID as Actual_QAFlagID, -- Should really be called Overall_QAFlagID
				QAFlag_Response.QAFlag AS Overall_QAFlag, 
				main.Question, 
				APP_QuestionLevel.QuestionLevel, 
				main.Response_Text, 
				main.Response_Numeric, 
				main.Score, 
				main.QuestionComments, 
				main.Question_QAFlagID, 
				QAFlag_Question.QAFlag
		FROM @Halfway as main
		LEFT OUTER JOIN	#tmpQAFlag AS QAFlag_Question ON main.Question_QAFlagID = QAFlag_Question.QAFlagID 
		LEFT OUTER JOIN	APP_QuestionLevel ON main.QuestionLevelID = APP_QuestionLevel.QuestionLevelID 
		LEFT OUTER JOIN	dbo.FNC_ResponseHeader_Selected(@UserID) AS ResponseHeader_Selected ON main.ResponseID = ResponseHeader_Selected.ResponseID 
		LEFT OUTER JOIN	#tmpQAFlag AS QAFlag_Response ON main.Response_QAFlagID = QAFlag_Response.QAFlagID
		LEFT OUTER JOIN	dbo.FNC_ResponseHeader() AS ResponseHeader ON main.ResponseID = ResponseHeader.ResponseID 
		WHERE    dbo.FNC_Response_IsMultiCheck(main.ResponseID) IN (0, 2)
		ORDER BY main.EndDate DESC
	END
	ELSE
	BEGIN
		SELECT	QMS_Response.ResponseID, 
				QMS_Assessment.AssessmentName,
				ResponseHeader.Location,  
				QMS_Response.EndDate, 
				QAFlag_Response.QAFlagID as Actual_QAFlagID, -- Should really be called Overall_QAFlagID
				QAFlag_Response.QAFlag AS Overall_QAFlag, 
				QMS_QuestionPool.Question, 
				APP_QuestionLevel.QuestionLevel, 
				QMS_ResponseQuestion.Response_Text, 
				QMS_ResponseQuestion.Response_Numeric, 
				ROUND(QMS_ResponseQuestion.Score, 2) AS Score, 
				QMS_ResponseQuestion.QuestionComments, 
				QMS_ResponseQuestion.Recommended_QAFlagID AS Question_QAFlagID, 
				QAFlag_Question.QAFlag
		FROM        QMS_QuestionPool 
		INNER JOIN	QMS_AssessmentQuestion ON QMS_QuestionPool.QuestionID = QMS_AssessmentQuestion.QuestionID 
		INNER JOIN	QMS_Response 
					INNER JOIN	QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID 
					INNER JOIN	dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON QMS_Response.ResponseID = User_ResponseList.ResponseID 
					INNER JOIN	QMS_ResponseQuestion ON QMS_Response.ResponseID = QMS_ResponseQuestion.ResponseID ON QMS_AssessmentQuestion.AssessmentQuestionID = QMS_ResponseQuestion.AssessmentQuestionID 
		LEFT OUTER JOIN	#tmpQAFlag AS QAFlag_Question ON QMS_ResponseQuestion.Recommended_QAFlagID = QAFlag_Question.QAFlagID 
		LEFT OUTER JOIN	APP_QuestionLevel ON QMS_AssessmentQuestion.QuestionLevelID = APP_QuestionLevel.QuestionLevelID 
		LEFT OUTER JOIN	#tmpQAFlag AS QAFlag_Response ON CASE WHEN @OverrideQAFlag = 0 THEN QMS_Response.Actual_QAFlagID ELSE ISNULL(QMS_Response.Override_QAFlagID, QMS_Response.Actual_QAFlagID) END = QAFlag_Response.QAFlagID
		LEFT OUTER JOIN	dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID 
		WHERE     (QMS_Response.AssessmentTypeID IN (SELECT convert(int, Token) FROM dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',')))
		AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, 2)
		ORDER BY QMS_Response.EndDate DESC
	END
END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_Response_MultiCheckQuestion')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_Response_MultiCheckQuestion] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author		: Jon Edwards
-- Create date	: 10/01/2014
-- Description	: Returns data for Multi Check parent and child questions
-- Version		: 5
-- =============================================
-- JE TFS 8717		- amended filter to show archived questions if they have a response against them
-- JE TFS 13-1588	: added RecordStatus column
-- JE TFS 13-2555	: display Score after response when Score > 0 and return NoOfQuestionsToHide
-- JE TFS 13-43069	: Changed functionality when applying 'Suppress Blanks' setting.  The process now excludes questions where there are null responses, but does not exclude null responses from included questions. The score and qa flags are duplicated in every question response of a check and exlcusion may result in the score/qa flag being shown on the report.
-- =============================================
ALTER PROCEDURE [dbo].[RPT_Response_MultiCheckQuestion]
	@UserID int
AS
BEGIN

--declare @UserID int
--select @UserID = 1

	DECLARE @SupressBlanks bit;
	SET @SupressBlanks = ISNULL(CAST(dbo.FNC_GetOptionValue(36) AS bit),0);

	IF NULLIF(object_id('tempdb..#tmpAllAssesmentQuestions'), 0) IS NOT NULL DROP TABLE #tmpAllAssesmentQuestions
	SELECT
		question.AssessmentID,
		sort.AssessmentQuestionID,
		sort.SortIndex,
		sort.QuestionNumber
	INTO #tmpAllAssesmentQuestions
	FROM dbo.FNC_AssessmentQuestion_Sort_Multi(@UserID) sort -- This is the most expensive bit of code.
	INNER JOIN QMS_AssessmentQuestion question
		ON sort.AssessmentQuestionID = question.AssessmentQuestionID;

	IF NULLIF(object_id('tempdb..#tmpShownAssessmentQuestions'), 0) IS NOT NULL DROP TABLE #tmpShownAssessmentQuestions
	SELECT responseQuestion.ResponseID, includedAssessmentQuestions.AssessmentQuestionID 
	INTO #tmpShownAssessmentQuestions
	FROM
	(
		SELECT assessmentQuestions.AssessmentQuestionID
		FROM #tmpAllAssesmentQuestions AS assessmentQuestions
		INNER JOIN QMS_ResponseQuestion
			ON QMS_ResponseQuestion.AssessmentQuestionID = assessmentQuestions.AssessmentQuestionID
		INNER JOIN TEMP_Selected
			ON QMS_ResponseQuestion.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID
		INNER JOIN QMS_Response AS response
			ON response.ResponseID = TEMP_Selected.SelectedID
		WHERE
			response.ParentResponseGUID IS NOT NULL
			AND
			(
				(
					@SupressBlanks = 1
					AND
					(
						QMS_ResponseQuestion.Response_Text IS NOT NULL
						OR LEN(LTRIM(RTRIM(QMS_ResponseQuestion.Response_Text))) > 0
					)
				)
				OR
				(
					@SupressBlanks = 0
				)
			)
		GROUP BY assessmentQuestions.AssessmentQuestionID
	) AS includedAssessmentQuestions
	INNER JOIN QMS_ResponseQuestion AS responseQuestion
		ON responseQuestion.AssessmentQuestionID = includedAssessmentQuestions.AssessmentQuestionID
	INNER JOIN TEMP_Selected
		ON responseQuestion.ResponseID = TEMP_Selected.SelectedID
		AND TEMP_Selected.UserID = @UserID;

	-- Add headers for shown items if suppressing blanks.
	IF (@SupressBlanks = 1)
		INSERT #tmpShownAssessmentQuestions
		(
			ResponseID,
			AssessmentQuestionID
		)
		SELECT DISTINCT
			shown.ResponseID,
			questionChildren.AssessmentQuestionID
		FROM
		(
			SELECT	questions.AssessmentQuestionID,
			(
				SELECT '|' + convert(varchar(10), AssessmentQuestionID
			)
			FROM #tmpAllAssesmentQuestions
			WHERE
				QuestionNumber LIKE questions.QuestionNumber + '.%'
				AND AssessmentID = questions.AssessmentID FOR XML PATH('')) + '|' Children
			FROM #tmpAllAssesmentQuestions questions
		) AS questionChildren
		INNER JOIN #tmpShownAssessmentQuestions AS shown
			ON questionChildren.Children like '%|' + convert(varchar(10), shown.AssessmentQuestionID) + '|%' 
		WHERE
			NOT EXISTS
			(
				SELECT *
				FROM #tmpShownAssessmentQuestions
				WHERE
					ResponseID = shown.ResponseID
					AND AssessmentQuestionID = questionChildren.AssessmentQuestionID
			);

	--get Question data for parent and child checks
	SELECT
		QMS_ResponseQuestion.ResponseID,
		QMS_AssessmentQuestion.AssessmentQuestionID,
		AssessmentQuestion_Sort.QuestionNumber,
		QMS_QuestionPool.Question,
		APP_QuestionLevel.QuestionLevel,
		APP_QuestionLevel.ListIndex AS QuestionLevelSort,
		QMS_ResponseQuestion.Response_Text +
						CASE WHEN ISNULL(QMS_ResponseQuestion.Score, 0) = 0 THEN ''
								ELSE ' (' + CAST(QMS_ResponseQuestion.Score AS varchar(5)) + ')' END AS Response_Text, --include score after response when it is > 0
		QMS_ResponseQuestion.Recommended_QAFlagID,
		QMS_ResponseQuestion.QuestionComments,
		QMS_QuestionPool_Parent.Question AS ParentQuestion,
		APP_QAFlag.QAFlagBackColour,
		APP_QAFlag.QAFlagForeColour,
		QMS_ResponseQuestion.ResponseQuestionID,
		IsSectionHeaderRow = convert(bit, CASE WHEN NULLIF(QMS_AssessmentQuestion.DataTypeID, 0) IS NULL THEN 1 ELSE 0 END),
		TEMP_Selected.SelectedValue, --this indicates if question row is belongs to a parent or child check
		QMS_Response.Actual_Score,
		OverallFlag.QAFlag AS OverallFlag,
		OverallFlag.QAFlagBackColour AS OverallBackColour,
		QMS_Response.RecordStatus,
		--questions are in blocks of 5, the below calculates how many we need to hide
		CASE WHEN
		(
			ROUND(NumberOfChildChecks.NoOfChildChecks / 5, 0) * 5) < NumberOfChildChecks.NoOfChildChecks
			THEN (((ROUND(NumberOfChildChecks.NoOfChildChecks / 5, 0) * 5) + 5) - NumberOfChildChecks.NoOfChildChecks)
			ELSE ((ROUND(NumberOfChildChecks.NoOfChildChecks / 5, 0) * 5) - NumberOfChildChecks.NoOfChildChecks
		) END AS NoOfQuestionsToHide
	FROM	TEMP_Selected
	INNER JOIN QMS_Response
		ON TEMP_Selected.SelectedID = QMS_Response.ResponseID
	INNER JOIN APP_QAFlag AS OverallFlag
		ON QMS_Response.Actual_QAFlagID = OverallFlag.QAFlagID
	INNER JOIN QMS_ResponseQuestion
		ON TEMP_Selected.SelectedID = QMS_ResponseQuestion.ResponseID
	INNER JOIN QMS_AssessmentQuestion
		ON QMS_ResponseQuestion.AssessmentQuestionID = QMS_AssessmentQuestion.AssessmentQuestionID
	INNER JOIN #tmpShownAssessmentQuestions AS visibleQuestions
		ON visibleQuestions.AssessmentQuestionID = QMS_ResponseQuestion.AssessmentQuestionID
	INNER JOIN #tmpAllAssesmentQuestions AS AssessmentQuestion_Sort
		ON visibleQuestions.AssessmentQuestionID = AssessmentQuestion_Sort.AssessmentQuestionID
	INNER JOIN QMS_QuestionPool
		ON QMS_AssessmentQuestion.QuestionID = QMS_QuestionPool.QuestionID
	LEFT OUTER JOIN APP_QAFlag
		ON QMS_ResponseQuestion.Recommended_QAFlagID = APP_QAFlag.QAFlagID
	LEFT OUTER JOIN QMS_AssessmentQuestion AS QMS_AssessmentQuestion_Parent
		ON QMS_AssessmentQuestion.ParentQuestionID = QMS_AssessmentQuestion_Parent.AssessmentQuestionID
	LEFT OUTER JOIN APP_QuestionLevel
		ON QMS_AssessmentQuestion.QuestionLevelID = APP_QuestionLevel.QuestionLevelID
	LEFT OUTER JOIN QMS_QuestionPool AS QMS_QuestionPool_Parent
		ON QMS_AssessmentQuestion_Parent.QuestionID = QMS_QuestionPool_Parent.QuestionID
	LEFT OUTER JOIN APP_SampleSummary
		ON QMS_AssessmentQuestion.SampleSummaryID = APP_SampleSummary.SampleSummaryID
	LEFT OUTER JOIN
	(
		SELECT
			SelectedValue AS ParentCheckID,
			COUNT(SelectedID) As NoOfChildChecks
		FROM TEMP_Selected
		WHERE UserID = @UserID
			AND SelectedValue IS NOT NULL
		GROUP BY SelectedValue
	) AS NumberOfChildChecks
		ON TEMP_Selected.SelectedValue = NumberOfChildChecks.ParentCheckID
WHERE
	TEMP_Selected.UserID = @UserID
	AND QMS_ResponseQuestion.RecordStatus < 201
	AND
	(
		(
			QMS_AssessmentQuestion.RecordStatus = 202
			AND QMS_ResponseQuestion.Response_Text IS NOT NULL
		) --show archived questions if they have a Response against them
		OR
		QMS_AssessmentQuestion.RecordStatus < 201
	)
	AND QMS_AssessmentQuestion.HideFromReports = 0
	AND	EXISTS
	(
		SELECT ResponseID
		FROM #tmpShownAssessmentQuestions
		WHERE
			ResponseID = QMS_ResponseQuestion.ResponseID
			AND AssessmentQuestionID = QMS_AssessmentQuestion.AssessmentQuestionID
	)
ORDER BY QMS_ResponseQuestion.ResponseID, AssessmentQuestion_Sort.SortIndex;

END

GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_SubRejections')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_SubRejections] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 15/12/2009
-- Description:	Last 5 rejects for supplier profile
-- Version:		11
-- =============================================
-- JE TFS 3940: report will include QAS 2 responses if IsRejection flag set
-- JE TFS 4770: added override QA Flag config check
-- JE TFS 5906: updated to make use of warehouse table
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7507: All paths return same columns
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- JT TFS 13-42222: 02/03/2016 - Use TokenUserID for looking up stockcodes
-- MartinP TFS 13-45085 29/04/2016 - Removed UserID AND replaced with transactionID within  delete statement warehouse in CanUseDataWarehouse = 1
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_SubRejections
	@UserID int,
	@Offline bit=0,
	@RejectionCount int=5,
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	DECLARE @TokenUserID int;
	SELECT @TokenUserID = dbo.FNC_GetTempTransaction_TokenUserID(@TransactionID, @UserID)

	--set Override QAFlag option
	DECLARE @OverrideQAFlag bit
	SET @OverrideQAFlag = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)

	--create temp table of QA Flag values to join on in main query
	IF nullif(object_id('tempdb..#tmpQAFlag'), 0) IS NOT NULL DROP TABLE #tmpQAFlag

	SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
	INTO #tmpQAFlag
	FROM         APP_QAFlag

	IF @Offline = 0
	BEGIN
		IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
		BEGIN
			DELETE FROM TEMP_Selected
			FROM         TEMP_Selected 
			LEFT OUTER JOIN	(SELECT TOP (@RejectionCount) RPT_Warehouse_Response.ResponseID, RPT_Warehouse_Response.EndDate
							 FROM		TEMP_Selected AS Selected 
							 INNER JOIN RPT_Warehouse_Response ON Selected.SelectedID = RPT_Warehouse_Response.ResponseID 
							 INNER JOIN APP_QAFlag ON RPT_Warehouse_Response.QAFlagID = APP_QAFlag.QAFlagID
							 WHERE	(Selected.TransactionID = @TransactionID) 
							 AND (APP_QAFlag.IsRejection = 1) --check IsRejection flag
							 ORDER BY RPT_Warehouse_Response.EndDate DESC) AS Rejections ON TEMP_Selected.SelectedID = Rejections.ResponseID
			WHERE     TEMP_Selected.TransactionID = @TransactionID AND (Rejections.ResponseID IS NULL)

			SELECT  Location.LocationName, 
					PM_VW_StockCode.DisplayText AS StockCode, 
					RPT_Warehouse_Response.EndDate, 
					RPT_Warehouse_Response.RedResponses, 
					RPT_Warehouse_Response.AmberResponses, 
					RPT_Warehouse_Response.Comments, 
					RPT_Warehouse_Response.QuantityRejected, 
					@RejectionCount AS RejectionCount,
					CASE WHEN QAFlag.QAFlagID = 2 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END AS AmberRejected,
					CASE WHEN QAFlag.QAFlagID = 3 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END AS RedRejected	
			FROM         PM_VW_Location AS Location RIGHT OUTER JOIN
								  RPT_Warehouse_Response INNER JOIN
								  PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
								  TEMP_Selected ON RPT_Warehouse_Response.ResponseID = TEMP_Selected.SelectedID ON 
								  Location.LocationID = RPT_Warehouse_Response.LocationID LEFT OUTER JOIN
								  #tmpQAFlag AS QAFlag ON RPT_Warehouse_Response.QAFlagID = QAFlag.QAFlagID
			WHERE     TEMP_Selected.TransactionID = @TransactionID 
			ORDER BY RPT_Warehouse_Response.EndDate DESC					
		END
		ELSE --don't use warehouse table
		BEGIN
			DELETE FROM TEMP_Selected --if so, leave QAS 2 and QAS 3 responses in TEMP_Selected
			FROM         TEMP_Selected 
			LEFT OUTER JOIN	  (SELECT TOP (@RejectionCount)
							   QMS_Response.ResponseID, 
							   QMS_Response.EndDate
							   FROM TEMP_Selected AS TEMP_Selected_1 
							   INNER JOIN QMS_Response ON TEMP_Selected_1.SelectedID = QMS_Response.ResponseID  
							   INNER JOIN APP_QAFlag ON 
												(CASE WHEN @OverrideQAFlag = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID)END)= APP_QAFlag.QAFlagID
							   WHERE      TEMP_Selected_1.TransactionID = @TransactionID 
							   AND (APP_QAFlag.IsRejection = 1) --check IsRejection flag
							   ORDER BY QMS_Response.EndDate DESC) AS Rejections ON TEMP_Selected.SelectedID = Rejections.ResponseID
			WHERE     TEMP_Selected.TransactionID = @TransactionID 
			AND (Rejections.ResponseID IS NULL)

			SELECT  Response_Location.LocationName, 
					PM_VW_StockCode.DisplayText AS StockCode, 
					QMS_Response.EndDate,
					
					CASE WHEN ResponseQuestionsRedAmber.RedResponses IS NULL THEN
						CASE WHEN ResponseQuestionsRedAmber.RedLevel IS NULL THEN '' ELSE ResponseQuestionsRedAmber.RedLevel + ' Defects' END
					ELSE ResponseQuestionsRedAmber.RedResponses + CASE WHEN ResponseQuestionsRedAmber.RedLevel IS NULL THEN ''
						ELSE ', ' + ResponseQuestionsRedAmber.RedLevel + ' Defects' END END AS RedResponses,
						
					CASE WHEN (SELECT IsRejection FROM APP_QAFlag WHERE QAFlagID = 2) = 1 THEN --check if QAS 2 is set to be a rejection
						CASE WHEN ResponseQuestionsRedAmber.AmberResponses IS NULL THEN
							CASE WHEN ResponseQuestionsRedAmber.AmberLevel IS NULL THEN '' ELSE ResponseQuestionsRedAmber.AmberLevel + ' Defects' END
						ELSE ResponseQuestionsRedAmber.AmberResponses + CASE WHEN ResponseQuestionsRedAmber.AmberLevel IS NULL THEN ''
							ELSE ', ' + ResponseQuestionsRedAmber.AmberLevel + ' Defects' END END
					ELSE '' END AS AmberResponses,	
					
					QMS_Response.Comments, 
					QMS_Response.QuantityRejected, 
					@RejectionCount AS RejectionCount,
					CASE WHEN QAFlag.QAFlagID = 2 THEN QMS_Response.QuantityRejected ELSE 0 END AS AmberRejected,
					CASE WHEN QAFlag.QAFlagID = 3 THEN QMS_Response.QuantityRejected ELSE 0 END AS RedRejected	
			FROM         PM_VW_StockCode INNER JOIN
								  dbo.FNC_ResponseHeader_StockCodeID_Selected(@TokenUserID) AS ResponseHeader_StockCodeID ON 
								  PM_VW_StockCode.StockCodeID = ResponseHeader_StockCodeID.StockCodeID INNER JOIN
								  QMS_Response ON ResponseHeader_StockCodeID.ResponseID = QMS_Response.ResponseID LEFT OUTER JOIN
								  dbo.FNC_ResponseQuestion_Selected(@TokenUserID, -2) AS ResponseQuestionsRedAmber ON QMS_Response.ResponseID = ResponseQuestionsRedAmber.ResponseID LEFT OUTER JOIN
									  (SELECT     ResponseHeader_Location.ResponseID, ResponseHeader_Location.Header_SourceID AS LocationID, PM_VW_Location.LocationName
										FROM          QMS_ResponseHeader AS ResponseHeader_Location INNER JOIN
															   QMS_AssessmentHeader AS AssessmentHeader_Location ON 
															   ResponseHeader_Location.AssessmentHeaderID = AssessmentHeader_Location.AssessmentHeaderID INNER JOIN
															   PM_VW_Location ON ResponseHeader_Location.Header_SourceID = PM_VW_Location.LocationID INNER JOIN
															   TEMP_Selected ON ResponseHeader_Location.ResponseID = TEMP_Selected.SelectedID
										WHERE      (AssessmentHeader_Location.DataTypeID = 9) AND (AssessmentHeader_Location.DataLookupID = 4) AND TEMP_Selected.TransactionID = @TransactionID ) 
								  AS Response_Location ON QMS_Response.ResponseID = Response_Location.ResponseID LEFT OUTER JOIN
								  #tmpQAFlag AS QAFlag ON --check config value to join on Actual or Override QAFlagID
										(CASE WHEN @OverrideQAFlag = 0 THEN Actual_QAFlagID
											ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
									= QAFlag.QAFlagID
			ORDER BY QMS_Response.EndDate DESC
		END
	END
	ELSE
	--Clear records from temp table apart from previous rejections
	BEGIN

		DELETE FROM TEMP_Selected
		FROM         TEMP_Selected LEFT OUTER JOIN
						  (SELECT     TOP (@RejectionCount) QMS_SupplierResponse.ResponseID, QMS_SupplierResponse.EndDate
							FROM          TEMP_Selected AS Selected INNER JOIN
												   QMS_SupplierResponse ON Selected.SelectedID = QMS_SupplierResponse.ResponseID INNER JOIN
													   APP_QAFlag ON QMS_SupplierResponse.QAFlagID = APP_QAFlag.QAFlagID
								WHERE      Selected.TransactionID = @TransactionID  AND (APP_QAFlag.IsRejection = 1) --check IsRejection flag
							ORDER BY QMS_SupplierResponse.EndDate DESC) AS Rejections ON TEMP_Selected.SelectedID = Rejections.ResponseID
		WHERE    TEMP_Selected.TransactionID = @TransactionID  AND (Rejections.ResponseID IS NULL)

		SELECT  Location.LocationName, 
				PM_VW_StockCode.DisplayText AS StockCode, 
				QMS_SupplierResponse.EndDate, 
				QMS_SupplierResponse.RedResponses, 
				'' as AmberResponses,
				QMS_SupplierResponse.Comments, 
				QMS_SupplierResponse.QuantityRejected, 
				@RejectionCount AS RejectionCount,
				CASE WHEN QAFlag.QAFlagID = 2 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END AS AmberRejected,
				CASE WHEN QAFlag.QAFlagID = 3 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END AS RedRejected	
		FROM         PM_VW_Location AS Location RIGHT OUTER JOIN
							  QMS_SupplierResponse INNER JOIN
							  PM_VW_StockCode ON QMS_SupplierResponse.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
							  TEMP_Selected ON QMS_SupplierResponse.ResponseID = TEMP_Selected.SelectedID ON 
							  Location.LocationID = QMS_SupplierResponse.LocationID LEFT OUTER JOIN
							  #tmpQAFlag AS QAFlag ON QMS_SupplierResponse.QAFlagID = QAFlag.QAFlagID
		WHERE    TEMP_Selected.TransactionID = @TransactionID 
		ORDER BY QMS_SupplierResponse.EndDate DESC
	END
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'SAVE_Response')
	EXEC ('CREATE PROCEDURE [dbo].[SAVE_Response] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author		: Markp
-- Create date	: 28/07/2008
-- Description	: Saves assessment response
-- Version		: 8
-- =============================================
-- Updated 23/08/2011 - Added AssessmentTypeID (and calculated default for it)
-- JT TFS 5665 - Include new @AQCResponseGUID parameter as OPTIONAL which populates the GUID column from QMS_SupplierResponse
-- JE TFS 6271 - updated to include optional param for Multi Checks' ParentResponseGUID and made other params optional
-- JE TFS 13-466: added RecordStatus param for AQC Multi Checks
-- JE TFS 13-1616: added missing RecordStatus parameter
-- SF TFS 7843 - updated to allow @ResponseID to be optional, use Scope_Identity() and return new Id in column ResponseID
--				 for Entity Framework in WebChecks
-- PH TFS 13-46981 : v8 - 05/05/2016 - Added @ClientID for WebChecks Web API use (for iOS App).  Reformatted and added semi-colons. Changed method of getting new id to MS recommended method.
-- =============================================
ALTER PROCEDURE [dbo].[SAVE_Response]
	@ResponseID int = NULL,
	@AssessmentID int,
	@StartDate datetime,
	@EndDate datetime,
	@Comments nvarchar(2000) = NULL,
	@Recommended_Score int = NULL,
	@Recommended_QAFlagID int = NULL,
	@Actual_Score int = NULL,
	@Actual_QAFlagID int = NULL,
	@QuantityReceived int = NULL,
	@QuantityChecked int = NULL,
	@QuantityAccepted int = NULL,
	@QuantityRejected int = NULL,
	@QuantityReplaced int = NULL,
	@ResponseStatus int,
	@UserID int,
	@AssessmentTypeID int = NULL,
	@AQCResponseGUID uniqueidentifier = NULL,
	@ParentResponseGUID uniqueidentifier = NULL,
	@RecordStatus int = NULL,
	@ClientID int = NULL
AS
BEGIN
	SET NOCOUNT ON;

	IF NULLIF(@RecordStatus, 0) IS NULL 
		SET @RecordStatus = 2;
	 --only set to 150 for AQC Multi Check when a child check isn't included in Summary

	-- If not specified, use the default AssessmentTypeID from the Assessment table itself.
	IF NULLIF(@AssessmentTypeID, 0) IS NULL 
		BEGIN
			SELECT  @AssessmentTypeID = AssessmentTypeID
			FROM    QMS_Assessment
			WHERE   AssessmentID = @AssessmentID;
		END

	DECLARE @UTCOffset FLOAT
	SET @UTCOffset = DATEDIFF(mi, GETUTCDATE(), GETDATE()) / 60

	DECLARE @newId TABLE
	(
		ResponseID int
	);

	IF ISNULL(@ResponseID, 0) = 0 --New record
		BEGIN
			INSERT INTO QMS_Response
			(
				AssessmentID,
				StartDate,
				EndDate,
				Comments,
				Recommended_Score,
				Recommended_QAFlagID,
				Actual_Score,
				Actual_QAFlagID,
				QuantityReceived, 
				QuantityChecked,
				QuantityAccepted,
				QuantityRejected,
				QuantityReplaced,
				ResponseStatus,
				UserID,
				UTCOffset,
				AssessmentTypeID, 
				AQCResponseGUID,
				ParentResponseGUID,
				RecordStatus,
				ClientID
			)
			OUTPUT INSERTED.ResponseID INTO @newId
			VALUES
			(
				@AssessmentID,
				@StartDate,
				@EndDate,
				@Comments,
				@Recommended_Score,
				@Recommended_QAFlagID,
				@Actual_Score,
				@Actual_QAFlagID,
				@QuantityReceived,
				@QuantityChecked,
				@QuantityAccepted,
				@QuantityRejected,
				@QuantityReplaced,
				@ResponseStatus,
				@UserID,
				@UTCOffset,
				@AssessmentTypeID,
				@AQCResponseGUID,
				@ParentResponseGUID,
				@RecordStatus,
				@ClientID
			);
			--Get new ID
			SELECT  @ResponseID = ResponseID FROM @newId;
		END
	ELSE --Update record
		UPDATE    QMS_Response
		SET
			AssessmentID = @AssessmentID,
			EndDate = @EndDate,
			Comments = @Comments,
			Recommended_Score = @Recommended_Score,
			Recommended_QAFlagID = @Recommended_QAFlagID,
			Actual_Score = @Actual_Score,
			Actual_QAFlagID = @Actual_QAFlagID,
			QuantityReceived = @QuantityReceived,
			QuantityChecked = @QuantityChecked,
			QuantityAccepted = @QuantityAccepted,
			QuantityRejected = @QuantityRejected,
			QuantityReplaced = @QuantityReplaced,
			ResponseStatus = @ResponseStatus,
			UserID = @UserID,
			MachineName = HOST_NAME(),
			Updated = GETUTCDATE(),
			UTCOffset = @UTCOffset,
			AssessmentTypeID = @AssessmentTypeID,
			AQCResponseGUID = ISNULL(@AQCResponseGUID, AQCResponseGUID),
			ParentResponseGUID = ISNULL(@ParentResponseGUID, ParentResponseGUID),
			RecordStatus = ISNULL(@RecordStatus, RecordStatus),
			ClientID = @ClientID
		WHERE
			ResponseID = @ResponseID;

	SET NOCOUNT OFF;

	SELECT  @ResponseID AS ResponseID,
			GUID --return GUID to be used as ParentResponseGUID for Multi Checks
	FROM    QMS_Response
	WHERE	ResponseID = @ResponseID;
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'SAVE_Client')
	EXEC ('CREATE PROCEDURE [dbo].[SAVE_Client] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author		: MBS\Phil.Harding
-- Create date	: 05/05/2016
-- Description	: Saves client info
-- Version		: 3
-- =============================================
-- PH TFS 13-46856 : Created
-- PH TFS 13-46988 : Added Application parameter
-- PH TFS 13-46988 : Application parameter changed to nvarchar
-- =============================================
ALTER PROCEDURE [dbo].[SAVE_Client]
	@ClientID int = NULL,
	@ClientGuid uniqueidentifier = NULL,
	@MachineName nvarchar(50) = NULL,
	@Inserted datetime = NULL,
	@LastSync datetime = NULL,
	@ClientType tinyint = NULL,
	@Enabled bit,
	@UserID int = NULL,
	@Application nvarchar(255) = NULL
AS

BEGIN
	SET NOCOUNT ON

	DECLARE @idTable TABLE
	(
		ClientID int NOT NULL
	);

	IF ISNULL(@ClientID, 0) = 0 --New record
		BEGIN
			INSERT INTO SYNC_Client
			(
				ClientGuid,
				MachineName,
				Inserted,
				LastSync,
				ClientType,
				Enabled,
				UserID,
				Application
			)
			OUTPUT INSERTED.ClientID INTO @idTable
			VALUES    
			(
				NEWID(),
				@MachineName,
				GETUTCDATE(),
				NULL,
				@ClientType,
				@Enabled,
				@UserID,
				@Application
			)
			--Get new ID
			SELECT  @ClientID = ClientID FROM @idTable;

		END
	ELSE --Update record
		UPDATE    SYNC_Client
		SET
			MachineName = @MachineName,
			LastSync = @LastSync,
			ClientType = @ClientType,
			Enabled = @Enabled,
			UserID = @UserID,
			Application = @Application
		WHERE	ClientID = @ClientID;

	SET NOCOUNT OFF;

	RETURN @ClientID;

END
GO



IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_AssessmentSubType_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_AssessmentSubType_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author		: Markp
-- Create date	: 18/08/2009
-- Description	: Gets sub types for specified assessment type
-- Version		: 2
-- =============================================
-- PH TFS 13-46887 : Give assessment type id a default, so that iOS will sync without crashing - issue with sub type to be resolved later.
-- =============================================
ALTER PROCEDURE [dbo].[QRY_AssessmentSubType_Combo] 
	@AssessmentTypeID int = NULL
AS

SELECT
	AssessmentSubTypeID,
	AssessmentSubType
FROM	APP_AssessmentSubType
WHERE
	Hide = 0
	AND AssessmentTypeID = @AssessmentTypeID
ORDER BY ListIndex;

GO



PRINT 'Database updated successfully!'
GO
