-- =============================================
-- Author:		Jeremy Formby
-- Create date: 10/07/2015
-- Version:		1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO

-- BEGIN DATA TEMP_TransactionIDs_41688
BEGIN 
	IF NOT EXISTS (SELECT * FROM sys.tables t INNER JOIN sys.columns c ON t.object_id = c.object_id WHERE t.name = 'TEMP_Selected' AND c.name = 'TransactionID' )			ALTER TABLE TEMP_Selected ADD TransactionID uniqueidentifier
	IF NOT EXISTS (SELECT * FROM sys.tables t INNER JOIN sys.columns c ON t.object_id = c.object_id WHERE t.name = 'TEMP_SelectedItem' AND c.name = 'TransactionID' )		ALTER TABLE TEMP_SelectedItem ADD TransactionID uniqueidentifier
	IF NOT EXISTS (SELECT * FROM sys.tables t INNER JOIN sys.columns c ON t.object_id = c.object_id WHERE t.name = 'TEMP_ResponseFilter' AND c.name = 'TransactionID' )		ALTER TABLE TEMP_ResponseFilter ADD TransactionID uniqueidentifier
	IF NOT EXISTS (SELECT * FROM sys.tables t INNER JOIN sys.columns c ON t.object_id = c.object_id WHERE t.name = 'TEMP_ResponseGroupBy' AND c.name = 'TransactionID' )	ALTER TABLE TEMP_ResponseGroupBy ADD TransactionID uniqueidentifier
	IF NOT EXISTS (SELECT * FROM sys.tables t INNER JOIN sys.columns c ON t.object_id = c.object_id WHERE t.name = 'TEMP_ResponseHeader' AND c.name = 'TransactionID' )		ALTER TABLE TEMP_ResponseHeader ADD TransactionID uniqueidentifier

	IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'dbo.TEMP_Selected') and name='IX_TEMP_Selected_Transaction') CREATE INDEX IX_TEMP_Selected_Transaction ON TEMP_Selected(TransactionID)
	IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'dbo.TEMP_SelectedItem') and name='IX_TEMP_SelectedItem_Transaction') CREATE INDEX IX_TEMP_SelectedItem_Transaction ON TEMP_SelectedItem(TransactionID)
	IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'dbo.TEMP_ResponseFilter') and name='IX_TEMP_ResponseFilter_Transaction') CREATE INDEX IX_TEMP_ResponseFilter_Transaction ON TEMP_ResponseFilter(TransactionID)
	IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'dbo.TEMP_ResponseGroupBy') and name='IX_TEMP_ResponseGroupBy_Transaction') CREATE INDEX IX_TEMP_ResponseGroupBy_Transaction ON TEMP_ResponseGroupBy(TransactionID)
	IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'dbo.TEMP_ResponseHeader') and name='IX_TEMP_ResponseHeader_Transaction') CREATE INDEX IX_TEMP_ResponseHeader_Transaction ON TEMP_ResponseHeader(TransactionID)
END
GO
-- END DATA TEMP_TransactionIDs_41688

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_TEMP_ClearTransaction')
	EXEC ('CREATE PROCEDURE [dbo].[APP_TEMP_ClearTransaction] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Jonathan Toolan
-- Create date: 19/02/2016
-- Description:	Clear out specific transaction entries used in temp tables.
-- Version: 	1
-- =============================================
-- JT TFS 13-41688: 19/02/2016 - Created
-- =============================================
ALTER PROCEDURE APP_TEMP_ClearTransaction
	@TransactionID uniqueidentifier
AS
BEGIN
	DELETE TEMP_Selected WHERE TransactionID = @TransactionID
	DELETE TEMP_SelectedItem WHERE TransactionID = @TransactionID
	DELETE TEMP_ResponseFilter WHERE TransactionID = @TransactionID
	DELETE TEMP_ResponseGroupBy WHERE TransactionID = @TransactionID
	DELETE TEMP_ResponseHeader WHERE TransactionID = @TransactionID
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_Temp_ReportFilter_StockCode')
	EXEC ('CREATE PROCEDURE [dbo].[APP_Temp_ReportFilter_StockCode] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JToolan
-- Create date: 22/03/2012
-- Description:	Works out which Stockcodes need to be put in the filter.
--              Can be used by any report.
-- Version:		1
-- =============================================
-- JT TFS 13-40008: 22/02/2016 - Extracted from APP_Temp_ReportFilter for speed reasons.
-- =============================================
ALTER PROCEDURE [dbo].[APP_Temp_ReportFilter_StockCode]
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime,
	@AssessmentTypeID_LIST nvarchar(255),

	@ReportFilterID1 int = NULL,
	@FilterID_LIST nvarchar(4000),
	@FilterText_LIST nvarchar(4000),
	@ReportFilterID2 int = NULL,
	@FilterID2 int = NULL,
	@FilterText2 nvarchar(255) = NULL,
	
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	SET NOCOUNT ON
	
	DECLARE @DateCalled DATETIME
	SET @DateCalled = GETDATE()	
	
	DELETE TEMP_StockCodeFilter WHERE UserID = @UserID

	-- If the first criteria is empty, but the second is not, swap them over.
	IF (LEN(@FilterID_LIST) = 0 AND LEN(@FilterText_LIST) = 0 -- first criteria is empty?!
	AND @ReportFilterID2 IS NOT NULL AND (@FilterID2 IS NOT NULL OR @FilterText2 IS NOT NULL)) -- but the second is not...
	BEGIN -- The great fairness swaparoo! (in two steps)
		SELECT	@ReportFilterID1 = @ReportFilterID2,
				@FilterID_LIST = CONVERT(nvarchar(4000), @FilterID2),
				@FilterText_LIST = @FilterText2
				
		SELECT @ReportFilterID2 = NULL, @FilterID2 = NULL, @FilterText2 = NULL
	END

	-- Is the second question correctly formulated? You must have an id or a text (or both) but not neither.
	-- JE TFS 5037 - corrected this as it was always setting @ReportFilterID2 to NULL for filters with no ID (i.e. text only)
	IF @ReportFilterID2 IS NOT NULL AND (NULLIF(@FilterID2, 0) IS NULL) AND (LEN(ISNULL(@FilterText2, '')) = 0)
	BEGIN
		SET @ReportFilterID2 = NULL
	END

	DECLARE @FilterItem TABLE (pos int not null, id int, txt nvarchar(255))

	-- tfs 2281 - if filter doesn't have a sourceid, this wasn't working
	INSERT @FilterItem (pos, id, txt)
	SELECT 	ISNULL(a.Position, b.Position) pos, CONVERT(int, ISNULL(a.Token,0)) id, replace(b.Token, '&comma;', ',') txt
	FROM dbo.FNC_SplitString(@FilterID_LIST, ',') a
	FULL JOIN dbo.FNC_SplitString(@FilterText_LIST, ',') b on a.Position = b.Position

	DECLARE @AssessmentType TABLE (id int not null)
	INSERT @AssessmentType (id) 
	SELECT CONVERT(int, Token) FROM dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',')
	INNER JOIN APP_AssessmentType ON CONVERT(int, Token) = AssessmentTypeID 

	IF NOT EXISTS (SELECT * FROM @AssessmentType) 
	BEGIN -- if we tried to join, and got nothing, that means that this report wants all non-hidden types.
		INSERT @AssessmentType (id) SELECT AssessmentTypeID FROM APP_AssessmentType WHERE Hide = 0
	END

	DECLARE @FilteredStockCode TABLE (id int not null)
	WHILE EXISTS (SELECT * FROM @FilterItem)
	BEGIN
		DECLARE @pos int, @FilterID int, @FilterText nvarchar(255)
		SELECT @pos = pos,  @FilterID = id, @FilterText = txt
		FROM @FilterItem
		WHERE pos = (SELECT MIN(pos) FROM @FilterItem)

		INSERT @FilteredStockCode (id)
		SELECT StockCodeID
		FROM dbo.FNC_ReportFilter_StockCode(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo)
		
		DELETE @FilterItem WHERE pos = @pos
	END

	-- If there is another filter, it acts as an "AND" (restricting results returned), not an "OR" (expanding results returned). 
	IF @ReportFilterID2 IS NOT NULL
	BEGIN 
	
		DELETE @FilteredStockCode 
		WHERE id NOT IN (
			SELECT StockCodeID 
			FROM dbo.FNC_ReportFilter_StockCode (@ReportFilterID2, @FilterID2, @FilterText2, @DateFrom, @DateTo)
		)
	END

	-- Only return stock codes that the user is allowed to see
	INSERT TEMP_StockCodeFilter (UserID, StockCodeID)
	SELECT @UserID as UserID, fs.id
	FROM @FilteredStockCode fs 
	INNER JOIN FNC_User_StockCodeList(@UserID) sl ON fs.id = sl.StockCodeID 
	--GROUP BY fs.id --removed in TFS 4740 as this was slowing the query down a LOT on JS UAT
	
	DECLARE @MonitorEnabled TINYINT
	SELECT @MonitorEnabled = ISNULL(dbo.FNC_GetOptionValue(92),0)

	IF @MonitorEnabled = 1 --Log filter to QMS_PerformanceLog
	BEGIN
		DECLARE @ReportString NVARCHAR(max)
		SET @ReportString = 'APP_Temp_ReportFilter_StockCode '
		SET @ReportString = @ReportString + '@UserID = ' + CAST(@UserID AS NVARCHAR(10))
		SET @ReportString = @ReportString + ', @DateFrom = ' + CAST(@DateFrom AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @DateTo = ' + CAST(@DateTo AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @AssessmentTypeID_LIST = ' + CASE WHEN LEN(@AssessmentTypeID_LIST)=0 THEN 'NULL' ELSE @AssessmentTypeID_LIST END
		IF @ReportFilterID1>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID1 = ' + CAST(@ReportFilterID1 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID_LIST = ' + CASE WHEN LEN(@FilterID_LIST)=0 THEN 'NULL' ELSE @FilterID_LIST END
			SET @ReportString = @ReportString + ', @FilterText_LIST = ' + CASE WHEN LEN(@FilterText_LIST)=0 THEN 'NULL' ELSE @FilterText_LIST END
		END
		IF @ReportFilterID2>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID2 = ' + CAST(@ReportFilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID2 = ' + CAST(@FilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterText2 = ' + CASE WHEN LEN(@FilterText2)=0 THEN 'NULL' ELSE @FilterText2 END
		END
		SET @ReportString = @ReportString + ', @TransactionID = ' + CASE WHEN @TransactionID IS NULL THEN 'NULL' ELSE '''' + CAST(@TransactionID AS NVARCHAR(36)) + '''' END

		DECLARE @ResponseCount INT
		SELECT @ResponseCount = COUNT(ResponseID) FROM TEMP_ResponseFilter WHERE UserID = @UserID
		
		INSERT INTO dbo.QMS_PerformanceLog
				( Action ,
				  StartTime ,
				  EndTime ,
				  RecordsAffected
				)
		VALUES  ( @ReportString ,
				  @DateCalled ,
				  GETDATE() ,
				  @ResponseCount
				  )
	END

	SET NOCOUNT OFF
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_Temp_ReportFilter')
	EXEC ('CREATE PROCEDURE [dbo].[APP_Temp_ReportFilter] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JToolan
-- Create date: 22/03/2012
-- Description:	Works out which ResponseIDs need to be put in the filter.
--              Can be used by any report.
-- Version:		20
-- =============================================
-- j.toolan 30-mar-2012: If @AssessmentTypeID_LIST is empty, then join all assessments
-- j.toolan 13 Apr 2012: Fixed bug with 2nd filter incorrectly clearing.
-- j.toolan 18 Apr 2012: Added Code to filter by stockcode where appropriate (renamed to ReportFilter)
-- JE 25 Apr 2012: made ReportFilterID1 optional, so can just filter by AssessmentType if necessary (TFS 3382)
-- j.toolan 6 July 2012: TFS 4013: Dynamic filtering added.
-- JE TFS 4740: removed GROUP BY for performance reasons
-- JE TFS 5037: corrected @ReportFilterID2 check as 2nd tier filter wasn't working
-- MP TFS 5289: changed date format used for custom filter functions
-- JT TFS 5396: changed code to allow commas in FilterTexts passed in as CSV. (replace "," with "&comma;" )
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 6689: FNC_ReportFilter_Response takes UserID parameter.
-- JT TFS 8117: Added offline filtering.
-- MP TFS 8227: Made temp table user specific and added performance logging
-- JE TFS 8274: Put result set of FNC_User_ResponseList into temp table before using it
-- JT TFS 13-667: Do not filter responses if the date range is too big.
-- JT TFS 13-1076: Search either QMS_Response or QMS_SupplierResponse, but not both.
-- JF TFS 2281 - if filter doesn't have a sourceid, this wasn't working
-- JT TFS 13-17467: Only include checks which are eligible candidates for the data warehouse.
-- JT TFS 13-38578: Convert SP to use QRY_User_ResponseList (rather than FNC_UserResponseList)
-- JT TFS 13-40008: 22/02/2016 - Added transaction IDs to response list generation, and extracted stockcode filtering
-- =============================================
ALTER PROCEDURE [dbo].[APP_Temp_ReportFilter]
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime,
	@AssessmentTypeID_LIST nvarchar(255),

	@ReportFilterID1 int = NULL,
	@FilterID_LIST nvarchar(4000),
	@FilterText_LIST nvarchar(4000),
	@ReportFilterID2 int = NULL,
	@FilterID2 int = NULL,
	@FilterText2 nvarchar(255) = NULL,
	
	@Offline bit = NULL,
	@SearchLocalChecksIfOffline bit = 1, -- Use QMS_Response even if offline (default)
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	SET NOCOUNT ON
	
	DECLARE @DateCalled DATETIME
	SET @DateCalled = GETDATE()	
	IF @Offline IS NULL SELECT @Offline = CASE WHEN db_name() = 'GreenlightQC_Mobile' THEN 1 ELSE 0 END
	
	DECLARE @SearchOffline bit -- if this is set, then we search in QMS_SupplierResponse.
	SELECT @SearchOffline  = CASE WHEN @Offline = 1 AND @SearchLocalChecksIfOffline = 0 THEN 1 ELSE 0 END
		
	-- Do not filter responses if the date range is not specified
	DECLARE @performResponseFiltering bit 
	SELECT @performResponseFiltering = CASE WHEN @DateFrom <= '1 Jan 2001' and datediff(year, @DateFrom, @DateTo) >= 10 THEN 0 ELSE 1 END
	
	DELETE TEMP_ResponseFilter WHERE UserID = @UserID AND (@TransactionID IS NULL OR TransactionID = @TransactionID)

	-- If the first criteria is empty, but the second is not, swap them over.
	IF (LEN(@FilterID_LIST) = 0 AND LEN(@FilterText_LIST) = 0 -- first criteria is empty?!
	AND @ReportFilterID2 IS NOT NULL AND (@FilterID2 IS NOT NULL OR @FilterText2 IS NOT NULL)) -- but the second is not...
	BEGIN -- The great fairness swaparoo! (in two steps)
		SELECT	@ReportFilterID1 = @ReportFilterID2,
				@FilterID_LIST = CONVERT(nvarchar(4000), @FilterID2),
				@FilterText_LIST = @FilterText2
				
		SELECT @ReportFilterID2 = NULL, @FilterID2 = NULL, @FilterText2 = NULL
	END

	-- Is the second question correctly formulated? You must have an id or a text (or both) but not neither.
	-- JE TFS 5037 - corrected this as it was always setting @ReportFilterID2 to NULL for filters with no ID (i.e. text only)
	IF @ReportFilterID2 IS NOT NULL AND (NULLIF(@FilterID2, 0) IS NULL) AND (LEN(ISNULL(@FilterText2, '')) = 0)
	BEGIN
		SET @ReportFilterID2 = NULL
	END

	DECLARE @FilterItem TABLE (pos int not null, id int, txt nvarchar(255))
	INSERT @FilterItem (pos, id, txt)
	
	-- tfs 2281 - if filter doesn't have a sourceidm this wasn't working
	SELECT 	ISNULL(a.Position, b.Position) pos, CONVERT(int, ISNULL(a.Token,0)) id, replace(b.Token, '&comma;', ',') txt
	FROM dbo.FNC_SplitString(@FilterID_LIST, ',') a
	FULL JOIN dbo.FNC_SplitString(@FilterText_LIST, ',') b on a.Position = b.Position

	DECLARE @AssessmentType TABLE (id int not null)
	INSERT @AssessmentType (id) 
	SELECT CONVERT(int, Token) FROM dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',')
	INNER JOIN APP_AssessmentType ON CONVERT(int, Token) = AssessmentTypeID 
	IF NOT EXISTS (SELECT * FROM @AssessmentType) 
	BEGIN -- if we tried to join, and got nothing, that means that this report wants all non-hidden types.
		INSERT @AssessmentType (id) SELECT AssessmentTypeID FROM APP_AssessmentType WHERE Hide = 0
	END

	-- Do some custom filtering if it is set up. This acts like "AND" (restricting results)
	DECLARE @CustomFilterFunction nvarchar(50)
	SET @CustomFilterFunction = NULL
	SELECT @CustomFilterFunction = APP_ReportFilter.FilterFunction 
	FROM APP_ReportFilter 
	WHERE ReportFilterID = @ReportFilterID1
	
	DECLARE @customSql nvarchar(1000)
	SET @customSql = NULL
	CREATE TABLE #tmpCustomFilter (UserID INT, ResponseID int, GroupByText nvarchar(50))

	DECLARE @FilteredResponse TABLE (id int not null)
	WHILE EXISTS (SELECT * FROM @FilterItem)
	BEGIN
		DECLARE @pos int, @FilterID int, @FilterText nvarchar(255)
		SELECT @pos = pos,  @FilterID = id, @FilterText = txt
		FROM @FilterItem
		WHERE pos = (SELECT MIN(pos) FROM @FilterItem)

		IF @performResponseFiltering = 1
		BEGIN
			-- either the custom function runs, *OR* the generic one, *OR* the offline one, but not more than one.
			IF @SearchOffline = 1
			BEGIN
				INSERT @FilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response_Offline(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
			END
			ELSE IF @CustomFilterFunction IS NOT NULL
			BEGIN
				SELECT @customSql = 'INSERT #tmpCustomFilter (UserID, ResponseID, GroupByText) SELECT ' + convert(nvarchar(10), @UserID) + ' AS UserID, ResponseID, GroupByText FROM dbo.[' + @CustomFilterFunction + ']' + 
									'(' + convert(nvarchar(10), @UserID) + ', ' +
									convert(nvarchar(30), @FilterID) + ', ' +
									'''' + convert(nvarchar(50), @FilterText) + ''', ' + 
									'''' + convert(nvarchar(30), @DateFrom, 13) + ''', ' +
									'''' + convert(nvarchar(30), @DateTo, 13) + ''');'
				
				PRINT @customSql
				
				IF @customSql IS NOT NULL
					EXECUTE (@customSql)
			END
			ELSE
			BEGIN
				INSERT @FilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
			END
		END
		
		DELETE @FilterItem WHERE pos = @pos
	END

	-- insert all our new custom filtered items (added to temp table in loop above)
	INSERT @FilteredResponse (id)
	SELECT ResponseID 
	FROM #tmpCustomFilter
	WHERE UserID = @UserID

	-- If there is another filter, it acts as an "AND" (restricting results returned), not an "OR" (expanding results returned). 
	IF @ReportFilterID2 IS NOT NULL
	BEGIN 
		IF @performResponseFiltering = 1
		BEGIN
			DECLARE @SecondFilteredResponse TABLE (id int)
			
			SELECT @CustomFilterFunction = APP_ReportFilter.FilterFunction 
			FROM APP_ReportFilter 
			WHERE ReportFilterID = @ReportFilterID2
		
			IF @SearchOffline = 1
			BEGIN
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response_Offline(@ReportFilterID2, @FilterID2, @FilterText2, @DateFrom, @DateTo, @UserID)
			END 
			ELSE IF @CustomFilterFunction IS NOT NULL
			BEGIN
				DELETE #tmpCustomFilter WHERE UserID = @UserID --clear it out for second filter.
				
				SELECT @customSql = 'INSERT #tmpCustomFilter (UserID, ResponseID, GroupByText) SELECT ' + convert(nvarchar(10), @UserID) + ' AS UserID, ResponseID, GroupByText FROM dbo.[' + @CustomFilterFunction + ']' + 
									'(' + convert(nvarchar(10), @UserID) + ', ' +
									convert(nvarchar(30), @FilterID2) + ', ' +
									'''' + convert(nvarchar(50), @FilterText2) + ''', ' + 
									'''' + convert(nvarchar(30), @DateFrom, 13) + ''', ' +
									'''' + convert(nvarchar(30), @DateTo, 13) + ''');'

				IF @customSql IS NOT NULL
					EXECUTE (@customSql)
				
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID FROM #tmpCustomFilter WHERE UserID = @UserID
			END
			ELSE
			BEGIN
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response (@ReportFilterID2, @FilterID2, @FilterText2, @DateFrom, @DateTo, @UserID)
			END
			
			DELETE @FilteredResponse 
			WHERE id NOT IN (
				SELECT id FROM @SecondFilteredResponse
			)
		END
	END

	-- Remove all responses which are not suitable candidates for the data warehouse
	DELETE @FilteredResponse
	WHERE dbo.FNC_Response_ShouldIncludeInDataWarehouse(id) = 0


	IF @performResponseFiltering = 1
	BEGIN
		IF @SearchOffline = 1
		BEGIN
			IF @ReportFilterID1 IS NOT NULL
				INSERT TEMP_ResponseFilter (UserID, ResponseID)
				SELECT @UserID as UserID, r.ResponseID 
				FROM QMS_SupplierResponse r
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				INNER JOIN @FilteredResponse fr ON fr.id = r.ResponseID
				WHERE EndDate BETWEEN @DateFrom AND @DateTo
				AND   r.ResponseID NOT IN (SELECT ResponseID FROM TEMP_ResponseFilter WHERE UserID = @UserID)
				GROUP BY r.ResponseID
			ELSE
				INSERT TEMP_ResponseFilter (UserID, ResponseID)
				SELECT @UserID as UserID, r.ResponseID 
				FROM QMS_SupplierResponse r 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id	
				WHERE EndDate BETWEEN @DateFrom AND @DateTo
				AND   r.ResponseID NOT IN (SELECT ResponseID FROM TEMP_ResponseFilter WHERE UserID = @UserID)
				GROUP BY r.ResponseID
		END
		ELSE
		BEGIN
			--JE TFS 82774 - temp table of responses user can view (moved from direct access in query to improve performance)
			IF nullif(object_id('tempdb..#tmpUserResponses'), 0) IS NOT NULL DROP TABLE #tmpUserResponses
			CREATE TABLE #tmpUserResponses (UserID int, ResponseID int, ResponseStatus int, QAFlagID int)
			INSERT #tmpUserResponses (UserID, ResponseID, ResponseStatus, QAFlagID)
			EXEC QRY_User_ResponseList @UserID, @DateFrom, @DateTo

			IF @ReportFilterID1 IS NOT NULL
				-- Only return responses that the user is allowed to see, filter by Assessment Type, and use filter
				INSERT TEMP_ResponseFilter (UserID, ResponseID, TransactionID)
				SELECT @UserID as UserID, rl.ResponseID, @TransactionID
				FROM #tmpUserResponses AS rl
				INNER JOIN QMS_Response r ON rl.ResponseID = r.ResponseID 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				INNER JOIN @FilteredResponse fr ON fr.id = r.ResponseID
				WHERE rl.UserID = @UserID
				GROUP BY rl.ResponseID
			
			IF @ReportFilterID1 IS NULL
				-- Only return responses that the user is allowed to see, but there has been no filter specified, so only filter by Assessment Type (if specified)
				INSERT TEMP_ResponseFilter (UserID, ResponseID, TransactionID)
				SELECT @UserID as UserID, rl.ResponseID, @TransactionID
				FROM #tmpUserResponses AS rl
				INNER JOIN QMS_Response r ON rl.ResponseID = r.ResponseID 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				WHERE rl.UserID = @UserID	
				GROUP BY rl.ResponseID
			
			DROP TABLE #tmpUserResponses
		END

	END
	
	DECLARE @MonitorEnabled TINYINT
	SELECT @MonitorEnabled = ISNULL(dbo.FNC_GetOptionValue(92),0)

	IF @MonitorEnabled = 1 --Log filter to QMS_PerformanceLog
	BEGIN
		DECLARE @ReportString NVARCHAR(max)
		SET @ReportString = 'APP_Temp_ReportFilter '
		SET @ReportString = @ReportString + '@UserID = ' + CAST(@UserID AS NVARCHAR(10))
		SET @ReportString = @ReportString + ', @DateFrom = ' + CAST(@DateFrom AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @DateTo = ' + CAST(@DateTo AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @AssessmentTypeID_LIST = ' + CASE WHEN LEN(@AssessmentTypeID_LIST)=0 THEN 'NULL' ELSE @AssessmentTypeID_LIST END
		IF @ReportFilterID1>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID1 = ' + CAST(@ReportFilterID1 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID_LIST = ' + CASE WHEN LEN(@FilterID_LIST)=0 THEN 'NULL' ELSE @FilterID_LIST END
			SET @ReportString = @ReportString + ', @FilterText_LIST = ' + CASE WHEN LEN(@FilterText_LIST)=0 THEN 'NULL' ELSE @FilterText_LIST END
		END
		IF @ReportFilterID2>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID2 = ' + CAST(@ReportFilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID2 = ' + CAST(@FilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterText2 = ' + CASE WHEN LEN(@FilterText2)=0 THEN 'NULL' ELSE @FilterText2 END
		END
		SET @ReportString = @ReportString + ', @TransactionID = ' + CASE WHEN @TransactionID IS NULL THEN 'NULL' ELSE '''' + CAST(@TransactionID AS NVARCHAR(36)) + '''' END

		DECLARE @ResponseCount INT
		SELECT @ResponseCount = COUNT(ResponseID) FROM TEMP_ResponseFilter WHERE UserID = @UserID
		
		INSERT INTO dbo.QMS_PerformanceLog
				( Action ,
				  StartTime ,
				  EndTime ,
				  RecordsAffected
				)
		VALUES  ( @ReportString ,
				  @DateCalled ,
				  GETDATE() ,
				  @ResponseCount
				  )
	END

	SET NOCOUNT OFF
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 14/12/2009
-- Description:	Gets summary data for supplier profile report
-- Version:		11
-- =============================================
-- JE TFS 3940 - added QA Flag colour columns used for colouring text in report
-- JT TFS 4708 - Added Assessment Type names list to SP output. (v4 from passed in list)
-- JE TFS 4770 - added QA Flag config check and Amber Rejected col
-- JE TFS 5061 - removed old filter params and amended to use two tier filtering
-- JE TFS 5906: updated to make use of warehouse table
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7507: All paths return same columns
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile
	@UserID int,
	@SupplierID int,
	@DateFrom datetime,
	@DateTo datetime,
	@Offline bit=0,
	@AssessmentTypeID_LIST nvarchar(1000) = NULL,
	@TransactionID uniqueidentifier
AS

--get list of Assessment Types from IDs passed in
DECLARE @AssessmentTypes nvarchar(1000)
SET @AssessmentTypes = dbo.FNC_AssessmentNamesFromIDs(@AssessmentTypeID_LIST, @UserID)

IF LEN(@AssessmentTypes) >= 70
BEGIN
	--cut off end of Assessment Types list if too big to fit on report header
	SET @AssessmentTypes = SUBSTRING(@AssessmentTypes, 1, 67) + '...'
END

BEGIN
	SET NOCOUNT ON
	
	DECLARE @TotalReceived int, @TotalPrice money
	SELECT     @TotalReceived = SUM(SourceObjectID), @TotalPrice = SUM(CAST(Header_Value AS money))
	FROM         TEMP_ResponseHeader
	WHERE     (UserID = @UserID)
	AND TransactionID = @TransactionID 

	SET NOCOUNT OFF

	IF @Offline = 0
	BEGIN
		IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
			BEGIN
				SELECT	@DateFrom AS DateFrom,
						@DateTo AS DateTo,
						PM_VW_Supplier.SupplierID,
						PM_VW_Supplier.SupplierCode,
						PM_VW_Supplier.SupplierName,
						
						COUNT(RPT_Warehouse_Response.ResponseID) AS NumChecks,
						SUM(RPT_Warehouse_Response.QuantityReceived) AS QuantityReceived,
						SUM(TEMP_Selected.SelectedValue * RPT_Warehouse_Response.QuantityReceived) AS ReceivedValue,
						SUM(RPT_Warehouse_Response.QuantityChecked) AS QuantityChecked,
						SUM(RPT_Warehouse_Response.QuantityAccepted) AS QuantityAccepted,
						
						SUM(CASE WHEN QAFlagID = 2 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END) AS AmberRejected,
						SUM(CASE WHEN QAFlagID = 3 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END) AS RedRejected,
						SUM(RPT_Warehouse_Response.QuantityRejected) AS QuantityRejected,
						SUM(CASE WHEN QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen,
						SUM(CASE WHEN QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber,
						
						SUM(CASE WHEN QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed,
						SUM(CASE WHEN QAFlagID = 3 THEN QuantityReceived ELSE 0 END) AS QuantityReceivedRed,
						@TotalReceived AS ConsignmentReceived,
						@TotalPrice AS ConsignmentReceivedValue,
						SUM(TEMP_Selected.SelectedValue * RPT_Warehouse_Response.QuantityRejected) AS RejectedValue,
						
						SUM(CASE WHEN QAFlagID = 3 THEN TEMP_Selected.SelectedValue * RPT_Warehouse_Response.QuantityReceived ELSE 0 END) AS RedValue,
						--return QAS Flag colours as separate cols so these can be used to set text colours in report
						(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
						(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
						(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour,
						@AssessmentTypes AS AssessmentTypes
				FROM	TEMP_Selected INNER JOIN
						  RPT_Warehouse_Response ON TEMP_Selected.SelectedID = RPT_Warehouse_Response.ResponseID INNER JOIN
						  PM_VW_Supplier ON RPT_Warehouse_Response.SupplierID = PM_VW_Supplier.SupplierID
				WHERE		(TEMP_Selected.UserID = @UserID)
				AND TEMP_Selected.TransactionID = @TransactionID 
				GROUP BY	PM_VW_Supplier.SupplierCode, PM_VW_Supplier.SupplierName, PM_VW_Supplier.SupplierID
				HAVING		(PM_VW_Supplier.SupplierID = @SupplierID)			
			END
		ELSE
			BEGIN	
				SELECT		@DateFrom AS DateFrom, 
							@DateTo AS DateTo, 
							PM_VW_Supplier.SupplierID, 
							PM_VW_Supplier.SupplierCode, 
							PM_VW_Supplier.SupplierName, 
							
							COUNT(QMS_Response.ResponseID) AS NumChecks, 
							SUM(QMS_Response.QuantityReceived) AS QuantityReceived, 
							SUM(TEMP_Selected.SelectedValue * QMS_Response.QuantityReceived) AS ReceivedValue, 
							SUM(QMS_Response.QuantityChecked) AS QuantityChecked, 
							SUM(QMS_Response.QuantityAccepted) AS QuantityAccepted,
							
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 2 THEN QMS_Response.QuantityRejected ELSE 0 END) AS AmberRejected,
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN QMS_Response.QuantityRejected ELSE 0 END) AS RedRejected,
							SUM(QMS_Response.QuantityRejected) AS QuantityRejected,
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen,
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber, 

							SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed,
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN QuantityReceived ELSE 0 END) AS QuantityReceivedRed,
							@TotalReceived AS ConsignmentReceived, 
							@TotalPrice AS ConsignmentReceivedValue,
							SUM(TEMP_Selected.SelectedValue * QMS_Response.QuantityRejected) AS RejectedValue,
							
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN TEMP_Selected.SelectedValue * QMS_Response.QuantityReceived ELSE 0 END) AS RedValue,
							--return QAS Flag colours as separate cols so these can be used to set text colours in report
							(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
							(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
							(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour,
							@AssessmentTypes AS AssessmentTypes
				FROM         PM_VW_Supplier CROSS JOIN
									  TEMP_Selected INNER JOIN QMS_Response ON TEMP_Selected.SelectedID = QMS_Response.ResponseID INNER JOIN
							  APP_QAFlag ON --check config value to join on Actual or Override QAFlagID
							  (CASE WHEN CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit) = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
									 = APP_QAFlag.QAFlagID
				WHERE     (TEMP_Selected.UserID = @UserID)
				AND TEMP_Selected.TransactionID = @TransactionID 
				GROUP BY PM_VW_Supplier.SupplierCode, PM_VW_Supplier.SupplierName, PM_VW_Supplier.SupplierID
				HAVING      (PM_VW_Supplier.SupplierID = @SupplierID)
			END	
	END
	ELSE
	BEGIN
		SELECT		@DateFrom AS DateFrom,
					@DateTo AS DateTo,
					PM_VW_Supplier.SupplierID,
					PM_VW_Supplier.SupplierCode,
					PM_VW_Supplier.SupplierName,
					
					COUNT(QMS_SupplierResponse.ResponseID) AS NumChecks,
					SUM(QMS_SupplierResponse.QuantityReceived) AS QuantityReceived,
					SUM(TEMP_Selected.SelectedValue * QMS_SupplierResponse.QuantityReceived) AS ReceivedValue,
					SUM(QMS_SupplierResponse.QuantityChecked) AS QuantityChecked,
					SUM(QMS_SupplierResponse.QuantityAccepted) AS QuantityAccepted,
					
					SUM(CASE WHEN QAFlagID = 2 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS AmberRejected,
					SUM(CASE WHEN QAFlagID = 3 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS RedRejected,
					SUM(QMS_SupplierResponse.QuantityRejected) AS QuantityRejected,
					SUM(CASE WHEN QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen,
					SUM(CASE WHEN QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber,
					
					SUM(CASE WHEN QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed,
					SUM(CASE WHEN QAFlagID = 3 THEN QuantityReceived ELSE 0 END) AS QuantityReceivedRed,
					@TotalReceived AS ConsignmentReceived,
					@TotalPrice AS ConsignmentReceivedValue,
					SUM(TEMP_Selected.SelectedValue * QMS_SupplierResponse.QuantityRejected) AS RejectedValue,
					
					SUM(CASE WHEN QAFlagID = 3 THEN TEMP_Selected.SelectedValue * QMS_SupplierResponse.QuantityReceived ELSE 0 END) AS RedValue,
					--return QAS Flag colours as separate cols so these can be used to set text colours in report
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour,
					'All' AS AssessmentTypes
		FROM         TEMP_Selected INNER JOIN
							  QMS_SupplierResponse ON TEMP_Selected.SelectedID = QMS_SupplierResponse.ResponseID INNER JOIN
							  PM_VW_Supplier ON QMS_SupplierResponse.SupplierID = PM_VW_Supplier.SupplierID
		WHERE     (TEMP_Selected.UserID = @UserID)
		AND TEMP_Selected.TransactionID = @TransactionID 
		GROUP BY PM_VW_Supplier.SupplierCode, PM_VW_Supplier.SupplierName, PM_VW_Supplier.SupplierID
		HAVING      (PM_VW_Supplier.SupplierID = @SupplierID)
	END
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_SubStockCode')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_SubStockCode] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 15/12/2009
-- Description:	Gets product data for supplier profile report
-- Version: 8
-- =============================================
-- JE TFS 4770: added override QA Flag config check and colour cols
-- JE TFS 5906: updated to make use of warehouse table
-- JE TFS 6187: added NULL values for Amber and Red Rejected cols so report picks up correcet values when using warehouse
-- MP TFS 6659: fixed issue with records being duplicated for each QA flag
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7507: All paths return same columns
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_SubStockCode
	@UserID int,
	@Offline bit=0,
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	--set Override QAFlag option
	DECLARE @OverrideQAFlag bit
	SET @OverrideQAFlag = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)

	IF @Offline = 0
	BEGIN
		IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
		BEGIN
			SELECT	PM_VW_StockCode.DisplayText AS StockCode, 
					COUNT(RPT_Warehouse_Response.ResponseID) AS NumChecks, 
					SUM(RPT_Warehouse_Response.QuantityReceived) AS QuantityReceived, 
					SUM(RPT_Warehouse_Response.QuantityChecked) AS QuantityChecked,
					SUM(RPT_Warehouse_Response.QuantityAccepted) AS QuantityAccepted, 
					SUM(CASE WHEN QAFlagID = 2 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END) AS AmberRejected,
					SUM(CASE WHEN QAFlagID = 3 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END) AS RedRejected,
					SUM(RPT_Warehouse_Response.QuantityRejected) AS QuantityRejected, 
					SUM(CASE WHEN QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen, 
					SUM(CASE WHEN QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber, 
					SUM(CASE WHEN QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed, 
					SUM(RPT_Warehouse_Response.QuantityRejected * TEMP_Selected.SelectedValue) AS RejectedValue,
					--return QAS Flag colours as separate cols so these can be used to set text colours in report
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour
			FROM	RPT_Warehouse_Response INNER JOIN
					  PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
					  TEMP_Selected ON RPT_Warehouse_Response.ResponseID = TEMP_Selected.SelectedID
			WHERE	(TEMP_Selected.UserID = @UserID)
			AND TEMP_Selected.TransactionID = @TransactionID 
			GROUP BY PM_VW_StockCode.DisplayText
			ORDER BY NumRed DESC, QuantityRejected DESC, NumAmber DESC, NumGreen DESC		
		END
		ELSE
		BEGIN
			SELECT	PM_VW_StockCode.DisplayText AS StockCode, 
					COUNT(QMS_Response.ResponseID) AS NumChecks, 
					SUM(QMS_Response.QuantityReceived) AS QuantityReceived, 
					SUM(QMS_Response.QuantityChecked) AS QuantityChecked, 
					SUM(QMS_Response.QuantityAccepted) AS QuantityAccepted,
					SUM(CASE WHEN APP_QAFlag.QAFlagID = 2 THEN QMS_Response.QuantityRejected ELSE 0 END) AS AmberRejected,
					SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN QMS_Response.QuantityRejected ELSE 0 END) AS RedRejected,
					SUM(QMS_Response.QuantityRejected) as QuantityRejected,
					SUM(CASE WHEN APP_QAFlag.QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen,
					SUM(CASE WHEN APP_QAFlag.QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber,
					SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed,
					SUM(QMS_Response.QuantityRejected * TEMP_Selected.SelectedValue) AS RejectedValue,
					--return QAS Flag colours as separate cols so these can be used to set text colours in report
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour
			FROM	QMS_Response INNER JOIN
					  dbo.FNC_ResponseHeader_StockCodeID_Selected(@UserID) AS ResponseHeader_StockCodeID ON 
					  QMS_Response.ResponseID = ResponseHeader_StockCodeID.ResponseID INNER JOIN
					  PM_VW_StockCode ON ResponseHeader_StockCodeID.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
					  TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID INNER JOIN
						APP_QAFlag ON --check config value to join on Actual or Override QAFlagID
						 (CASE WHEN @OverrideQAFlag = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
							= APP_QAFlag.QAFlagID
			WHERE	(TEMP_Selected.UserID = @UserID)
			AND TEMP_Selected.TransactionID = @TransactionID 
			GROUP BY PM_VW_StockCode.DisplayText
			ORDER BY NumRed DESC, RedRejected DESC, AmberRejected DESC, NumAmber DESC, NumGreen DESC
		END
	END
	ELSE
	BEGIN
		SELECT	PM_VW_StockCode.DisplayText AS StockCode,
				COUNT(QMS_SupplierResponse.ResponseID) AS NumChecks, 
				SUM(QMS_SupplierResponse.QuantityReceived) AS QuantityReceived, 
				SUM(QMS_SupplierResponse.QuantityChecked) AS QuantityChecked, 
				SUM(QMS_SupplierResponse.QuantityAccepted) AS QuantityAccepted, 
				SUM(CASE WHEN QAFlagID = 2 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS AmberRejected,
				SUM(CASE WHEN QAFlagID = 3 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS RedRejected,
				SUM(QMS_SupplierResponse.QuantityRejected) AS QuantityRejected, 
				SUM(CASE WHEN QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen, 
				SUM(CASE WHEN QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber, 
				SUM(CASE WHEN QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed, 
				SUM(QMS_SupplierResponse.QuantityRejected * TEMP_Selected.SelectedValue) AS RejectedValue,
				--return QAS Flag colours as separate cols so these can be used to set text colours in report
				(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
				(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
				(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour
		FROM	QMS_SupplierResponse INNER JOIN
				  PM_VW_StockCode ON QMS_SupplierResponse.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
				  TEMP_Selected ON QMS_SupplierResponse.ResponseID = TEMP_Selected.SelectedID
		WHERE	(TEMP_Selected.UserID = @UserID)
		AND TEMP_Selected.TransactionID = @TransactionID 
		GROUP BY PM_VW_StockCode.DisplayText
		ORDER BY NumRed DESC, QuantityRejected DESC, NumAmber DESC, NumGreen DESC
	END
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_SubRejections')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_SubRejections] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 15/12/2009
-- Description:	Last 5 rejects for supplier profile
-- Version:		9
-- =============================================
-- JE TFS 3940: report will include QAS 2 responses if IsRejection flag set
-- JE TFS 4770: added override QA Flag config check
-- JE TFS 5906: updated to make use of warehouse table
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7507: All paths return same columns
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_SubRejections
	@UserID int,
	@Offline bit=0,
	@RejectionCount int=5,
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	--set Override QAFlag option
	DECLARE @OverrideQAFlag bit
	SET @OverrideQAFlag = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)

	--create temp table of QA Flag values to join on in main query
	IF nullif(object_id('tempdb..#tmpQAFlag'), 0) IS NOT NULL DROP TABLE #tmpQAFlag

	SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
	INTO #tmpQAFlag
	FROM         APP_QAFlag

	IF @Offline = 0
	BEGIN
		IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
		BEGIN
			DELETE FROM TEMP_Selected
			FROM         TEMP_Selected LEFT OUTER JOIN
							  (SELECT TOP (@RejectionCount) RPT_Warehouse_Response.ResponseID, RPT_Warehouse_Response.EndDate
								FROM	TEMP_Selected AS Selected INNER JOIN
										   RPT_Warehouse_Response ON Selected.SelectedID = RPT_Warehouse_Response.ResponseID INNER JOIN
										   APP_QAFlag ON RPT_Warehouse_Response.QAFlagID = APP_QAFlag.QAFlagID
								WHERE	(Selected.UserID = @UserID) AND (APP_QAFlag.IsRejection = 1) --check IsRejection flag
								ORDER BY RPT_Warehouse_Response.EndDate DESC) AS Rejections ON TEMP_Selected.SelectedID = Rejections.ResponseID
			WHERE     (TEMP_Selected.UserID = @UserID)  AND TEMP_Selected.TransactionID = @TransactionID AND (Rejections.ResponseID IS NULL)

			SELECT  Location.LocationName, 
					PM_VW_StockCode.DisplayText AS StockCode, 
					RPT_Warehouse_Response.EndDate, 
					RPT_Warehouse_Response.RedResponses, 
					RPT_Warehouse_Response.AmberResponses, 
					RPT_Warehouse_Response.Comments, 
					RPT_Warehouse_Response.QuantityRejected, 
					@RejectionCount AS RejectionCount,
					CASE WHEN QAFlag.QAFlagID = 2 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END AS AmberRejected,
					CASE WHEN QAFlag.QAFlagID = 3 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END AS RedRejected	
			FROM         PM_VW_Location AS Location RIGHT OUTER JOIN
								  RPT_Warehouse_Response INNER JOIN
								  PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
								  TEMP_Selected ON RPT_Warehouse_Response.ResponseID = TEMP_Selected.SelectedID ON 
								  Location.LocationID = RPT_Warehouse_Response.LocationID LEFT OUTER JOIN
								  #tmpQAFlag AS QAFlag ON RPT_Warehouse_Response.QAFlagID = QAFlag.QAFlagID
			WHERE     (TEMP_Selected.UserID = @UserID) AND TEMP_Selected.TransactionID = @TransactionID 
			ORDER BY RPT_Warehouse_Response.EndDate DESC					
		END
		ELSE --don't use warehouse table
		BEGIN
			DELETE FROM TEMP_Selected --if so, leave QAS 2 and QAS 3 responses in TEMP_Selected
			FROM         TEMP_Selected LEFT OUTER JOIN
									  (SELECT     TOP (@RejectionCount) QMS_Response.ResponseID, QMS_Response.EndDate
										FROM          TEMP_Selected AS TEMP_Selected_1 INNER JOIN
														QMS_Response ON TEMP_Selected_1.SelectedID = QMS_Response.ResponseID  INNER JOIN
														APP_QAFlag ON --check config value to join on Actual or Override QAFlagID
															(CASE WHEN @OverrideQAFlag = 0 THEN Actual_QAFlagID
																ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
															= APP_QAFlag.QAFlagID
									WHERE      (TEMP_Selected_1.UserID = @UserID) AND TEMP_Selected_1.TransactionID = @TransactionID 
									AND (APP_QAFlag.IsRejection = 1) --check IsRejection flag
										ORDER BY QMS_Response.EndDate DESC) AS Rejections ON TEMP_Selected.SelectedID = Rejections.ResponseID
			WHERE     (TEMP_Selected.UserID = @UserID) AND TEMP_Selected.TransactionID = @TransactionID 
			AND (Rejections.ResponseID IS NULL)

			SELECT  Response_Location.LocationName, 
					PM_VW_StockCode.DisplayText AS StockCode, 
					QMS_Response.EndDate,
					
					CASE WHEN ResponseQuestionsRedAmber.RedResponses IS NULL THEN
						CASE WHEN ResponseQuestionsRedAmber.RedLevel IS NULL THEN '' ELSE ResponseQuestionsRedAmber.RedLevel + ' Defects' END
					ELSE ResponseQuestionsRedAmber.RedResponses + CASE WHEN ResponseQuestionsRedAmber.RedLevel IS NULL THEN ''
						ELSE ', ' + ResponseQuestionsRedAmber.RedLevel + ' Defects' END END AS RedResponses,
						
					CASE WHEN (SELECT IsRejection FROM APP_QAFlag WHERE QAFlagID = 2) = 1 THEN --check if QAS 2 is set to be a rejection
						CASE WHEN ResponseQuestionsRedAmber.AmberResponses IS NULL THEN
							CASE WHEN ResponseQuestionsRedAmber.AmberLevel IS NULL THEN '' ELSE ResponseQuestionsRedAmber.AmberLevel + ' Defects' END
						ELSE ResponseQuestionsRedAmber.AmberResponses + CASE WHEN ResponseQuestionsRedAmber.AmberLevel IS NULL THEN ''
							ELSE ', ' + ResponseQuestionsRedAmber.AmberLevel + ' Defects' END END
					ELSE '' END AS AmberResponses,	
					
					QMS_Response.Comments, 
					QMS_Response.QuantityRejected, 
					@RejectionCount AS RejectionCount,
					CASE WHEN QAFlag.QAFlagID = 2 THEN QMS_Response.QuantityRejected ELSE 0 END AS AmberRejected,
					CASE WHEN QAFlag.QAFlagID = 3 THEN QMS_Response.QuantityRejected ELSE 0 END AS RedRejected	
			FROM         PM_VW_StockCode INNER JOIN
								  dbo.FNC_ResponseHeader_StockCodeID_Selected(@UserID) AS ResponseHeader_StockCodeID ON 
								  PM_VW_StockCode.StockCodeID = ResponseHeader_StockCodeID.StockCodeID INNER JOIN
								  QMS_Response ON ResponseHeader_StockCodeID.ResponseID = QMS_Response.ResponseID LEFT OUTER JOIN
								  dbo.FNC_ResponseQuestion_Selected(@UserID, -2) AS ResponseQuestionsRedAmber ON QMS_Response.ResponseID = ResponseQuestionsRedAmber.ResponseID LEFT OUTER JOIN
									  (SELECT     ResponseHeader_Location.ResponseID, ResponseHeader_Location.Header_SourceID AS LocationID, PM_VW_Location.LocationName
										FROM          QMS_ResponseHeader AS ResponseHeader_Location INNER JOIN
															   QMS_AssessmentHeader AS AssessmentHeader_Location ON 
															   ResponseHeader_Location.AssessmentHeaderID = AssessmentHeader_Location.AssessmentHeaderID INNER JOIN
															   PM_VW_Location ON ResponseHeader_Location.Header_SourceID = PM_VW_Location.LocationID INNER JOIN
															   TEMP_Selected ON ResponseHeader_Location.ResponseID = TEMP_Selected.SelectedID
										WHERE      (AssessmentHeader_Location.DataTypeID = 9) AND (AssessmentHeader_Location.DataLookupID = 4) AND (TEMP_Selected.UserID = @UserID) AND TEMP_Selected.TransactionID = @TransactionID ) 
								  AS Response_Location ON QMS_Response.ResponseID = Response_Location.ResponseID LEFT OUTER JOIN
								  #tmpQAFlag AS QAFlag ON --check config value to join on Actual or Override QAFlagID
										(CASE WHEN @OverrideQAFlag = 0 THEN Actual_QAFlagID
											ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
									= QAFlag.QAFlagID
			ORDER BY QMS_Response.EndDate DESC
		END
	END
	ELSE
	--Clear records from temp table apart from previous rejections
	BEGIN

		DELETE FROM TEMP_Selected
		FROM         TEMP_Selected LEFT OUTER JOIN
						  (SELECT     TOP (@RejectionCount) QMS_SupplierResponse.ResponseID, QMS_SupplierResponse.EndDate
							FROM          TEMP_Selected AS Selected INNER JOIN
												   QMS_SupplierResponse ON Selected.SelectedID = QMS_SupplierResponse.ResponseID INNER JOIN
													   APP_QAFlag ON QMS_SupplierResponse.QAFlagID = APP_QAFlag.QAFlagID
								WHERE      (Selected.UserID = @UserID) AND Selected.TransactionID = @TransactionID  AND (APP_QAFlag.IsRejection = 1) --check IsRejection flag
							ORDER BY QMS_SupplierResponse.EndDate DESC) AS Rejections ON TEMP_Selected.SelectedID = Rejections.ResponseID
		WHERE     (TEMP_Selected.UserID = @UserID) AND TEMP_Selected.TransactionID = @TransactionID  AND (Rejections.ResponseID IS NULL)

		SELECT  Location.LocationName, 
				PM_VW_StockCode.DisplayText AS StockCode, 
				QMS_SupplierResponse.EndDate, 
				QMS_SupplierResponse.RedResponses, 
				'' as AmberResponses,
				QMS_SupplierResponse.Comments, 
				QMS_SupplierResponse.QuantityRejected, 
				@RejectionCount AS RejectionCount,
				CASE WHEN QAFlag.QAFlagID = 2 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END AS AmberRejected,
				CASE WHEN QAFlag.QAFlagID = 3 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END AS RedRejected	
		FROM         PM_VW_Location AS Location RIGHT OUTER JOIN
							  QMS_SupplierResponse INNER JOIN
							  PM_VW_StockCode ON QMS_SupplierResponse.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
							  TEMP_Selected ON QMS_SupplierResponse.ResponseID = TEMP_Selected.SelectedID ON 
							  Location.LocationID = QMS_SupplierResponse.LocationID LEFT OUTER JOIN
							  #tmpQAFlag AS QAFlag ON QMS_SupplierResponse.QAFlagID = QAFlag.QAFlagID
		WHERE     (TEMP_Selected.UserID = @UserID)
		AND TEMP_Selected.TransactionID = @TransactionID 
		ORDER BY QMS_SupplierResponse.EndDate DESC
	END
END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_SubQuestions')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_SubQuestions] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 15/12/2009
-- Description:	Top problems for supplier profile
-- Version:		6
-- =============================================
-- JE TFS 4111: includes Amber checks if Amber set to be rejection
-- JE TFS 4770: 
-- jf tfs 8470: Make relevant changes to RPT_SupplierProfile_SubQuestions so that Question Levels are not returned when ShowQuestionLevelOnReports = 0
-- JF tfs 8706
-- JT TFS 13-38811: Enclose APP_QAFlag in table variable to get speed improvements.
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_SubQuestions
	@UserID int,
	@Offline bit=0,
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	--exec RPT_SupplierProfile_SubQuestions 1,0
	DECLARE @ShowQuestionLevelOnReports BIT
	SET @ShowQuestionLevelOnReports = CAST(ISNULL(dbo.FNC_GetOptionValue(109),0) AS bit)

	IF @Offline = 0
	BEGIN
		SET NOCOUNT ON

		DECLARE @tmpRejectFlags TABLE (	QAFlagID int )
		INSERT @tmpRejectFlags (QAFlagID) SELECT QAFlagID FROM APP_QAFlag WHERE IsRejection = 1

		DECLARE @Question TABLE (ResponseID int, Question nvarchar(255), QAFlagID int)
		--Fill red questions to temp table
		--TFS 4770: include QAFlag too, so we can display Amber and Red Rejected
		INSERT INTO @Question (ResponseID, Question, QAFlagID)
		SELECT     QMS_Response.ResponseID, QMS_QuestionPool.Question, flags.QAFlagID
		FROM         QMS_AssessmentQuestion 
		INNER JOIN	QMS_QuestionPool ON QMS_AssessmentQuestion.QuestionID = QMS_QuestionPool.QuestionID 
		INNER JOIN	QMS_Response ON QMS_AssessmentQuestion.AssessmentID = QMS_Response.AssessmentID 
		INNER JOIN	QMS_ResponseQuestion ON QMS_Response.ResponseID = QMS_ResponseQuestion.ResponseID 
										AND 	QMS_AssessmentQuestion.AssessmentQuestionID = QMS_ResponseQuestion.AssessmentQuestionID 
		INNER JOIN TEMP_Selected ON TEMP_Selected.UserID = @UserID AND TEMP_Selected.TransactionID = @TransactionID AND QMS_Response.ResponseID = TEMP_Selected.SelectedID 
		INNER JOIN	@tmpRejectFlags flags ON QMS_ResponseQuestion.Recommended_QAFlagID = flags.QAFlagID
	
		IF @ShowQuestionLevelOnReports = 1
			--Fill red question levels
			INSERT INTO @Question (ResponseID, Question, QAFlagID)
			SELECT     ResponseSummary.ResponseID, APP_QuestionLevel.QuestionLevel + ' Defects', APP_QAFlag.QAFlagID
			FROM         APP_AssessmentConstraintType INNER JOIN
								  APP_QuestionLevel ON APP_AssessmentConstraintType.QuestionLevelID = APP_QuestionLevel.QuestionLevelID INNER JOIN
								  QMS_AssessmentConstraint ON 
								  APP_AssessmentConstraintType.AssessmentConstraintTypeID = QMS_AssessmentConstraint.AssessmentConstraintTypeID INNER JOIN
									  (SELECT     QMS_ResponseQuestion.ResponseID, QMS_AssessmentQuestion.AssessmentID, QMS_AssessmentQuestion.QuestionLevelID, 
															   SUM(QMS_ResponseQuestion.Score) AS LevelScore
										FROM          QMS_ResponseQuestion INNER JOIN
															   QMS_Response ON QMS_ResponseQuestion.ResponseID = QMS_Response.ResponseID INNER JOIN
															   QMS_AssessmentQuestion ON QMS_ResponseQuestion.AssessmentQuestionID = QMS_AssessmentQuestion.AssessmentQuestionID INNER JOIN
															   TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID
										WHERE      (QMS_ResponseQuestion.RecordStatus < 201) AND (TEMP_Selected.UserID = @UserID) AND TEMP_Selected.TransactionID = @TransactionID 
										GROUP BY QMS_ResponseQuestion.ResponseID, QMS_AssessmentQuestion.AssessmentID, QMS_AssessmentQuestion.QuestionLevelID
										HAVING      (SUM(QMS_ResponseQuestion.Score) IS NOT NULL)) AS ResponseSummary ON 
								  QMS_AssessmentConstraint.AssessmentID = ResponseSummary.AssessmentID AND 
								  APP_AssessmentConstraintType.QuestionLevelID = ResponseSummary.QuestionLevelID INNER JOIN
								  APP_QAFlag ON QMS_AssessmentConstraint.QAFlagID = APP_QAFlag.QAFlagID
			WHERE     (QMS_AssessmentConstraint.RecordStatus < 201) AND (QMS_AssessmentConstraint.Constraint_Score_Min IS NOT NULL) AND 
								  (QMS_AssessmentConstraint.Constraint_Score_Max IS NOT NULL) AND (ResponseSummary.LevelScore >= QMS_AssessmentConstraint.Constraint_Score_Min) AND 
								  (ResponseSummary.LevelScore <= QMS_AssessmentConstraint.Constraint_Score_Max) AND (APP_QAFlag.IsRejection = 1) OR
								  (QMS_AssessmentConstraint.RecordStatus < 201) AND (QMS_AssessmentConstraint.Constraint_Score_Min IS NULL) AND 
								  (QMS_AssessmentConstraint.Constraint_Score_Max IS NOT NULL) AND (ResponseSummary.LevelScore <= QMS_AssessmentConstraint.Constraint_Score_Max) AND 
								  (APP_QAFlag.IsRejection = 1) OR
								  (QMS_AssessmentConstraint.RecordStatus < 201) AND (QMS_AssessmentConstraint.Constraint_Score_Min IS NOT NULL) AND 
								  (QMS_AssessmentConstraint.Constraint_Score_Max IS NULL) AND (ResponseSummary.LevelScore >= QMS_AssessmentConstraint.Constraint_Score_Min) AND 
								  (APP_QAFlag.IsRejection = 1)

		SET NOCOUNT OFF

		SELECT     TOP (10) Response_Question.Question, PM_VW_StockCode.DisplayText AS StockCode, COUNT(QMS_Response.ResponseID) AS NumChecks, 
						  SUM(QMS_Response.QuantityAccepted) AS QuantityAccepted,
						  SUM(CASE WHEN Response_Question.QAFlagID = 2 THEN QMS_Response.QuantityRejected ELSE 0 END) AS AmberRejected,
						  SUM(CASE WHEN Response_Question.QAFlagID = 3 THEN QMS_Response.QuantityRejected ELSE 0 END) AS RedRejected,
						  SUM(TEMP_Selected.SelectedValue * QMS_Response.QuantityRejected) AS RejectedValue
		FROM         QMS_Response 
		INNER JOIN	dbo.FNC_ResponseHeader_StockCodeID_Selected(@UserID) AS ResponseHeader_StockCodeID ON QMS_Response.ResponseID = ResponseHeader_StockCodeID.ResponseID 
		INNER JOIN	PM_VW_StockCode ON ResponseHeader_StockCodeID.StockCodeID = PM_VW_StockCode.StockCodeID 
		INNER JOIN	TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID 
		INNER JOIN	@Question AS Response_Question ON QMS_Response.ResponseID = Response_Question.ResponseID
		WHERE     (TEMP_Selected.UserID = @UserID) AND TEMP_Selected.TransactionID = @TransactionID 
		GROUP BY PM_VW_StockCode.DisplayText, Response_Question.Question--, QuantityRejected
		ORDER BY COUNT(QMS_Response.ResponseID) DESC, sum(QuantityRejected) DESC, Question
	END
	ELSE
	SELECT     TOP (10) QMS_QuestionPool.Question, PM_VW_StockCode.DisplayText AS StockCode, COUNT(QMS_SupplierResponseQuestion.ResponseQuestionID) 
						  AS NumChecks, SUM(QMS_SupplierResponse.QuantityAccepted) AS QuantityAccepted,
						  SUM(CASE WHEN APP_QAFlag.QAFlagID = 2 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS AmberRejected,
						  SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS RedRejected,
						  SUM(TEMP_Selected.SelectedValue * QMS_SupplierResponse.QuantityRejected) AS RejectedValue
	FROM         QMS_SupplierResponseQuestion INNER JOIN
						  QMS_QuestionPool ON QMS_SupplierResponseQuestion.QuestionID = QMS_QuestionPool.QuestionID INNER JOIN
						  QMS_SupplierResponse ON QMS_SupplierResponseQuestion.ResponseID = QMS_SupplierResponse.ResponseID INNER JOIN
						  PM_VW_StockCode ON QMS_SupplierResponse.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
						  TEMP_Selected ON QMS_SupplierResponse.ResponseID = TEMP_Selected.SelectedID INNER JOIN
						  APP_QAFlag ON QMS_SupplierResponseQuestion.QAFlagID = APP_QAFlag.QAFlagID
	WHERE     (TEMP_Selected.UserID = @UserID) AND TEMP_Selected.TransactionID = @TransactionID 
	AND (APP_QAFlag.IsRejection = 1)
	GROUP BY QMS_QuestionPool.Question, PM_VW_StockCode.DisplayText, QuantityRejected
	ORDER BY NumChecks DESC, QuantityRejected DESC, QMS_QuestionPool.Question
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_SubChart')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_SubChart] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 15/12/2009
-- Description:	Gets summary data for supplier profile RAG chart
-- Version: 6
-- JE TFS 4770: updated with QA Flag config check
-- JE TFS 5906: updated to make use of warehouse table
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_SubChart
	@UserID int,
	@Offline bit=0,
	@TransactionID uniqueidentifier
AS

--set Override QAFlag option
DECLARE @OverrideQAFlag bit
SET @OverrideQAFlag = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)

IF @Offline = 0
BEGIN
	IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
		BEGIN
			SELECT     QAFlag.QAFlagID, QAFlag.QAFlag, ISNULL(ReportChecks.NumChecks, 0) AS NumChecks
			FROM         (SELECT	QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag, ListIndex
						  FROM      APP_QAFlag) AS QAFlag LEFT OUTER JOIN
									  (SELECT     COUNT(RPT_Warehouse_Response.ResponseID) AS NumChecks, RPT_Warehouse_Response.QAFlagID
									   FROM          TEMP_Selected INNER JOIN
													   RPT_Warehouse_Response ON TEMP_Selected.SelectedID = RPT_Warehouse_Response.ResponseID
									   WHERE      (TEMP_Selected.UserID = @UserID) AND TEMP_Selected.TransactionID = @TransactionID 
									   GROUP BY RPT_Warehouse_Response.QAFlagID) AS ReportChecks
							ON QAFlag.QAFlagID = ReportChecks.QAFlagID
			ORDER BY QAFlag.ListIndex
		END
	ELSE
		BEGIN
			SELECT     QAFlag.QAFlagID, QAFlag.QAFlag, ISNULL(ReportChecks.NumChecks, 0) AS NumChecks
			FROM         (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag, ListIndex
								   FROM          APP_QAFlag) AS QAFlag LEFT OUTER JOIN
									  (SELECT     COUNT(QMS_Response.ResponseID) AS NumChecks, APP_QAFlag.QAFlagID
										FROM          TEMP_Selected INNER JOIN
														QMS_Response ON TEMP_Selected.SelectedID = QMS_Response.ResponseID INNER JOIN
														APP_QAFlag ON --check config value to join on Actual or Override QAFlagID
															(CASE WHEN @OverrideQAFlag = 0 THEN Actual_QAFlagID
																ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
														= APP_QAFlag.QAFlagID
										WHERE      (TEMP_Selected.UserID = @UserID) AND TEMP_Selected.TransactionID = @TransactionID 
										GROUP BY APP_QAFlag.QAFlagID) AS ReportChecks
								ON QAFlag.QAFlagID = ReportChecks.QAFlagID
			ORDER BY QAFlag.ListIndex
		END
END                            
ELSE
	SELECT     QAFlag.QAFlagID, QAFlag.QAFlag, ISNULL(ReportChecks.NumChecks, 0) AS NumChecks
	FROM         (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag, ListIndex
						   FROM          APP_QAFlag) AS QAFlag LEFT OUTER JOIN
							  (SELECT     COUNT(QMS_SupplierResponse.ResponseID) AS NumChecks, QMS_SupplierResponse.QAFlagID
								FROM          TEMP_Selected INNER JOIN
													   QMS_SupplierResponse ON TEMP_Selected.SelectedID = QMS_SupplierResponse.ResponseID
								WHERE      (TEMP_Selected.UserID = @UserID) AND TEMP_Selected.TransactionID = @TransactionID 
								GROUP BY QMS_SupplierResponse.QAFlagID) AS ReportChecks
						ON QAFlag.QAFlagID = ReportChecks.QAFlagID
	ORDER BY QAFlag.ListIndex
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_FillSelected')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_FillSelected] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 14/12/2009
-- Description:	Fills TEMP_Selected with ResponseID for data to show on supplier profile report
-- Version:		14
-- =============================================
-- Jonathan Toolan: 25-Aug-2011 AssessmentTypeID linked from Response, not Assessment.
-- JE TFS 4261: added filtering by FNC_User_StockCode so user permissions are enforced
-- JE TFS 5061: removed old filter options, simplified to use new two tier filtering and added Assessment Types filter
-- JT TFS 5585: Moved sub-query into temp table so that the amount of work that each query does is significantly reduced.
-- JT TFS 5634: Link permissions for stockcode through FNC_ResponseHeader_StockCodeID() to work out which stockcode to check on.
-- JE TFS 5906: updated to make use of warehouse table
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6743: Only filter offline on SupplierID
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7507: Offline is not pre-filtered, so has to be filtered by date.
-- JT CAS-35175-S8P0X3: TEMP_Selected no longer potentially updates wrong users SelectedValue.
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_FillSelected 
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime,
	@SupplierID int,
	@Offline bit,
	@TransactionID uniqueidentifier
	--note: don't need to pass in Assessment Types list as this is taken care of in TEMP_SelectedItem
AS

--Clear temp selected
DELETE FROM TEMP_Selected 
WHERE     (UserID = @UserID)
AND TransactionID = @TransactionID

--Insert into temp selected
IF @Offline = 0
	BEGIN
		IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
		BEGIN
			INSERT INTO TEMP_Selected (SelectedID, UserID, TransactionID)
			SELECT	RPT_Warehouse_Response.ResponseID, @UserID AS UserID, @TransactionID as TransactionID
			FROM    RPT_Warehouse_Response INNER JOIN
						TEMP_ResponseFilter ON RPT_Warehouse_Response.ResponseID = TEMP_ResponseFilter.ResponseID
			WHERE     (SupplierID = @SupplierID) AND (TEMP_ResponseFilter.UserID = @UserID)
			AND TEMP_ResponseFilter.TransactionID = @TransactionID 
		END
		ELSE
		BEGIN
			IF NULLIF(object_id('tempdb..#tmpPermissionResponse'), 0) IS NOT NULL DROP TABLE #tmpPermissionResponse
				SELECT TEMP_ResponseFilter.ResponseID --sub query to get Stock Codes user has permissions for
				INTO #tmpPermissionResponse
				FROM TEMP_ResponseFilter
				INNER JOIN  dbo.FNC_ResponseHeader_StockCodeID() AS ResponseStockCodes ON ResponseStockCodes.ResponseID = TEMP_ResponseFilter.ResponseID 
				INNER JOIN	dbo.FNC_User_StockCodeList(@UserID) AS StockCodes ON ResponseStockCodes.StockCodeID = StockCodes.StockCodeID
				WHERE TEMP_ResponseFilter.UserID = @UserID
				AND TEMP_ResponseFilter.TransactionID = @TransactionID 
			
				INSERT INTO TEMP_Selected (SelectedID, UserID, TransactionID)
				SELECT DISTINCT QMS_Response.ResponseID, @UserID AS UserID, @TransactionID
				FROM         QMS_Response 
				INNER JOIN	QMS_ResponseHeader ON QMS_Response.ResponseID = QMS_ResponseHeader.ResponseID 
				INNER JOIN	QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID 
				INNER JOIN	QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID 
				INNER JOIN	TEMP_SelectedItem ON QMS_Response.AssessmentTypeID = TEMP_SelectedItem.SelectedItemID 
											 AND TEMP_SelectedItem.TransactionID = @TransactionID 
											 AND TEMP_SelectedItem.UserID = @UserID 
				INNER JOIN	#tmpPermissionResponse AS StockCodeResponses ON QMS_Response.ResponseID = StockCodeResponses.ResponseID 
				INNER JOIN	TEMP_ResponseFilter ON QMS_Response.ResponseID = TEMP_ResponseFilter.ResponseID 
												AND TEMP_ResponseFilter.TransactionID = @TransactionID 
												AND (TEMP_ResponseFilter.UserID = @UserID)
				WHERE     (QMS_Response.ResponseStatus = 1) AND (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1) AND 
									  (QMS_Assessment.RecordStatus < 201) AND (QMS_ResponseHeader.Header_SourceID = @SupplierID) AND 
									  (QMS_Response.RecordStatus < 180) 
		END
	END
ELSE
	BEGIN
		INSERT INTO TEMP_Selected (SelectedID, UserID, TransactionID)
		SELECT QMS_SupplierResponse.ResponseID, @UserID AS UserID, @TransactionID 
		FROM         QMS_SupplierResponse 
		WHERE     (SupplierID = @SupplierID) AND (QMS_SupplierResponse.RecordStatus < 201) 
		AND (EndDate BETWEEN @DateFrom AND @DateTo)
		GROUP BY QMS_SupplierResponse.ResponseID
	END

--Update temp selected value with Consignment unit price
IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
	BEGIN
		UPDATE  TEMP_Selected
		SET		SelectedValue = PM_VW_ConsignmentItems.UnitPrice
		FROM    TEMP_Selected INNER JOIN
					  (SELECT     ResponseID, MAX(ConsignmentID) AS ConsignmentID, MAX(StockCodeID) AS StockCodeID
					   FROM          (SELECT	RPT_Warehouse_Response.ResponseID, RPT_Warehouse_Response.ConsignmentID, RPT_Warehouse_Response.StockCodeID
									  FROM		RPT_Warehouse_Response INNER JOIN
													TEMP_Selected AS TEMP_Selected_1 ON RPT_Warehouse_Response.ResponseID = TEMP_Selected_1.SelectedID
									  WHERE     (TEMP_Selected_1.UserID = @UserID)
									  AND TEMP_Selected_1.TransactionID = @TransactionID) AS Header
						GROUP BY ResponseID) AS Response_Consignment
							ON TEMP_Selected.SelectedID = Response_Consignment.ResponseID 
		INNER JOIN PM_VW_ConsignmentItems ON Response_Consignment.ConsignmentID = PM_VW_ConsignmentItems.ConsignmentID 
											AND Response_Consignment.StockCodeID = PM_VW_ConsignmentItems.StockCodeID
		WHERE TEMP_Selected.UserID = @UserID 
		AND TEMP_Selected.TransactionID = @TransactionID
	END
ELSE
	BEGIN
		UPDATE  TEMP_Selected
		SET		SelectedValue = PM_VW_ConsignmentItems.UnitPrice
		FROM    TEMP_Selected INNER JOIN
					  (SELECT     ResponseID, MAX(ConsignmentID) AS ConsignmentID, MAX(StockCodeID) AS StockCodeID
					   FROM          (SELECT	QMS_ResponseHeader.ResponseID, QMS_AssessmentHeader.DataLookupID, 
													CASE WHEN DataLookupID = 5 THEN Header_SourceID ELSE NULL END AS ConsignmentID, 
													CASE WHEN DataLookupID = 2 THEN Header_SourceID ELSE NULL END AS StockCodeID
									  FROM		QMS_ResponseHeader INNER JOIN
													QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
													TEMP_Selected AS TEMP_Selected_1 ON QMS_ResponseHeader.ResponseID = TEMP_Selected_1.SelectedID
									  WHERE     (QMS_AssessmentHeader.DataLookupID IS NOT NULL) AND (QMS_AssessmentHeader.DataTypeID = 9) AND 
													 (QMS_ResponseHeader.Header_Text IS NOT NULL) AND (QMS_ResponseHeader.RecordStatus < 201) AND 
													 (TEMP_Selected_1.UserID = @UserID)
													 AND TEMP_Selected_1.TransactionID = @TransactionID) AS Header
									  GROUP BY ResponseID) AS Response_Consignment
						ON TEMP_Selected.SelectedID = Response_Consignment.ResponseID INNER JOIN
					 PM_VW_ConsignmentItems ON Response_Consignment.ConsignmentID = PM_VW_ConsignmentItems.ConsignmentID AND 
													Response_Consignment.StockCodeID = PM_VW_ConsignmentItems.StockCodeID
		WHERE TEMP_Selected.UserID = @UserID 
		AND TEMP_Selected.TransactionID = @TransactionID
	END
	
--Clear temp response header
DELETE FROM TEMP_ResponseHeader
WHERE     (UserID = @UserID)
AND TransactionID = @TransactionID

IF EXISTS(SELECT ConsignmentID FROM PM_VW_Consignment WHERE SupplierID = @SupplierID) --Add consignment totals for same time period
	INSERT INTO TEMP_ResponseHeader
						  (UserID, AssessmentHeaderID, SourceObjectID, Header_Value, TransactionID)
	SELECT     @UserID AS UserID, MAX(PM_VW_Consignment.ConsignmentID) AS ConsignmentID, SUM(PM_VW_ConsignmentItems.QuantityAccepted) AS TotalReceived, 
                      SUM(ISNULL(PM_VW_ConsignmentItems.UnitPrice, 0) * PM_VW_ConsignmentItems.QuantityAccepted) AS ReceivedValue, @TransactionID
	FROM         PM_VW_Consignment INNER JOIN
						  PM_VW_ConsignmentItems ON PM_VW_Consignment.ConsignmentID = PM_VW_ConsignmentItems.ConsignmentID
	WHERE     (PM_VW_Consignment.DateRequired BETWEEN @DateFrom AND @DateTo) AND (PM_VW_Consignment.SupplierID = @SupplierID)
	HAVING      (MAX(PM_VW_Consignment.ConsignmentID) IS NOT NULL)
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_GetTempTransaction_TokenUserID')
	EXEC('CREATE FUNCTION [dbo].[FNC_GetTempTransaction_TokenUserID]() RETURNS bit AS BEGIN RETURN 0 END')
GO
-- =============================================
-- Author:		Jonathan Toolan
-- Create date: 02/03/2016
-- Description:	Gets a UserTokenID from the database for use in TEMP_Selected tables.
-- Version: 	1
-- =============================================
-- JT TFS 13-42222: 02/03/2016 - Created
-- =============================================
ALTER FUNCTION dbo.FNC_GetTempTransaction_TokenUserID (@TransactionID uniqueidentifier, @UserID int)
RETURNS int
AS
BEGIN
	RETURN (
		SELECT ISNULL(
			(	SELECT TokenUserID 
				FROM TEMP_Transaction
				WHERE TransactionID = @TransactionID
				AND ActualUserID = @UserID
			), @UserID
		)
	)
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_TEMP_ClearTransaction')
	EXEC ('CREATE PROCEDURE [dbo].[APP_TEMP_ClearTransaction] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Jonathan Toolan
-- Create date: 19/02/2016
-- Description:	Clear out specific transaction entries used in temp tables.
-- Version: 	2
-- =============================================
-- JT TFS 13-41688: 19/02/2016 - Created
-- JT TFS 13-42222: 02/03/2016 - Also delete expired transactions
-- =============================================
ALTER PROCEDURE APP_TEMP_ClearTransaction
	@TransactionID uniqueidentifier
AS
BEGIN
	-- Always clear out transactions older than a day. A transaction used in a report should only exist for a few minutes at most.
	DECLARE @expiryDate datetime
	SELECT @expiryDate = dateadd(day, -1, getdate())

	DECLARE @expiredTransactions TABLE (
		TransactionID uniqueidentifier
	)
	INSERT @expiredTransactions (TransactionID)
	SELECT TransactionID FROM TEMP_Transaction t
	WHERE t.StartTime < @expiryDate

	DELETE TEMP_Selected WHERE TransactionID = @TransactionID OR TransactionID IN (SELECT TransactionID FROM @expiredTransactions);
	DELETE TEMP_SelectedItem WHERE TransactionID = @TransactionID OR TransactionID IN (SELECT TransactionID FROM @expiredTransactions);
	DELETE TEMP_ResponseFilter WHERE TransactionID = @TransactionID OR TransactionID IN (SELECT TransactionID FROM @expiredTransactions);
	DELETE TEMP_ResponseGroupBy WHERE TransactionID = @TransactionID OR TransactionID IN (SELECT TransactionID FROM @expiredTransactions);
	DELETE TEMP_ResponseHeader WHERE TransactionID = @TransactionID OR TransactionID IN (SELECT TransactionID FROM @expiredTransactions);

	DELETE TEMP_Transaction WHERE TransactionID = @TransactionID OR TransactionID IN (SELECT TransactionID FROM @expiredTransactions);
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_Temp_ReportFilter')
	EXEC ('CREATE PROCEDURE [dbo].[APP_Temp_ReportFilter] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		JToolan
-- Create date: 22/03/2012
-- Description:	Works out which ResponseIDs need to be put in the filter.
--              Can be used by any report.
-- Version:		21
-- =============================================
-- j.toolan 30-mar-2012: If @AssessmentTypeID_LIST is empty, then join all assessments
-- j.toolan 13 Apr 2012: Fixed bug with 2nd filter incorrectly clearing.
-- j.toolan 18 Apr 2012: Added Code to filter by stockcode where appropriate (renamed to ReportFilter)
-- JE 25 Apr 2012: made ReportFilterID1 optional, so can just filter by AssessmentType if necessary (TFS 3382)
-- j.toolan 6 July 2012: TFS 4013: Dynamic filtering added.
-- JE TFS 4740: removed GROUP BY for performance reasons
-- JE TFS 5037: corrected @ReportFilterID2 check as 2nd tier filter wasn't working
-- MP TFS 5289: changed date format used for custom filter functions
-- JT TFS 5396: changed code to allow commas in FilterTexts passed in as CSV. (replace "," with "&comma;" )
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 6689: FNC_ReportFilter_Response takes UserID parameter.
-- JT TFS 8117: Added offline filtering.
-- MP TFS 8227: Made temp table user specific and added performance logging
-- JE TFS 8274: Put result set of FNC_User_ResponseList into temp table before using it
-- JT TFS 13-667: Do not filter responses if the date range is too big.
-- JT TFS 13-1076: Search either QMS_Response or QMS_SupplierResponse, but not both.
-- JF TFS 2281 - if filter doesn't have a sourceid, this wasn't working
-- JT TFS 13-17467: Only include checks which are eligible candidates for the data warehouse.
-- JT TFS 13-38578: Convert SP to use QRY_User_ResponseList (rather than FNC_UserResponseList)
-- JT TFS 13-40008: 22/02/2016 - Added transaction IDs to response list generation, and extracted stockcode filtering
-- JT TFS 13-42222: 02/03/2016 - Added link to TokenUserID so that reports can be run simultaneously.
-- =============================================
ALTER PROCEDURE [dbo].[APP_Temp_ReportFilter]
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime,
	@AssessmentTypeID_LIST nvarchar(255),

	@ReportFilterID1 int = NULL,
	@FilterID_LIST nvarchar(4000),
	@FilterText_LIST nvarchar(4000),
	@ReportFilterID2 int = NULL,
	@FilterID2 int = NULL,
	@FilterText2 nvarchar(255) = NULL,
	
	@Offline bit = NULL,
	@SearchLocalChecksIfOffline bit = 1, -- Use QMS_Response even if offline (default)
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	SET NOCOUNT ON
	
	EXEC SAVE_TEMP_Transaction @TransactionID, @UserID

	DECLARE @TokenUserID int;
	SELECT @TokenUserID = dbo.FNC_GetTempTransaction_TokenUserID(@TransactionID, @UserID)

	DECLARE @DateCalled DATETIME
	SET @DateCalled = GETDATE()	
	IF @Offline IS NULL SELECT @Offline = CASE WHEN db_name() = 'GreenlightQC_Mobile' THEN 1 ELSE 0 END
	
	DECLARE @SearchOffline bit -- if this is set, then we search in QMS_SupplierResponse.
	SELECT @SearchOffline  = CASE WHEN @Offline = 1 AND @SearchLocalChecksIfOffline = 0 THEN 1 ELSE 0 END
		
	-- Do not filter responses if the date range is not specified
	DECLARE @performResponseFiltering bit 
	SELECT @performResponseFiltering = CASE WHEN @DateFrom <= '1 Jan 2001' and datediff(year, @DateFrom, @DateTo) >= 10 THEN 0 ELSE 1 END
	
	DELETE TEMP_ResponseFilter WHERE UserID = @TokenUserID OR TransactionID = @TransactionID

	-- If the first criteria is empty, but the second is not, swap them over.
	IF (LEN(@FilterID_LIST) = 0 AND LEN(@FilterText_LIST) = 0 -- first criteria is empty?!
	AND @ReportFilterID2 IS NOT NULL AND (@FilterID2 IS NOT NULL OR @FilterText2 IS NOT NULL)) -- but the second is not...
	BEGIN -- The great fairness swaparoo! (in two steps)
		SELECT	@ReportFilterID1 = @ReportFilterID2,
				@FilterID_LIST = CONVERT(nvarchar(4000), @FilterID2),
				@FilterText_LIST = @FilterText2
				
		SELECT @ReportFilterID2 = NULL, @FilterID2 = NULL, @FilterText2 = NULL
	END

	-- Is the second question correctly formulated? You must have an id or a text (or both) but not neither.
	-- JE TFS 5037 - corrected this as it was always setting @ReportFilterID2 to NULL for filters with no ID (i.e. text only)
	IF @ReportFilterID2 IS NOT NULL AND (NULLIF(@FilterID2, 0) IS NULL) AND (LEN(ISNULL(@FilterText2, '')) = 0)
	BEGIN
		SET @ReportFilterID2 = NULL
	END

	DECLARE @FilterItem TABLE (pos int not null, id int, txt nvarchar(255))
	INSERT @FilterItem (pos, id, txt)
	
	-- tfs 2281 - if filter doesn't have a sourceidm this wasn't working
	SELECT 	ISNULL(a.Position, b.Position) pos, CONVERT(int, ISNULL(a.Token,0)) id, replace(b.Token, '&comma;', ',') txt
	FROM dbo.FNC_SplitString(@FilterID_LIST, ',') a
	FULL JOIN dbo.FNC_SplitString(@FilterText_LIST, ',') b on a.Position = b.Position

	DECLARE @AssessmentType TABLE (id int not null)
	INSERT @AssessmentType (id) 
	SELECT CONVERT(int, Token) FROM dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',')
	INNER JOIN APP_AssessmentType ON CONVERT(int, Token) = AssessmentTypeID 
	IF NOT EXISTS (SELECT * FROM @AssessmentType) 
	BEGIN -- if we tried to join, and got nothing, that means that this report wants all non-hidden types.
		INSERT @AssessmentType (id) SELECT AssessmentTypeID FROM APP_AssessmentType WHERE Hide = 0
	END

	-- Do some custom filtering if it is set up. This acts like "AND" (restricting results)
	DECLARE @CustomFilterFunction nvarchar(50)
	SET @CustomFilterFunction = NULL
	SELECT @CustomFilterFunction = APP_ReportFilter.FilterFunction 
	FROM APP_ReportFilter 
	WHERE ReportFilterID = @ReportFilterID1
	
	DECLARE @customSql nvarchar(1000)
	SET @customSql = NULL
	CREATE TABLE #tmpCustomFilter (UserID INT, ResponseID int, GroupByText nvarchar(50))

	DECLARE @FilteredResponse TABLE (id int not null)
	WHILE EXISTS (SELECT * FROM @FilterItem)
	BEGIN
		DECLARE @pos int, @FilterID int, @FilterText nvarchar(255)
		SELECT @pos = pos,  @FilterID = id, @FilterText = txt
		FROM @FilterItem
		WHERE pos = (SELECT MIN(pos) FROM @FilterItem)

		IF @performResponseFiltering = 1
		BEGIN
			-- either the custom function runs, *OR* the generic one, *OR* the offline one, but not more than one.
			IF @SearchOffline = 1
			BEGIN
				INSERT @FilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response_Offline(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
			END
			ELSE IF @CustomFilterFunction IS NOT NULL
			BEGIN
				SELECT @customSql = 'INSERT #tmpCustomFilter (UserID, ResponseID, GroupByText) SELECT ' + convert(nvarchar(10), @TokenUserID) + ' AS UserID, ResponseID, GroupByText FROM dbo.[' + @CustomFilterFunction + ']' + 
									'(' + convert(nvarchar(10), @UserID) + ', ' +
									convert(nvarchar(30), @FilterID) + ', ' +
									'''' + convert(nvarchar(50), @FilterText) + ''', ' + 
									'''' + convert(nvarchar(30), @DateFrom, 13) + ''', ' +
									'''' + convert(nvarchar(30), @DateTo, 13) + ''');'
				
				PRINT @customSql
				
				IF @customSql IS NOT NULL
					EXECUTE (@customSql)
			END
			ELSE
			BEGIN
				INSERT @FilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response(@ReportFilterID1, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)
			END
		END
		
		DELETE @FilterItem WHERE pos = @pos
	END

	-- insert all our new custom filtered items (added to temp table in loop above)
	INSERT @FilteredResponse (id)
	SELECT ResponseID 
	FROM #tmpCustomFilter
	WHERE UserID = @TokenUserID

	-- If there is another filter, it acts as an "AND" (restricting results returned), not an "OR" (expanding results returned). 
	IF @ReportFilterID2 IS NOT NULL
	BEGIN 
		IF @performResponseFiltering = 1
		BEGIN
			DECLARE @SecondFilteredResponse TABLE (id int)
			
			SELECT @CustomFilterFunction = APP_ReportFilter.FilterFunction 
			FROM APP_ReportFilter 
			WHERE ReportFilterID = @ReportFilterID2
		
			IF @SearchOffline = 1
			BEGIN
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response_Offline(@ReportFilterID2, @FilterID2, @FilterText2, @DateFrom, @DateTo, @UserID)
			END 
			ELSE IF @CustomFilterFunction IS NOT NULL
			BEGIN
				DELETE #tmpCustomFilter WHERE UserID = @TokenUserID --clear it out for second filter.
				
				SELECT @customSql = 'INSERT #tmpCustomFilter (UserID, ResponseID, GroupByText) SELECT ' + convert(nvarchar(10), @TokenUserID) + ' AS UserID, ResponseID, GroupByText FROM dbo.[' + @CustomFilterFunction + ']' + 
									'(' + convert(nvarchar(10), @UserID) + ', ' +
									convert(nvarchar(30), @FilterID2) + ', ' +
									'''' + convert(nvarchar(50), @FilterText2) + ''', ' + 
									'''' + convert(nvarchar(30), @DateFrom, 13) + ''', ' +
									'''' + convert(nvarchar(30), @DateTo, 13) + ''');'

				IF @customSql IS NOT NULL
					EXECUTE (@customSql)
				
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID FROM #tmpCustomFilter WHERE UserID = @TokenUserID
			END
			ELSE
			BEGIN
				INSERT @SecondFilteredResponse (id)
				SELECT ResponseID 
				FROM dbo.FNC_ReportFilter_Response (@ReportFilterID2, @FilterID2, @FilterText2, @DateFrom, @DateTo, @UserID)
			END
			
			DELETE @FilteredResponse 
			WHERE id NOT IN (
				SELECT id FROM @SecondFilteredResponse
			)
		END
	END

	-- Remove all responses which are not suitable candidates for the data warehouse
	DELETE @FilteredResponse
	WHERE dbo.FNC_Response_ShouldIncludeInDataWarehouse(id) = 0


	IF @performResponseFiltering = 1
	BEGIN
		IF @SearchOffline = 1
		BEGIN
			IF @ReportFilterID1 IS NOT NULL
				INSERT TEMP_ResponseFilter (UserID, ResponseID)
				SELECT @TokenUserID as UserID, r.ResponseID 
				FROM QMS_SupplierResponse r
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				INNER JOIN @FilteredResponse fr ON fr.id = r.ResponseID
				WHERE EndDate BETWEEN @DateFrom AND @DateTo
				AND   r.ResponseID NOT IN (SELECT ResponseID FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID)
				GROUP BY r.ResponseID
			ELSE
				INSERT TEMP_ResponseFilter (UserID, ResponseID)
				SELECT @TokenUserID as UserID, r.ResponseID 
				FROM QMS_SupplierResponse r 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id	
				WHERE EndDate BETWEEN @DateFrom AND @DateTo
				AND   r.ResponseID NOT IN (SELECT ResponseID FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID)
				GROUP BY r.ResponseID
		END
		ELSE
		BEGIN
			--JE TFS 82774 - temp table of responses user can view (moved from direct access in query to improve performance)
			IF nullif(object_id('tempdb..#tmpUserResponses'), 0) IS NOT NULL DROP TABLE #tmpUserResponses
			CREATE TABLE #tmpUserResponses (UserID int, ResponseID int, ResponseStatus int, QAFlagID int)
			INSERT #tmpUserResponses (UserID, ResponseID, ResponseStatus, QAFlagID)
			EXEC QRY_User_ResponseList @UserID, @DateFrom, @DateTo
			
			IF @ReportFilterID1 IS NOT NULL
				-- Only return responses that the user is allowed to see, filter by Assessment Type, and use filter
				INSERT TEMP_ResponseFilter (UserID, ResponseID, TransactionID)
				SELECT @TokenUserID as UserID, rl.ResponseID, @TransactionID
				FROM #tmpUserResponses AS rl
				INNER JOIN QMS_Response r ON rl.ResponseID = r.ResponseID 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				INNER JOIN @FilteredResponse fr ON fr.id = r.ResponseID
				WHERE rl.UserID = @UserID
				GROUP BY rl.ResponseID
			
			IF @ReportFilterID1 IS NULL
				-- Only return responses that the user is allowed to see, but there has been no filter specified, so only filter by Assessment Type (if specified)
				INSERT TEMP_ResponseFilter (UserID, ResponseID, TransactionID)
				SELECT @TokenUserID as UserID, rl.ResponseID, @TransactionID
				FROM #tmpUserResponses AS rl
				INNER JOIN QMS_Response r ON rl.ResponseID = r.ResponseID 
				INNER JOIN @AssessmentType at ON r.AssessmentTypeID = at.id
				WHERE rl.UserID = @UserID	
				GROUP BY rl.ResponseID
			
			DROP TABLE #tmpUserResponses
		END

	END
	
	DECLARE @MonitorEnabled TINYINT
	SELECT @MonitorEnabled = ISNULL(dbo.FNC_GetOptionValue(92),0)

	IF @MonitorEnabled = 1 --Log filter to QMS_PerformanceLog
	BEGIN
		DECLARE @ReportString NVARCHAR(max)
		SET @ReportString = 'APP_Temp_ReportFilter '
		SET @ReportString = @ReportString + '@UserID = ' + CAST(@UserID AS NVARCHAR(10))
		SET @ReportString = @ReportString + ', @DateFrom = ' + CAST(@DateFrom AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @DateTo = ' + CAST(@DateTo AS NVARCHAR(11))
		SET @ReportString = @ReportString + ', @AssessmentTypeID_LIST = ' + CASE WHEN LEN(@AssessmentTypeID_LIST)=0 THEN 'NULL' ELSE @AssessmentTypeID_LIST END
		IF @ReportFilterID1>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID1 = ' + CAST(@ReportFilterID1 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID_LIST = ' + CASE WHEN LEN(@FilterID_LIST)=0 THEN 'NULL' ELSE @FilterID_LIST END
			SET @ReportString = @ReportString + ', @FilterText_LIST = ' + CASE WHEN LEN(@FilterText_LIST)=0 THEN 'NULL' ELSE @FilterText_LIST END
		END
		IF @ReportFilterID2>0
		BEGIN
			SET @ReportString = @ReportString + ', @ReportFilterID2 = ' + CAST(@ReportFilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterID2 = ' + CAST(@FilterID2 AS NVARCHAR(10))
			SET @ReportString = @ReportString + ', @FilterText2 = ' + CASE WHEN LEN(@FilterText2)=0 THEN 'NULL' ELSE @FilterText2 END
		END
		SET @ReportString = @ReportString + ', @TransactionID = ' + CASE WHEN @TransactionID IS NULL THEN 'NULL' ELSE '''' + CAST(@TransactionID AS NVARCHAR(36)) + '''' END

		DECLARE @ResponseCount INT
		SELECT @ResponseCount = COUNT(ResponseID) FROM TEMP_ResponseFilter WHERE UserID = @TokenUserID
		
		INSERT INTO dbo.QMS_PerformanceLog
				( Action ,
				  StartTime ,
				  EndTime ,
				  RecordsAffected
				)
		VALUES  ( @ReportString ,
				  @DateCalled ,
				  GETDATE() ,
				  @ResponseCount
				  )
	END

	SET NOCOUNT OFF
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'APP_TEMP_SelectedItem')
	EXEC ('CREATE PROCEDURE [dbo].[APP_TEMP_SelectedItem] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 13/07/2009
-- Description:	Appends new temp selected item ID
-- Version:		2
-- =============================================
-- JT TFS 13-42222: 03/03/2016 - Selected Items can store a TransactionID now.
-- =============================================
ALTER PROCEDURE APP_TEMP_SelectedItem 
	@UserID int, 
	@SelectedID int,
	@SelectedText nvarchar(255) = NULL,
	@TransactionID uniqueidentifier = NULL
AS

IF NOT EXISTS(SELECT * FROM TEMP_SelectedItem WHERE (SelectedItemID = @SelectedID) AND (UserID = @UserID))
	INSERT INTO TEMP_SelectedItem
						  (UserID, SelectedItemID, SelectedItemText, TransactionID)
	VALUES     (@UserID,@SelectedID,@SelectedText,@TransactionID)
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_ResponseQuestion_MultiList')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_ResponseQuestion_MultiList] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Jon Edwards
-- Create date: 03/04/2013
-- Description:	Gets question responses for selected (parent) response
-- Version:		6
-- =============================================
-- JE TFS 6272 - added ChildCheckCount column so we can derive Postfix for each row
-- JE TFS 7953 - added Summary and Summary_QAFlagID columns (stored in parent check's record)
-- JE TFS 13-466: added Summary_SourceID col
-- JT TFS 13-612: Hide children which have been voided and cancelled.
-- SS TFS 42535: Added a temporary table which has ResponseID AND ResponseStatus inserted when a response has parent GUID (Performance improvement)
-- =============================================
ALTER PROCEDURE [dbo].[QRY_ResponseQuestion_MultiList]
@ResponseID int
AS
BEGIN
	--get GUID for parent response
	DECLARE @ParentGUID AS uniqueidentifier
	SELECT	@ParentGUID = GUID
	FROM	QMS_Response
	WHERE	(ResponseID = @ResponseID)

	--get count of Child Checks for @ParentGUID
	DECLARE	@ChildCheckCount AS int
	SELECT	@ChildCheckCount = COUNT(ResponseID)
	FROM	QMS_Response
	WHERE	(ParentResponseGUID = @ParentGUID)
	AND     (QMS_Response.RecordStatus < 180) -- Hide voided and cancelled checks.

	IF NULLIF(object_id('tempdb..#responses'), 0) IS NOT NULL DROP TABLE #responses
	CREATE TABLE #responses (
		ResponseID int,
		RecordStatus int
	)
	INSERT #responses(ResponseID, RecordStatus)
	SELECT ResponseID, RecordStatus
	FROM QMS_Response 
	WHERE ParentResponseGUID = @ParentGUID

	SELECT	QMS_ResponseQuestion.ResponseID, QMS_ResponseQuestion.ResponseQuestionID, QMS_ResponseQuestion.AssessmentQuestionID,
	QMS_ResponseQuestion.Response_Text, QMS_ResponseQuestion.Response_Boolean, QMS_ResponseQuestion.Response_Date,
	QMS_ResponseQuestion.Response_Numeric, QMS_ResponseQuestion.Response_SourceID, QMS_ResponseQuestion.Score,
	QMS_ResponseQuestion.Recommended_QAFlagID, QMS_ResponseQuestion.QuestionComments, QMS_ResponseQuestion.SampleNo,
	@ChildCheckCount AS  ChildCheckCount, Summary.Response_SourceID AS Summary_SourceID, Summary.Response_Text AS Summary,
	Summary.Recommended_QAFlagID AS Summary_QAFlagID
	FROM	QMS_ResponseQuestion 
	INNER JOIN #responses AS QMS_Response ON QMS_ResponseQuestion.ResponseID = QMS_Response.ResponseID 
	LEFT OUTER JOIN(SELECT		AssessmentQuestionID, Response_SourceID, Response_Text, QMS_ResponseQuestion.Recommended_QAFlagID
					FROM	QMS_ResponseQuestion INNER JOIN
					QMS_Response ON QMS_ResponseQuestion.ResponseID = QMS_Response.ResponseID
					WHERE	(QMS_Response.GUID = @ParentGUID)) AS Summary ON QMS_ResponseQuestion.AssessmentQuestionID = Summary.AssessmentQuestionID
	WHERE	(QMS_ResponseQuestion.RecordStatus < 201)
	AND     (QMS_Response.RecordStatus < 180) -- Hide voided and cancelled checks.
	ORDER BY QMS_ResponseQuestion.ResponseID
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_FillSelected')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_FillSelected] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 14/12/2009
-- Description:	Fills TEMP_Selected with ResponseID for data to show on supplier profile report
-- Version:		17
-- =============================================
-- Jonathan Toolan: 25-Aug-2011 AssessmentTypeID linked from Response, not Assessment.
-- JE TFS 4261: added filtering by FNC_User_StockCode so user permissions are enforced
-- JE TFS 5061: removed old filter options, simplified to use new two tier filtering and added Assessment Types filter
-- JT TFS 5585: Moved sub-query into temp table so that the amount of work that each query does is significantly reduced.
-- JT TFS 5634: Link permissions for stockcode through FNC_ResponseHeader_StockCodeID() to work out which stockcode to check on.
-- JE TFS 5906: updated to make use of warehouse table
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6743: Only filter offline on SupplierID
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7507: Offline is not pre-filtered, so has to be filtered by date.
-- JT CAS-35175-S8P0X3: TEMP_Selected no longer potentially updates wrong users SelectedValue.
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- JT TFS 13-42222: 02/03/2016 - Use TokenUserID instead of UserID directly (+fix on FNC_User_StockCodeList call, +optimisation)
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_FillSelected 
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime,
	@SupplierID int,
	@Offline bit,
	@TransactionID uniqueidentifier
	--note: don't need to pass in Assessment Types list as this is taken care of in TEMP_SelectedItem
AS

DECLARE @TokenUserID int;
SELECT @TokenUserID = dbo.FNC_GetTempTransaction_TokenUserID(@TransactionID, @UserID)


--Clear temp selected
DELETE FROM TEMP_Selected 
WHERE     (UserID = @TokenUserID)

--Insert into temp selected
IF @Offline = 0
	BEGIN
		IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
		BEGIN
			INSERT INTO TEMP_Selected (SelectedID, UserID, TransactionID)
			SELECT	RPT_Warehouse_Response.ResponseID, @TokenUserID AS UserID, @TransactionID as TransactionID
			FROM    RPT_Warehouse_Response INNER JOIN
						TEMP_ResponseFilter ON RPT_Warehouse_Response.ResponseID = TEMP_ResponseFilter.ResponseID
			WHERE     (SupplierID = @SupplierID) 
			AND		TEMP_ResponseFilter.TransactionID = @TransactionID 
		END
		ELSE
		BEGIN
			-- put the values from reponsefilter in TEMP_Selected briefly so that we can use FNC_.._Selected function. 
			INSERT TEMP_Selected (UserID, SelectedID, TransactionID)
			SELECT UserId, ResponseID, TransactionID
			FROM TEMP_ResponseFilter
			WHERE TransactionID = @TransactionID

			IF NULLIF(object_id('tempdb..#tmpPermissionResponse'), 0) IS NOT NULL DROP TABLE #tmpPermissionResponse
			SELECT TEMP_ResponseFilter.ResponseID --sub query to get Stock Codes user has permissions for
			INTO #tmpPermissionResponse
			FROM TEMP_ResponseFilter
			INNER JOIN  dbo.FNC_ResponseHeader_StockCodeID_Selected(@TokenUserID) AS ResponseStockCodes ON ResponseStockCodes.ResponseID = TEMP_ResponseFilter.ResponseID 
			INNER JOIN	dbo.FNC_User_StockCodeList(@UserID) AS StockCodes ON ResponseStockCodes.StockCodeID = StockCodes.StockCodeID
			WHERE TEMP_ResponseFilter.TransactionID = @TransactionID 

			-- Remove the values afterwards.
			DELETE TEMP_Selected
			WHERE TransactionID = @TransactionID
			
			-- Put the real ones we actually want in there.
			INSERT INTO TEMP_Selected (SelectedID, UserID, TransactionID)
			SELECT DISTINCT QMS_Response.ResponseID, @TokenUserID AS UserID, @TransactionID
			FROM         QMS_Response 
			INNER JOIN	QMS_ResponseHeader ON QMS_Response.ResponseID = QMS_ResponseHeader.ResponseID 
			INNER JOIN	QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID 
			INNER JOIN	QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID 
			INNER JOIN	TEMP_SelectedItem ON QMS_Response.AssessmentTypeID = TEMP_SelectedItem.SelectedItemID 
											AND TEMP_SelectedItem.TransactionID = @TransactionID 
			INNER JOIN	#tmpPermissionResponse AS StockCodeResponses ON QMS_Response.ResponseID = StockCodeResponses.ResponseID 
			INNER JOIN	TEMP_ResponseFilter ON QMS_Response.ResponseID = TEMP_ResponseFilter.ResponseID 
											AND TEMP_ResponseFilter.TransactionID = @TransactionID 
			WHERE     (QMS_Response.ResponseStatus = 1) AND (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1) AND 
									(QMS_Assessment.RecordStatus < 201) AND (QMS_ResponseHeader.Header_SourceID = @SupplierID) AND 
									(QMS_Response.RecordStatus < 180) 
		END
	END
ELSE
	BEGIN
		INSERT INTO TEMP_Selected (SelectedID, UserID, TransactionID)
		SELECT QMS_SupplierResponse.ResponseID, @TokenUserID AS UserID, @TransactionID 
		FROM         QMS_SupplierResponse 
		WHERE     (SupplierID = @SupplierID) AND (QMS_SupplierResponse.RecordStatus < 201) 
		AND (EndDate BETWEEN @DateFrom AND @DateTo)
		GROUP BY QMS_SupplierResponse.ResponseID
	END

--Update temp selected value with Consignment unit price
IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
	BEGIN
		UPDATE  TEMP_Selected
		SET		SelectedValue = PM_VW_ConsignmentItems.UnitPrice
		FROM    TEMP_Selected INNER JOIN
					  (SELECT     ResponseID, MAX(ConsignmentID) AS ConsignmentID, MAX(StockCodeID) AS StockCodeID
					   FROM          (SELECT	RPT_Warehouse_Response.ResponseID, RPT_Warehouse_Response.ConsignmentID, RPT_Warehouse_Response.StockCodeID
									  FROM		RPT_Warehouse_Response INNER JOIN
													TEMP_Selected AS TEMP_Selected_1 ON RPT_Warehouse_Response.ResponseID = TEMP_Selected_1.SelectedID
									  WHERE     TEMP_Selected_1.TransactionID = @TransactionID) AS Header
						GROUP BY ResponseID) AS Response_Consignment
							ON TEMP_Selected.SelectedID = Response_Consignment.ResponseID 
		INNER JOIN PM_VW_ConsignmentItems ON Response_Consignment.ConsignmentID = PM_VW_ConsignmentItems.ConsignmentID 
											AND Response_Consignment.StockCodeID = PM_VW_ConsignmentItems.StockCodeID
		WHERE TEMP_Selected.TransactionID = @TransactionID
	END
ELSE
	BEGIN
		UPDATE  TEMP_Selected
		SET		SelectedValue = PM_VW_ConsignmentItems.UnitPrice
		FROM    TEMP_Selected INNER JOIN
					  (SELECT     ResponseID, MAX(ConsignmentID) AS ConsignmentID, MAX(StockCodeID) AS StockCodeID
					   FROM          (SELECT	QMS_ResponseHeader.ResponseID, QMS_AssessmentHeader.DataLookupID, 
													CASE WHEN DataLookupID = 5 THEN Header_SourceID ELSE NULL END AS ConsignmentID, 
													CASE WHEN DataLookupID = 2 THEN Header_SourceID ELSE NULL END AS StockCodeID
									  FROM		QMS_ResponseHeader INNER JOIN
													QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
													TEMP_Selected AS TEMP_Selected_1 ON QMS_ResponseHeader.ResponseID = TEMP_Selected_1.SelectedID
									  WHERE     (QMS_AssessmentHeader.DataLookupID IS NOT NULL) AND (QMS_AssessmentHeader.DataTypeID = 9) AND 
													 (QMS_ResponseHeader.Header_Text IS NOT NULL) AND (QMS_ResponseHeader.RecordStatus < 201) AND 
													 TEMP_Selected_1.TransactionID = @TransactionID) AS Header
									  GROUP BY ResponseID) AS Response_Consignment
						ON TEMP_Selected.SelectedID = Response_Consignment.ResponseID INNER JOIN
					 PM_VW_ConsignmentItems ON Response_Consignment.ConsignmentID = PM_VW_ConsignmentItems.ConsignmentID AND 
													Response_Consignment.StockCodeID = PM_VW_ConsignmentItems.StockCodeID
		WHERE TEMP_Selected.TransactionID = @TransactionID
	END
	
--Clear temp response header
DELETE FROM TEMP_ResponseHeader
WHERE     (UserID = @TokenUserID)

IF EXISTS(SELECT ConsignmentID FROM PM_VW_Consignment WHERE SupplierID = @SupplierID) --Add consignment totals for same time period
	INSERT INTO TEMP_ResponseHeader
						  (UserID, AssessmentHeaderID, SourceObjectID, Header_Value, TransactionID)
	SELECT     @TokenUserID AS UserID, MAX(PM_VW_Consignment.ConsignmentID) AS ConsignmentID, SUM(PM_VW_ConsignmentItems.QuantityAccepted) AS TotalReceived, 
                      SUM(ISNULL(PM_VW_ConsignmentItems.UnitPrice, 0) * PM_VW_ConsignmentItems.QuantityAccepted) AS ReceivedValue, @TransactionID
	FROM         PM_VW_Consignment INNER JOIN
						  PM_VW_ConsignmentItems ON PM_VW_Consignment.ConsignmentID = PM_VW_ConsignmentItems.ConsignmentID
	WHERE     (PM_VW_Consignment.DateRequired BETWEEN @DateFrom AND @DateTo) AND (PM_VW_Consignment.SupplierID = @SupplierID)
	HAVING      (MAX(PM_VW_Consignment.ConsignmentID) IS NOT NULL)
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_SubChart')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_SubChart] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 15/12/2009
-- Description:	Gets summary data for supplier profile RAG chart
-- Version:		7
-- JE TFS 4770: updated with QA Flag config check
-- JE TFS 5906: updated to make use of warehouse table
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- JT TFS 13-42222: 02/03/2016 - Do not use UserID directly for TEMP tables
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_SubChart
	@UserID int,
	@Offline bit=0,
	@TransactionID uniqueidentifier
AS

--set Override QAFlag option
DECLARE @OverrideQAFlag bit
SET @OverrideQAFlag = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)

IF @Offline = 0
BEGIN
	IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
		BEGIN
			SELECT     QAFlag.QAFlagID, QAFlag.QAFlag, ISNULL(ReportChecks.NumChecks, 0) AS NumChecks
			FROM         (SELECT	QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag, ListIndex
						  FROM      APP_QAFlag) AS QAFlag LEFT OUTER JOIN
									  (SELECT     COUNT(RPT_Warehouse_Response.ResponseID) AS NumChecks, RPT_Warehouse_Response.QAFlagID
									   FROM          TEMP_Selected INNER JOIN
													   RPT_Warehouse_Response ON TEMP_Selected.SelectedID = RPT_Warehouse_Response.ResponseID
									   WHERE      TEMP_Selected.TransactionID = @TransactionID 
									   GROUP BY RPT_Warehouse_Response.QAFlagID) AS ReportChecks
							ON QAFlag.QAFlagID = ReportChecks.QAFlagID
			ORDER BY QAFlag.ListIndex
		END
	ELSE
		BEGIN
			SELECT     QAFlag.QAFlagID, QAFlag.QAFlag, ISNULL(ReportChecks.NumChecks, 0) AS NumChecks
			FROM         (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag, ListIndex
								   FROM          APP_QAFlag) AS QAFlag LEFT OUTER JOIN
									  (SELECT     COUNT(QMS_Response.ResponseID) AS NumChecks, APP_QAFlag.QAFlagID
										FROM          TEMP_Selected INNER JOIN
														QMS_Response ON TEMP_Selected.SelectedID = QMS_Response.ResponseID INNER JOIN
														APP_QAFlag ON --check config value to join on Actual or Override QAFlagID
															(CASE WHEN @OverrideQAFlag = 0 THEN Actual_QAFlagID
																ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
														= APP_QAFlag.QAFlagID
										WHERE      TEMP_Selected.TransactionID = @TransactionID 
										GROUP BY APP_QAFlag.QAFlagID) AS ReportChecks
								ON QAFlag.QAFlagID = ReportChecks.QAFlagID
			ORDER BY QAFlag.ListIndex
		END
END                            
ELSE
	SELECT     QAFlag.QAFlagID, QAFlag.QAFlag, ISNULL(ReportChecks.NumChecks, 0) AS NumChecks
	FROM         (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag, ListIndex
						   FROM          APP_QAFlag) AS QAFlag LEFT OUTER JOIN
							  (SELECT     COUNT(QMS_SupplierResponse.ResponseID) AS NumChecks, QMS_SupplierResponse.QAFlagID
								FROM          TEMP_Selected INNER JOIN
													   QMS_SupplierResponse ON TEMP_Selected.SelectedID = QMS_SupplierResponse.ResponseID
								WHERE      TEMP_Selected.TransactionID = @TransactionID 
								GROUP BY QMS_SupplierResponse.QAFlagID) AS ReportChecks
						ON QAFlag.QAFlagID = ReportChecks.QAFlagID
	ORDER BY QAFlag.ListIndex
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_SubQuestions')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_SubQuestions] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 15/12/2009
-- Description:	Top problems for supplier profile
-- Version:		7
-- =============================================
-- JE TFS 4111: includes Amber checks if Amber set to be rejection
-- JE TFS 4770: 
-- jf tfs 8470: Make relevant changes to RPT_SupplierProfile_SubQuestions so that Question Levels are not returned when ShowQuestionLevelOnReports = 0
-- JF tfs 8706
-- JT TFS 13-38811: Enclose APP_QAFlag in table variable to get speed improvements.
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- JT TFS 13-42222: 02/03/2016 - Do not use UserID directly for TEMP tables
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_SubQuestions
	@UserID int,
	@Offline bit=0,
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	DECLARE @TokenUserID int;
	SELECT @TokenUserID = dbo.FNC_GetTempTransaction_TokenUserID(@TransactionID, @UserID)

	--exec RPT_SupplierProfile_SubQuestions 1,0
	DECLARE @ShowQuestionLevelOnReports BIT
	SET @ShowQuestionLevelOnReports = CAST(ISNULL(dbo.FNC_GetOptionValue(109),0) AS bit)

	IF @Offline = 0
	BEGIN
		SET NOCOUNT ON

		DECLARE @tmpRejectFlags TABLE (	QAFlagID int )
		INSERT @tmpRejectFlags (QAFlagID) SELECT QAFlagID FROM APP_QAFlag WHERE IsRejection = 1

		DECLARE @Question TABLE (ResponseID int, Question nvarchar(255), QAFlagID int)
		--Fill red questions to temp table
		--TFS 4770: include QAFlag too, so we can display Amber and Red Rejected
		INSERT INTO @Question (ResponseID, Question, QAFlagID)
		SELECT     QMS_Response.ResponseID, QMS_QuestionPool.Question, flags.QAFlagID
		FROM         QMS_AssessmentQuestion 
		INNER JOIN	QMS_QuestionPool ON QMS_AssessmentQuestion.QuestionID = QMS_QuestionPool.QuestionID 
		INNER JOIN	QMS_Response ON QMS_AssessmentQuestion.AssessmentID = QMS_Response.AssessmentID 
		INNER JOIN	QMS_ResponseQuestion ON QMS_Response.ResponseID = QMS_ResponseQuestion.ResponseID 
										AND 	QMS_AssessmentQuestion.AssessmentQuestionID = QMS_ResponseQuestion.AssessmentQuestionID 
		INNER JOIN TEMP_Selected ON TEMP_Selected.TransactionID = @TransactionID AND QMS_Response.ResponseID = TEMP_Selected.SelectedID 
		INNER JOIN	@tmpRejectFlags flags ON QMS_ResponseQuestion.Recommended_QAFlagID = flags.QAFlagID
	
		IF @ShowQuestionLevelOnReports = 1
			--Fill red question levels
			INSERT INTO @Question (ResponseID, Question, QAFlagID)
			SELECT     ResponseSummary.ResponseID, APP_QuestionLevel.QuestionLevel + ' Defects', APP_QAFlag.QAFlagID
			FROM         APP_AssessmentConstraintType INNER JOIN
								  APP_QuestionLevel ON APP_AssessmentConstraintType.QuestionLevelID = APP_QuestionLevel.QuestionLevelID INNER JOIN
								  QMS_AssessmentConstraint ON 
								  APP_AssessmentConstraintType.AssessmentConstraintTypeID = QMS_AssessmentConstraint.AssessmentConstraintTypeID INNER JOIN
									  (SELECT     QMS_ResponseQuestion.ResponseID, QMS_AssessmentQuestion.AssessmentID, QMS_AssessmentQuestion.QuestionLevelID, 
															   SUM(QMS_ResponseQuestion.Score) AS LevelScore
										FROM          QMS_ResponseQuestion INNER JOIN
															   QMS_Response ON QMS_ResponseQuestion.ResponseID = QMS_Response.ResponseID INNER JOIN
															   QMS_AssessmentQuestion ON QMS_ResponseQuestion.AssessmentQuestionID = QMS_AssessmentQuestion.AssessmentQuestionID INNER JOIN
															   TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID
										WHERE      (QMS_ResponseQuestion.RecordStatus < 201) AND TEMP_Selected.TransactionID = @TransactionID 
										GROUP BY QMS_ResponseQuestion.ResponseID, QMS_AssessmentQuestion.AssessmentID, QMS_AssessmentQuestion.QuestionLevelID
										HAVING      (SUM(QMS_ResponseQuestion.Score) IS NOT NULL)) AS ResponseSummary ON 
								  QMS_AssessmentConstraint.AssessmentID = ResponseSummary.AssessmentID AND 
								  APP_AssessmentConstraintType.QuestionLevelID = ResponseSummary.QuestionLevelID INNER JOIN
								  APP_QAFlag ON QMS_AssessmentConstraint.QAFlagID = APP_QAFlag.QAFlagID
			WHERE     (QMS_AssessmentConstraint.RecordStatus < 201) AND (QMS_AssessmentConstraint.Constraint_Score_Min IS NOT NULL) AND 
								  (QMS_AssessmentConstraint.Constraint_Score_Max IS NOT NULL) AND (ResponseSummary.LevelScore >= QMS_AssessmentConstraint.Constraint_Score_Min) AND 
								  (ResponseSummary.LevelScore <= QMS_AssessmentConstraint.Constraint_Score_Max) AND (APP_QAFlag.IsRejection = 1) OR
								  (QMS_AssessmentConstraint.RecordStatus < 201) AND (QMS_AssessmentConstraint.Constraint_Score_Min IS NULL) AND 
								  (QMS_AssessmentConstraint.Constraint_Score_Max IS NOT NULL) AND (ResponseSummary.LevelScore <= QMS_AssessmentConstraint.Constraint_Score_Max) AND 
								  (APP_QAFlag.IsRejection = 1) OR
								  (QMS_AssessmentConstraint.RecordStatus < 201) AND (QMS_AssessmentConstraint.Constraint_Score_Min IS NOT NULL) AND 
								  (QMS_AssessmentConstraint.Constraint_Score_Max IS NULL) AND (ResponseSummary.LevelScore >= QMS_AssessmentConstraint.Constraint_Score_Min) AND 
								  (APP_QAFlag.IsRejection = 1)

		SET NOCOUNT OFF

		SELECT     TOP (10) Response_Question.Question, PM_VW_StockCode.DisplayText AS StockCode, COUNT(QMS_Response.ResponseID) AS NumChecks, 
						  SUM(QMS_Response.QuantityAccepted) AS QuantityAccepted,
						  SUM(CASE WHEN Response_Question.QAFlagID = 2 THEN QMS_Response.QuantityRejected ELSE 0 END) AS AmberRejected,
						  SUM(CASE WHEN Response_Question.QAFlagID = 3 THEN QMS_Response.QuantityRejected ELSE 0 END) AS RedRejected,
						  SUM(TEMP_Selected.SelectedValue * QMS_Response.QuantityRejected) AS RejectedValue
		FROM         QMS_Response 
		INNER JOIN	dbo.FNC_ResponseHeader_StockCodeID_Selected(@TokenUserID) AS ResponseHeader_StockCodeID ON QMS_Response.ResponseID = ResponseHeader_StockCodeID.ResponseID 
		INNER JOIN	PM_VW_StockCode ON ResponseHeader_StockCodeID.StockCodeID = PM_VW_StockCode.StockCodeID 
		INNER JOIN	TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID 
		INNER JOIN	@Question AS Response_Question ON QMS_Response.ResponseID = Response_Question.ResponseID
		WHERE     TEMP_Selected.TransactionID = @TransactionID 
		GROUP BY PM_VW_StockCode.DisplayText, Response_Question.Question--, QuantityRejected
		ORDER BY COUNT(QMS_Response.ResponseID) DESC, sum(QuantityRejected) DESC, Question
	END
	ELSE
	SELECT     TOP (10) QMS_QuestionPool.Question, PM_VW_StockCode.DisplayText AS StockCode, COUNT(QMS_SupplierResponseQuestion.ResponseQuestionID) 
						  AS NumChecks, SUM(QMS_SupplierResponse.QuantityAccepted) AS QuantityAccepted,
						  SUM(CASE WHEN APP_QAFlag.QAFlagID = 2 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS AmberRejected,
						  SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS RedRejected,
						  SUM(TEMP_Selected.SelectedValue * QMS_SupplierResponse.QuantityRejected) AS RejectedValue
	FROM         QMS_SupplierResponseQuestion INNER JOIN
						  QMS_QuestionPool ON QMS_SupplierResponseQuestion.QuestionID = QMS_QuestionPool.QuestionID INNER JOIN
						  QMS_SupplierResponse ON QMS_SupplierResponseQuestion.ResponseID = QMS_SupplierResponse.ResponseID INNER JOIN
						  PM_VW_StockCode ON QMS_SupplierResponse.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
						  TEMP_Selected ON QMS_SupplierResponse.ResponseID = TEMP_Selected.SelectedID INNER JOIN
						  APP_QAFlag ON QMS_SupplierResponseQuestion.QAFlagID = APP_QAFlag.QAFlagID
	WHERE     TEMP_Selected.TransactionID = @TransactionID 
	AND (APP_QAFlag.IsRejection = 1)
	GROUP BY QMS_QuestionPool.Question, PM_VW_StockCode.DisplayText, QuantityRejected
	ORDER BY NumChecks DESC, QuantityRejected DESC, QMS_QuestionPool.Question
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_SubRejections')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_SubRejections] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 15/12/2009
-- Description:	Last 5 rejects for supplier profile
-- Version:		10
-- =============================================
-- JE TFS 3940: report will include QAS 2 responses if IsRejection flag set
-- JE TFS 4770: added override QA Flag config check
-- JE TFS 5906: updated to make use of warehouse table
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7507: All paths return same columns
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- JT TFS 13-42222: 02/03/2016 - Use TokenUserID for looking up stockcodes
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_SubRejections
	@UserID int,
	@Offline bit=0,
	@RejectionCount int=5,
	@TransactionID uniqueidentifier = NULL
AS
BEGIN
	DECLARE @TokenUserID int;
	SELECT @TokenUserID = dbo.FNC_GetTempTransaction_TokenUserID(@TransactionID, @UserID)

	--set Override QAFlag option
	DECLARE @OverrideQAFlag bit
	SET @OverrideQAFlag = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)

	--create temp table of QA Flag values to join on in main query
	IF nullif(object_id('tempdb..#tmpQAFlag'), 0) IS NOT NULL DROP TABLE #tmpQAFlag

	SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
	INTO #tmpQAFlag
	FROM         APP_QAFlag

	IF @Offline = 0
	BEGIN
		IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
		BEGIN
			DELETE FROM TEMP_Selected
			FROM         TEMP_Selected LEFT OUTER JOIN
							  (SELECT TOP (@RejectionCount) RPT_Warehouse_Response.ResponseID, RPT_Warehouse_Response.EndDate
								FROM	TEMP_Selected AS Selected INNER JOIN
										   RPT_Warehouse_Response ON Selected.SelectedID = RPT_Warehouse_Response.ResponseID INNER JOIN
										   APP_QAFlag ON RPT_Warehouse_Response.QAFlagID = APP_QAFlag.QAFlagID
								WHERE	(Selected.UserID = @UserID) AND (APP_QAFlag.IsRejection = 1) --check IsRejection flag
								ORDER BY RPT_Warehouse_Response.EndDate DESC) AS Rejections ON TEMP_Selected.SelectedID = Rejections.ResponseID
			WHERE     TEMP_Selected.TransactionID = @TransactionID AND (Rejections.ResponseID IS NULL)

			SELECT  Location.LocationName, 
					PM_VW_StockCode.DisplayText AS StockCode, 
					RPT_Warehouse_Response.EndDate, 
					RPT_Warehouse_Response.RedResponses, 
					RPT_Warehouse_Response.AmberResponses, 
					RPT_Warehouse_Response.Comments, 
					RPT_Warehouse_Response.QuantityRejected, 
					@RejectionCount AS RejectionCount,
					CASE WHEN QAFlag.QAFlagID = 2 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END AS AmberRejected,
					CASE WHEN QAFlag.QAFlagID = 3 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END AS RedRejected	
			FROM         PM_VW_Location AS Location RIGHT OUTER JOIN
								  RPT_Warehouse_Response INNER JOIN
								  PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
								  TEMP_Selected ON RPT_Warehouse_Response.ResponseID = TEMP_Selected.SelectedID ON 
								  Location.LocationID = RPT_Warehouse_Response.LocationID LEFT OUTER JOIN
								  #tmpQAFlag AS QAFlag ON RPT_Warehouse_Response.QAFlagID = QAFlag.QAFlagID
			WHERE     TEMP_Selected.TransactionID = @TransactionID 
			ORDER BY RPT_Warehouse_Response.EndDate DESC					
		END
		ELSE --don't use warehouse table
		BEGIN
			DELETE FROM TEMP_Selected --if so, leave QAS 2 and QAS 3 responses in TEMP_Selected
			FROM         TEMP_Selected LEFT OUTER JOIN
									  (SELECT     TOP (@RejectionCount) QMS_Response.ResponseID, QMS_Response.EndDate
										FROM          TEMP_Selected AS TEMP_Selected_1 INNER JOIN
														QMS_Response ON TEMP_Selected_1.SelectedID = QMS_Response.ResponseID  INNER JOIN
														APP_QAFlag ON --check config value to join on Actual or Override QAFlagID
															(CASE WHEN @OverrideQAFlag = 0 THEN Actual_QAFlagID
																ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
															= APP_QAFlag.QAFlagID
									WHERE      TEMP_Selected_1.TransactionID = @TransactionID 
									AND (APP_QAFlag.IsRejection = 1) --check IsRejection flag
										ORDER BY QMS_Response.EndDate DESC) AS Rejections ON TEMP_Selected.SelectedID = Rejections.ResponseID
			WHERE     TEMP_Selected.TransactionID = @TransactionID 
			AND (Rejections.ResponseID IS NULL)

			SELECT  Response_Location.LocationName, 
					PM_VW_StockCode.DisplayText AS StockCode, 
					QMS_Response.EndDate,
					
					CASE WHEN ResponseQuestionsRedAmber.RedResponses IS NULL THEN
						CASE WHEN ResponseQuestionsRedAmber.RedLevel IS NULL THEN '' ELSE ResponseQuestionsRedAmber.RedLevel + ' Defects' END
					ELSE ResponseQuestionsRedAmber.RedResponses + CASE WHEN ResponseQuestionsRedAmber.RedLevel IS NULL THEN ''
						ELSE ', ' + ResponseQuestionsRedAmber.RedLevel + ' Defects' END END AS RedResponses,
						
					CASE WHEN (SELECT IsRejection FROM APP_QAFlag WHERE QAFlagID = 2) = 1 THEN --check if QAS 2 is set to be a rejection
						CASE WHEN ResponseQuestionsRedAmber.AmberResponses IS NULL THEN
							CASE WHEN ResponseQuestionsRedAmber.AmberLevel IS NULL THEN '' ELSE ResponseQuestionsRedAmber.AmberLevel + ' Defects' END
						ELSE ResponseQuestionsRedAmber.AmberResponses + CASE WHEN ResponseQuestionsRedAmber.AmberLevel IS NULL THEN ''
							ELSE ', ' + ResponseQuestionsRedAmber.AmberLevel + ' Defects' END END
					ELSE '' END AS AmberResponses,	
					
					QMS_Response.Comments, 
					QMS_Response.QuantityRejected, 
					@RejectionCount AS RejectionCount,
					CASE WHEN QAFlag.QAFlagID = 2 THEN QMS_Response.QuantityRejected ELSE 0 END AS AmberRejected,
					CASE WHEN QAFlag.QAFlagID = 3 THEN QMS_Response.QuantityRejected ELSE 0 END AS RedRejected	
			FROM         PM_VW_StockCode INNER JOIN
								  dbo.FNC_ResponseHeader_StockCodeID_Selected(@TokenUserID) AS ResponseHeader_StockCodeID ON 
								  PM_VW_StockCode.StockCodeID = ResponseHeader_StockCodeID.StockCodeID INNER JOIN
								  QMS_Response ON ResponseHeader_StockCodeID.ResponseID = QMS_Response.ResponseID LEFT OUTER JOIN
								  dbo.FNC_ResponseQuestion_Selected(@TokenUserID, -2) AS ResponseQuestionsRedAmber ON QMS_Response.ResponseID = ResponseQuestionsRedAmber.ResponseID LEFT OUTER JOIN
									  (SELECT     ResponseHeader_Location.ResponseID, ResponseHeader_Location.Header_SourceID AS LocationID, PM_VW_Location.LocationName
										FROM          QMS_ResponseHeader AS ResponseHeader_Location INNER JOIN
															   QMS_AssessmentHeader AS AssessmentHeader_Location ON 
															   ResponseHeader_Location.AssessmentHeaderID = AssessmentHeader_Location.AssessmentHeaderID INNER JOIN
															   PM_VW_Location ON ResponseHeader_Location.Header_SourceID = PM_VW_Location.LocationID INNER JOIN
															   TEMP_Selected ON ResponseHeader_Location.ResponseID = TEMP_Selected.SelectedID
										WHERE      (AssessmentHeader_Location.DataTypeID = 9) AND (AssessmentHeader_Location.DataLookupID = 4) AND TEMP_Selected.TransactionID = @TransactionID ) 
								  AS Response_Location ON QMS_Response.ResponseID = Response_Location.ResponseID LEFT OUTER JOIN
								  #tmpQAFlag AS QAFlag ON --check config value to join on Actual or Override QAFlagID
										(CASE WHEN @OverrideQAFlag = 0 THEN Actual_QAFlagID
											ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
									= QAFlag.QAFlagID
			ORDER BY QMS_Response.EndDate DESC
		END
	END
	ELSE
	--Clear records from temp table apart from previous rejections
	BEGIN

		DELETE FROM TEMP_Selected
		FROM         TEMP_Selected LEFT OUTER JOIN
						  (SELECT     TOP (@RejectionCount) QMS_SupplierResponse.ResponseID, QMS_SupplierResponse.EndDate
							FROM          TEMP_Selected AS Selected INNER JOIN
												   QMS_SupplierResponse ON Selected.SelectedID = QMS_SupplierResponse.ResponseID INNER JOIN
													   APP_QAFlag ON QMS_SupplierResponse.QAFlagID = APP_QAFlag.QAFlagID
								WHERE      Selected.TransactionID = @TransactionID  AND (APP_QAFlag.IsRejection = 1) --check IsRejection flag
							ORDER BY QMS_SupplierResponse.EndDate DESC) AS Rejections ON TEMP_Selected.SelectedID = Rejections.ResponseID
		WHERE    TEMP_Selected.TransactionID = @TransactionID  AND (Rejections.ResponseID IS NULL)

		SELECT  Location.LocationName, 
				PM_VW_StockCode.DisplayText AS StockCode, 
				QMS_SupplierResponse.EndDate, 
				QMS_SupplierResponse.RedResponses, 
				'' as AmberResponses,
				QMS_SupplierResponse.Comments, 
				QMS_SupplierResponse.QuantityRejected, 
				@RejectionCount AS RejectionCount,
				CASE WHEN QAFlag.QAFlagID = 2 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END AS AmberRejected,
				CASE WHEN QAFlag.QAFlagID = 3 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END AS RedRejected	
		FROM         PM_VW_Location AS Location RIGHT OUTER JOIN
							  QMS_SupplierResponse INNER JOIN
							  PM_VW_StockCode ON QMS_SupplierResponse.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
							  TEMP_Selected ON QMS_SupplierResponse.ResponseID = TEMP_Selected.SelectedID ON 
							  Location.LocationID = QMS_SupplierResponse.LocationID LEFT OUTER JOIN
							  #tmpQAFlag AS QAFlag ON QMS_SupplierResponse.QAFlagID = QAFlag.QAFlagID
		WHERE    TEMP_Selected.TransactionID = @TransactionID 
		ORDER BY QMS_SupplierResponse.EndDate DESC
	END
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile_SubStockCode')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile_SubStockCode] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 15/12/2009
-- Description:	Gets product data for supplier profile report
-- Version: 9
-- =============================================
-- JE TFS 4770: added override QA Flag config check and colour cols
-- JE TFS 5906: updated to make use of warehouse table
-- JE TFS 6187: added NULL values for Amber and Red Rejected cols so report picks up correcet values when using warehouse
-- MP TFS 6659: fixed issue with records being duplicated for each QA flag
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7507: All paths return same columns
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- JT TFS 13-42222: 02/03/2016 - Use TokenUserID for looking up stockcodes
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile_SubStockCode
	@UserID int,
	@Offline bit=0,
	@TransactionID uniqueidentifier = NULL
AS
BEGIN

	DECLARE @TokenUserID int;
	SELECT @TokenUserID = dbo.FNC_GetTempTransaction_TokenUserID(@TransactionID, @UserID)

	--set Override QAFlag option
	DECLARE @OverrideQAFlag bit
	SET @OverrideQAFlag = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)

	IF @Offline = 0
	BEGIN
		IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
		BEGIN
			SELECT	PM_VW_StockCode.DisplayText AS StockCode, 
					COUNT(RPT_Warehouse_Response.ResponseID) AS NumChecks, 
					SUM(RPT_Warehouse_Response.QuantityReceived) AS QuantityReceived, 
					SUM(RPT_Warehouse_Response.QuantityChecked) AS QuantityChecked,
					SUM(RPT_Warehouse_Response.QuantityAccepted) AS QuantityAccepted, 
					SUM(CASE WHEN QAFlagID = 2 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END) AS AmberRejected,
					SUM(CASE WHEN QAFlagID = 3 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END) AS RedRejected,
					SUM(RPT_Warehouse_Response.QuantityRejected) AS QuantityRejected, 
					SUM(CASE WHEN QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen, 
					SUM(CASE WHEN QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber, 
					SUM(CASE WHEN QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed, 
					SUM(RPT_Warehouse_Response.QuantityRejected * TEMP_Selected.SelectedValue) AS RejectedValue,
					--return QAS Flag colours as separate cols so these can be used to set text colours in report
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour
			FROM	RPT_Warehouse_Response INNER JOIN
					  PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
					  TEMP_Selected ON RPT_Warehouse_Response.ResponseID = TEMP_Selected.SelectedID
			WHERE	TEMP_Selected.TransactionID = @TransactionID 
			GROUP BY PM_VW_StockCode.DisplayText
			ORDER BY NumRed DESC, QuantityRejected DESC, NumAmber DESC, NumGreen DESC		
		END
		ELSE
		BEGIN
			SELECT	PM_VW_StockCode.DisplayText AS StockCode, 
					COUNT(QMS_Response.ResponseID) AS NumChecks, 
					SUM(QMS_Response.QuantityReceived) AS QuantityReceived, 
					SUM(QMS_Response.QuantityChecked) AS QuantityChecked, 
					SUM(QMS_Response.QuantityAccepted) AS QuantityAccepted,
					SUM(CASE WHEN APP_QAFlag.QAFlagID = 2 THEN QMS_Response.QuantityRejected ELSE 0 END) AS AmberRejected,
					SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN QMS_Response.QuantityRejected ELSE 0 END) AS RedRejected,
					SUM(QMS_Response.QuantityRejected) as QuantityRejected,
					SUM(CASE WHEN APP_QAFlag.QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen,
					SUM(CASE WHEN APP_QAFlag.QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber,
					SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed,
					SUM(QMS_Response.QuantityRejected * TEMP_Selected.SelectedValue) AS RejectedValue,
					--return QAS Flag colours as separate cols so these can be used to set text colours in report
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour
			FROM	QMS_Response INNER JOIN
					  dbo.FNC_ResponseHeader_StockCodeID_Selected(@TokenUserID) AS ResponseHeader_StockCodeID ON 
					  QMS_Response.ResponseID = ResponseHeader_StockCodeID.ResponseID INNER JOIN
					  PM_VW_StockCode ON ResponseHeader_StockCodeID.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
					  TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID INNER JOIN
						APP_QAFlag ON --check config value to join on Actual or Override QAFlagID
						 (CASE WHEN @OverrideQAFlag = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
							= APP_QAFlag.QAFlagID
			WHERE	TEMP_Selected.TransactionID = @TransactionID 
			GROUP BY PM_VW_StockCode.DisplayText
			ORDER BY NumRed DESC, RedRejected DESC, AmberRejected DESC, NumAmber DESC, NumGreen DESC
		END
	END
	ELSE
	BEGIN
		SELECT	PM_VW_StockCode.DisplayText AS StockCode,
				COUNT(QMS_SupplierResponse.ResponseID) AS NumChecks, 
				SUM(QMS_SupplierResponse.QuantityReceived) AS QuantityReceived, 
				SUM(QMS_SupplierResponse.QuantityChecked) AS QuantityChecked, 
				SUM(QMS_SupplierResponse.QuantityAccepted) AS QuantityAccepted, 
				SUM(CASE WHEN QAFlagID = 2 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS AmberRejected,
				SUM(CASE WHEN QAFlagID = 3 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS RedRejected,
				SUM(QMS_SupplierResponse.QuantityRejected) AS QuantityRejected, 
				SUM(CASE WHEN QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen, 
				SUM(CASE WHEN QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber, 
				SUM(CASE WHEN QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed, 
				SUM(QMS_SupplierResponse.QuantityRejected * TEMP_Selected.SelectedValue) AS RejectedValue,
				--return QAS Flag colours as separate cols so these can be used to set text colours in report
				(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
				(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
				(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour
		FROM	QMS_SupplierResponse INNER JOIN
				  PM_VW_StockCode ON QMS_SupplierResponse.StockCodeID = PM_VW_StockCode.StockCodeID INNER JOIN
				  TEMP_Selected ON QMS_SupplierResponse.ResponseID = TEMP_Selected.SelectedID
		WHERE	TEMP_Selected.TransactionID = @TransactionID 
		GROUP BY PM_VW_StockCode.DisplayText
		ORDER BY NumRed DESC, QuantityRejected DESC, NumAmber DESC, NumGreen DESC
	END
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'RPT_SupplierProfile')
	EXEC ('CREATE PROCEDURE [dbo].[RPT_SupplierProfile] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 14/12/2009
-- Description:	Gets summary data for supplier profile report
-- Version:		12
-- =============================================
-- JE TFS 3940 - added QA Flag colour columns used for colouring text in report
-- JT TFS 4708 - Added Assessment Type names list to SP output. (v4 from passed in list)
-- JE TFS 4770 - added QA Flag config check and Amber Rejected col
-- JE TFS 5061 - removed old filter params and amended to use two tier filtering
-- JE TFS 5906: updated to make use of warehouse table
-- JT TFS 6743: Only use warehouse if RPT_Warehouse_Response contains rows.
-- JT TFS 6689: FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7507: All paths return same columns
-- JT TFS 13-40008: 22/02/2016 - Added transactional processing
-- JT TFS 13-42222: 02/03/2016 - Do not use UserID directly for TEMP tables
-- =============================================
ALTER PROCEDURE RPT_SupplierProfile
	@UserID int,
	@SupplierID int,
	@DateFrom datetime,
	@DateTo datetime,
	@Offline bit=0,
	@AssessmentTypeID_LIST nvarchar(1000) = NULL,
	@TransactionID uniqueidentifier
AS

--get list of Assessment Types from IDs passed in
DECLARE @AssessmentTypes nvarchar(1000)
SET @AssessmentTypes = dbo.FNC_AssessmentNamesFromIDs(@AssessmentTypeID_LIST, @UserID)

IF LEN(@AssessmentTypes) >= 70
BEGIN
	--cut off end of Assessment Types list if too big to fit on report header
	SET @AssessmentTypes = SUBSTRING(@AssessmentTypes, 1, 67) + '...'
END

BEGIN
	SET NOCOUNT ON
	
	DECLARE @TotalReceived int, @TotalPrice money
	SELECT     @TotalReceived = SUM(SourceObjectID), @TotalPrice = SUM(CAST(Header_Value AS money))
	FROM      TEMP_ResponseHeader
	WHERE     TransactionID = @TransactionID 

	SET NOCOUNT OFF

	IF @Offline = 0
	BEGIN
		IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 --use warehouse table
			BEGIN
				SELECT	@DateFrom AS DateFrom,
						@DateTo AS DateTo,
						PM_VW_Supplier.SupplierID,
						PM_VW_Supplier.SupplierCode,
						PM_VW_Supplier.SupplierName,
						
						COUNT(RPT_Warehouse_Response.ResponseID) AS NumChecks,
						SUM(RPT_Warehouse_Response.QuantityReceived) AS QuantityReceived,
						SUM(TEMP_Selected.SelectedValue * RPT_Warehouse_Response.QuantityReceived) AS ReceivedValue,
						SUM(RPT_Warehouse_Response.QuantityChecked) AS QuantityChecked,
						SUM(RPT_Warehouse_Response.QuantityAccepted) AS QuantityAccepted,
						
						SUM(CASE WHEN QAFlagID = 2 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END) AS AmberRejected,
						SUM(CASE WHEN QAFlagID = 3 THEN RPT_Warehouse_Response.QuantityRejected ELSE 0 END) AS RedRejected,
						SUM(RPT_Warehouse_Response.QuantityRejected) AS QuantityRejected,
						SUM(CASE WHEN QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen,
						SUM(CASE WHEN QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber,
						
						SUM(CASE WHEN QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed,
						SUM(CASE WHEN QAFlagID = 3 THEN QuantityReceived ELSE 0 END) AS QuantityReceivedRed,
						@TotalReceived AS ConsignmentReceived,
						@TotalPrice AS ConsignmentReceivedValue,
						SUM(TEMP_Selected.SelectedValue * RPT_Warehouse_Response.QuantityRejected) AS RejectedValue,
						
						SUM(CASE WHEN QAFlagID = 3 THEN TEMP_Selected.SelectedValue * RPT_Warehouse_Response.QuantityReceived ELSE 0 END) AS RedValue,
						--return QAS Flag colours as separate cols so these can be used to set text colours in report
						(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
						(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
						(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour,
						@AssessmentTypes AS AssessmentTypes
				FROM	TEMP_Selected INNER JOIN
						  RPT_Warehouse_Response ON TEMP_Selected.SelectedID = RPT_Warehouse_Response.ResponseID INNER JOIN
						  PM_VW_Supplier ON RPT_Warehouse_Response.SupplierID = PM_VW_Supplier.SupplierID
				WHERE		TEMP_Selected.TransactionID = @TransactionID 
				GROUP BY	PM_VW_Supplier.SupplierCode, PM_VW_Supplier.SupplierName, PM_VW_Supplier.SupplierID
				HAVING		(PM_VW_Supplier.SupplierID = @SupplierID)			
			END
		ELSE
			BEGIN	
				SELECT		@DateFrom AS DateFrom, 
							@DateTo AS DateTo, 
							PM_VW_Supplier.SupplierID, 
							PM_VW_Supplier.SupplierCode, 
							PM_VW_Supplier.SupplierName, 
							
							COUNT(QMS_Response.ResponseID) AS NumChecks, 
							SUM(QMS_Response.QuantityReceived) AS QuantityReceived, 
							SUM(TEMP_Selected.SelectedValue * QMS_Response.QuantityReceived) AS ReceivedValue, 
							SUM(QMS_Response.QuantityChecked) AS QuantityChecked, 
							SUM(QMS_Response.QuantityAccepted) AS QuantityAccepted,
							
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 2 THEN QMS_Response.QuantityRejected ELSE 0 END) AS AmberRejected,
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN QMS_Response.QuantityRejected ELSE 0 END) AS RedRejected,
							SUM(QMS_Response.QuantityRejected) AS QuantityRejected,
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen,
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber, 

							SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed,
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN QuantityReceived ELSE 0 END) AS QuantityReceivedRed,
							@TotalReceived AS ConsignmentReceived, 
							@TotalPrice AS ConsignmentReceivedValue,
							SUM(TEMP_Selected.SelectedValue * QMS_Response.QuantityRejected) AS RejectedValue,
							
							SUM(CASE WHEN APP_QAFlag.QAFlagID = 3 THEN TEMP_Selected.SelectedValue * QMS_Response.QuantityReceived ELSE 0 END) AS RedValue,
							--return QAS Flag colours as separate cols so these can be used to set text colours in report
							(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
							(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
							(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour,
							@AssessmentTypes AS AssessmentTypes
				FROM         PM_VW_Supplier CROSS JOIN
									  TEMP_Selected INNER JOIN QMS_Response ON TEMP_Selected.SelectedID = QMS_Response.ResponseID INNER JOIN
							  APP_QAFlag ON --check config value to join on Actual or Override QAFlagID
							  (CASE WHEN CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit) = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
									 = APP_QAFlag.QAFlagID
				WHERE    TEMP_Selected.TransactionID = @TransactionID 
				GROUP BY PM_VW_Supplier.SupplierCode, PM_VW_Supplier.SupplierName, PM_VW_Supplier.SupplierID
				HAVING      (PM_VW_Supplier.SupplierID = @SupplierID)
			END	
	END
	ELSE
	BEGIN
		SELECT		@DateFrom AS DateFrom,
					@DateTo AS DateTo,
					PM_VW_Supplier.SupplierID,
					PM_VW_Supplier.SupplierCode,
					PM_VW_Supplier.SupplierName,
					
					COUNT(QMS_SupplierResponse.ResponseID) AS NumChecks,
					SUM(QMS_SupplierResponse.QuantityReceived) AS QuantityReceived,
					SUM(TEMP_Selected.SelectedValue * QMS_SupplierResponse.QuantityReceived) AS ReceivedValue,
					SUM(QMS_SupplierResponse.QuantityChecked) AS QuantityChecked,
					SUM(QMS_SupplierResponse.QuantityAccepted) AS QuantityAccepted,
					
					SUM(CASE WHEN QAFlagID = 2 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS AmberRejected,
					SUM(CASE WHEN QAFlagID = 3 THEN QMS_SupplierResponse.QuantityRejected ELSE 0 END) AS RedRejected,
					SUM(QMS_SupplierResponse.QuantityRejected) AS QuantityRejected,
					SUM(CASE WHEN QAFlagID = 1 THEN 1 ELSE 0 END) AS NumGreen,
					SUM(CASE WHEN QAFlagID = 2 THEN 1 ELSE 0 END) AS NumAmber,
					
					SUM(CASE WHEN QAFlagID = 3 THEN 1 ELSE 0 END) AS NumRed,
					SUM(CASE WHEN QAFlagID = 3 THEN QuantityReceived ELSE 0 END) AS QuantityReceivedRed,
					@TotalReceived AS ConsignmentReceived,
					@TotalPrice AS ConsignmentReceivedValue,
					SUM(TEMP_Selected.SelectedValue * QMS_SupplierResponse.QuantityRejected) AS RejectedValue,
					
					SUM(CASE WHEN QAFlagID = 3 THEN TEMP_Selected.SelectedValue * QMS_SupplierResponse.QuantityReceived ELSE 0 END) AS RedValue,
					--return QAS Flag colours as separate cols so these can be used to set text colours in report
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 1) AS GreenColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 2) AS AmberColour,
					(SELECT QAFlagBackColour FROM APP_QAFlag WHERE QAFlagID = 3) AS RedColour,
					'All' AS AssessmentTypes
		FROM         TEMP_Selected INNER JOIN
							  QMS_SupplierResponse ON TEMP_Selected.SelectedID = QMS_SupplierResponse.ResponseID INNER JOIN
							  PM_VW_Supplier ON QMS_SupplierResponse.SupplierID = PM_VW_Supplier.SupplierID
		WHERE     TEMP_Selected.TransactionID = @TransactionID 
		GROUP BY PM_VW_Supplier.SupplierCode, PM_VW_Supplier.SupplierName, PM_VW_Supplier.SupplierID
		HAVING      (PM_VW_Supplier.SupplierID = @SupplierID)
	END
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'SAVE_TEMP_Transaction')
	EXEC ('CREATE PROCEDURE [dbo].[SAVE_TEMP_Transaction] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Jonathan Toolan
-- Create date: 02/03/2016
-- Description:	Stores a transactionID in the database and generates a new TokenUserID
-- Version: 	1
-- =============================================
-- JT TFS 13-42222: 02/03/2016 - Created
-- =============================================
ALTER PROCEDURE SAVE_TEMP_Transaction
	@TransactionID uniqueidentifier,
	@UserID int
AS
BEGIN
	IF @TransactionID IS NULL OR @UserID IS NULL
		RETURN;

	-- Don't re-save this information if it already exists.
	IF EXISTS (SELECT * FROM TEMP_Transaction WHERE TransactionID = @TransactionID)
		RETURN;

	DECLARE @TokenUserID int;
	SELECT @TokenUserID = (RAND() * 1000000000) + 100000;
	
	WHILE EXISTS (SELECT * FROM TEMP_Transaction WHERE TokenUserID = @TokenUserID)
	   OR EXISTS (SELECT * FROM QMS_User WHERE UserID = @TokenUserID)
		SELECT @TokenUserID = (RAND() * 1000000000) + 100000;

	INSERT TEMP_Transaction (TransactionID, TokenUserID, ActualUserID)
	VALUES (@TransactionID, @TokenUserID, @UserID);

END
GO

-- BEGIN DATA TEMP_Transaction
BEGIN TRY
IF NOT EXISTS (SELECT * FROM sys.tables WHERE name = N'TEMP_Transaction')
BEGIN 
	CREATE TABLE TEMP_Transaction (
		TransactionID uniqueidentifier NOT NULL PRIMARY KEY,
		TokenUserID int NOT NULL,
		ActualUserID int NOT NULL,
		StartTime datetime NOT NULL DEFAULT (getdate())
	)
END

IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'TEMP_Transaction') and name='IX_TEMP_Transaction_TokenUserID')
	CREATE INDEX IX_TEMP_Transaction_TokenUserID ON TEMP_Transaction(TokenUserID)
END TRY 
BEGIN CATCH

	INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName, ErrorMessage)
	SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'DATA TEMP_Transaction', ERROR_MESSAGE()
	FROM #tmpScriptName
	
END CATCH
GO
-- END DATA TEMP_Transaction


BEGIN 

	IF EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'[dbo].[QMS_Response]') and name='IX_QMS_Response_ParentResponseGUID')
		DROP INDEX QMS_Response.IX_QMS_Response_ParentResponseGUID

	IF NOT EXISTS (SELECT * FROM dbo.sysindexes WHERE id=OBJECT_ID(N'[dbo].[QMS_Response]') and name='IX_QMS_Response_ParentResponseGUID')
	BEGIN 
	CREATE NONCLUSTERED INDEX [IX_QMS_Response_ParentResponseGUID] ON [dbo].[QMS_Response]
	(
		[ParentResponseGUID] ASC,
		[RecordStatus] ASC
	)WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, SORT_IN_TEMPDB = OFF, DROP_EXISTING = OFF, ONLINE = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON)

	DBCC DBREINDEX ('QMS_Response')

	
	END
END 
GO




PRINT 'Database updated successfully!'
GO
