-- =============================================
-- Author:		Jeremy Formby
-- Create date: 10/07/2015
-- Version:		1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO



PRINT 'Sorting tables...'

-- BEGIN DATA MOBILE_PM_Consignment

--this is to update mobile database as they are different from office
IF NOT EXISTS (SELECT * FROM sys.tables t INNER JOIN sys.columns c ON t.object_id = c.object_id WHERE t.name = 'PM_Consignment' AND c.name = 'Text3' )

	alter table PM_Consignment add
		Text3 nvarchar(100),
		Text4 nvarchar(100),
		CountryCode  nvarchar(100)

		  
GO
-- END DATA MOBILE_PM_Consignment

PRINT 'Sorting indexes...'

PRINT 'Sorting Views...'

PRINT 'Sorting User Defined Functions...'

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_User_ResponseList')
	EXEC('CREATE FUNCTION [dbo].[FNC_User_ResponseList]() RETURNS @x TABLE (y bit) AS BEGIN RETURN END')
GO
-- =============================================
-- Author:		Markp
-- Create date: 17/08/2009
-- Description:	Returns list of responses that user can access
-- Version: 	13
-- =============================================
-- JT TFS 4194 - 14/08/2012 - When stock code is restricted, this now returns checks where no stock code is specified in the header.
-- JE TFS 4622 - returns QAFlagID based on ReportQAFlag config setting
-- JT TFS 5801 - performance improvements
-- JE TFS 5905 - updated to make use of Reporting Warehouse table
-- MP TFS 6031 - changed non-admin filter to ensure that temp checks are not included
-- MP TFS 6152 - fix to handle supplier site permissions
-- JT TFS 6689 - FNC_CanUserDataWareHouse takes UserID parameter.
-- JT TFS 7612 - Void checks visibility is controlled by UserSetting for specific reports.
-- JE TFS 13-3570: added PRIMARY KEY to @Table_StockCode to improve performance
-- MP TFS 13-12783: fixed issue with ambiguous Location permission linking
-- jf tfs 16880 - UBE Check view performance for user with 2000+ stock code permissions - added lots of filters for updates like ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)
-- MP TFS 23409 - Added conditions in the CanUseWarehouse block so that Suppliers (userlevel=9) use supplier site code and non supplier (userlevel<>9) use locations
-- =============================================
ALTER FUNCTION [dbo].[FNC_User_ResponseList]
(
	@UserID int,
	@DateFrom datetime,
	@DateTo datetime
)
RETURNS 
@Response TABLE 
(
	ResponseID int,
	ResponseStatus tinyint,
	QAFlagID int
)
AS
BEGIN

--declare @UserID int, @DateFrom datetime,	@DateTo datetime
--select @UserID = 836, @DateFrom = '1 Jun 2013 00:00:00',  @DateTo = '21 May 2014 23:59:59'

	DECLARE @UnfilteredResponse TABLE (
		ResponseID int PRIMARY KEY,
		ResponseStatus tinyint,
		LocationID int NULL,
		CustomerID int NULL,
		SupplierID int NULL,
		StockCodeID int NULL,
		QAFlagID int NULL
	)

	--get date reporting warehouse table last built
	DECLARE @WarehouseBuildDate datetime
	SET @WarehouseBuildDate = (SELECT Updated FROM APP_Config WHERE OptionID = 78)

	DECLARE @UserSettingDisplayVoids bit 
	SET @UserSettingDisplayVoids = ISNULL(cast(dbo.FNC_GetUserSetting(@UserID, 'DisplayVoidChecks') as bit), 0)

	-- Lookup user level
	DECLARE @UserLevel int, @RecordStatus tinyint
	SELECT     @UserLevel = ISNULL(UserLevel,0)
	FROM         QMS_User
	WHERE     (UserID = @UserID)
	IF @UserSettingDisplayVoids = 1 -- Include all recordstatuses when we set a report to do the same by setting the User Setting.
		SET @RecordStatus = 255 -- maximum possible. 
	ELSE IF @UserLevel = 1  --Admin can always see hidden/void checks.
		SET @RecordStatus = 201
	ELSE
		SET @RecordStatus = 100

	-- Permissions that this user has.
	DECLARE @PermissionLink TABLE (
		UserID int, SupplierID int, CustomerID int, LocationID int, StockCodeID int, 
		StockCodeSubGroup nvarchar(100), BusinessGroup nvarchar(100)
	)
	INSERT INTO	@PermissionLink (UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup)
	SELECT	UserID, SupplierID, CustomerID, LocationID, StockCodeID, StockCodeSubGroup, BusinessGroup
	FROM	VW_PermissionLink PermissionLink
	WHERE	PermissionLink.PermissionAllow = 1 AND PermissionLink.PermissionType IN (101,102,103,104,105,106) AND UserID = @UserID

	DECLARE @Table_StockCode TABLE (
		StockCodeID int PRIMARY KEY
	)
	
	--check for permissions in three levels of hierarchy: Business Group, Sub Group, Stock Code, and translate to StockCodeID
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		BusinessGroup IN (SELECT BusinessGroup FROM @PermissionLink WHERE BusinessGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeSubGroup IN (SELECT StockCodeSubGroup FROM @PermissionLink WHERE StockCodeSubGroup IS NOT NULL)
	
	INSERT INTO @Table_StockCode (StockCodeID)
	SELECT		StockCodeID
	FROM        PM_VW_StockCode
	WHERE		StockCodeID IN (SELECT StockCodeID FROM @PermissionLink WHERE StockCodeID IS NOT NULL)

	BEGIN
	IF dbo.FNC_CanUseDataWarehouse(@UserID) = 1 AND @DateTo <= @WarehouseBuildDate --use warehouse table if built within specified date range
		BEGIN
			INSERT INTO	@UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
			SELECT		RPT_Warehouse_Response.ResponseID, QMS_Response.ResponseStatus, RPT_Warehouse_Response.QAFlagID --Actual or Override QAFlag setting checked when building warehouse table
			FROM		RPT_Warehouse_Response INNER JOIN
							QMS_Response ON RPT_Warehouse_Response.ResponseID = QMS_Response.ResponseID
			WHERE		(QMS_Response.RecordStatus < @RecordStatus) AND (RPT_Warehouse_Response.EndDate BETWEEN @DateFrom AND @DateTo)
				
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
				UPDATE	UnfilteredResponse
				SET	LocationID = RPT_Warehouse_Response.SupplierSiteID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
				WHERE ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(RPT_Warehouse_Response.SupplierSiteID,0)
 
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
				UPDATE	UnfilteredResponse
				SET	LocationID = RPT_Warehouse_Response.LocationID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
				WHERE ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(RPT_Warehouse_Response.LocationID,0)

			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
				UPDATE	UnfilteredResponse
				SET		SupplierID = RPT_Warehouse_Response.SupplierID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
				WHERE ISNULL(UnfilteredResponse.SupplierID,0) <> ISNULL(RPT_Warehouse_Response.SupplierID,0)
				
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
				UPDATE	UnfilteredResponse
				SET		CustomerID = PM_VW_StockCode.CustomerID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID INNER JOIN
							PM_VW_StockCode ON RPT_Warehouse_Response.StockCodeID = PM_VW_StockCode.StockCodeID
				WHERE ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

			IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
				UPDATE	UnfilteredResponse
				SET		StockCodeID = RPT_Warehouse_Response.StockCodeID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							RPT_Warehouse_Response ON UnfilteredResponse.ResponseID = RPT_Warehouse_Response.ResponseID
		END
	ELSE --don't use warehouse table
		BEGIN
			DECLARE @override bit 
			SET @override = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)
			
			INSERT INTO @UnfilteredResponse (ResponseID, ResponseStatus, QAFlagID)
			SELECT ResponseID, ResponseStatus,
					--check config value to use Actual or Override QAFlagID
					CASE WHEN @override = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END AS QAFlagID
			FROM	QMS_Response
			WHERE	(RecordStatus < @RecordStatus) AND (EndDate BETWEEN @DateFrom AND @DateTo)
			
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel = 9)
				UPDATE	UnfilteredResponse
				SET		LocationID = Location.LocationID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
							PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
				WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 53) and
				ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(Location.LocationID,0)

			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE LocationID IS NOT NULL) AND (@UserLevel <> 9)
				UPDATE	UnfilteredResponse
				SET		LocationID = Location.LocationID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
							PM_VW_Location AS Location ON QMS_ResponseHeader.Header_SourceID = Location.LocationID
				WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 4) and 
				ISNULL(UnfilteredResponse.LocationID,0) <> ISNULL(Location.LocationID,0)

			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
				UPDATE	UnfilteredResponse
				SET		SupplierID = QMS_ResponseHeader.Header_SourceID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							QMS_ResponseHeader ON UnfilteredResponse.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
							QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
				WHERE   (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 1) AND
				ISNULL(UnfilteredResponse.SupplierID,0) <> ISNULL(QMS_ResponseHeader.Header_SourceID,0)
				
			IF EXISTS (SELECT UserID FROM @PermissionLink WHERE CustomerID IS NOT NULL) 
				UPDATE	UnfilteredResponse
				SET		CustomerID = PM_VW_StockCode.CustomerID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID INNER JOIN
							PM_VW_StockCode ON PM_VW_StockCode.StockCodeID = HeaderStockCode.StockCodeID
				WHERE ISNULL(UnfilteredResponse.CustomerID,0) <> ISNULL(PM_VW_StockCode.CustomerID,0)

			IF EXISTS (SELECT StockCodeID FROM @Table_StockCode)
				UPDATE	UnfilteredResponse
				SET		StockCodeID = HeaderStockCode.StockCodeID
				FROM	@UnfilteredResponse UnfilteredResponse INNER JOIN
							dbo.FNC_ResponseHeader_StockCodeID() HeaderStockCode ON UnfilteredResponse.ResponseID = HeaderStockCode.ResponseID
				where ISNULL(UnfilteredResponse.StockCodeID,0) <> ISNULL(HeaderStockCode.StockCodeID,0)
		END
	END

	-- Only return responses which match the filter specified, or for which we do not have a filter.
	-- This means that admin users (any user with no permissions set) can see all responses.
	INSERT INTO @Response (ResponseID, ResponseStatus, QAFlagID)
	SELECT	ResponseID, ResponseStatus, QAFlagID
	FROM	@UnfilteredResponse UnfilteredResponse
	WHERE (	NOT EXISTS (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL)
		OR	LocationID IN (SELECT LocationID FROM @PermissionLink WHERE LocationID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL)
		OR	SupplierID IN (SELECT SupplierID FROM @PermissionLink WHERE SupplierID IS NOT NULL) )
	AND (	NOT EXISTS (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL)
		OR	CustomerID IN (SELECT CustomerID FROM @PermissionLink WHERE CustomerID IS NOT NULL) )
	AND (	StockCodeID IS NULL -- Allow unlinked stockcode responses
		OR	NOT EXISTS (SELECT StockCodeID FROM @Table_StockCode)
		OR	StockCodeID IN (SELECT StockCodeID FROM @Table_StockCode) )
	
	RETURN
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_GetActiveAssessmentID')
	EXEC('CREATE FUNCTION [dbo].[FNC_GetActiveAssessmentID]() RETURNS bit AS BEGIN RETURN 0 END')
GO
-- =============================================
-- Author:		Markp
-- Create date: 06/04/2010
-- Description:	Gets active assessment for selected stock code
-- Version:		2
-- =============================================
-- JT TFS 13-22155: Only get master assessments when looking for stockcode without supplier.
-- =============================================
ALTER FUNCTION dbo.FNC_GetActiveAssessmentID 
(
	@StockCodeID int
)
RETURNS int
AS
BEGIN
	DECLARE @AssessmentID int

	SELECT     @AssessmentID = MIN(QMS_Assessment.AssessmentID)
	FROM         QMS_Assessment 
	INNER JOIN	QMS_AssessmentTarget ON QMS_Assessment.AssessmentID = QMS_AssessmentTarget.AssessmentID 
	INNER JOIN	APP_AssessmentStatus ON QMS_Assessment.AssessmentStatusID = APP_AssessmentStatus.AssessmentStatusID
	WHERE   (APP_AssessmentStatus.Active = 1) 
	AND		(QMS_Assessment.RecordStatus < 201) 
	AND		(QMS_Assessment.AssessmentTargetTypeID = 1) 
	AND		(QMS_AssessmentTarget.SourceObjectID = @StockCodeID) 
	AND		(QMS_AssessmentTarget.RecordStatus < 201)
	AND		(QMS_Assessment.TempSpec = 0)

	IF @AssessmentID IS NULL --No active assessment for selected stock code
		SELECT     @AssessmentID = MIN(QMS_Assessment.AssessmentID)
		FROM         QMS_Assessment 
		INNER JOIN	APP_AssessmentStatus ON QMS_Assessment.AssessmentStatusID = APP_AssessmentStatus.AssessmentStatusID 
		LEFT OUTER JOIN	(	SELECT     QMS_Assessment_1.AssessmentID, COUNT(QMS_AssessmentTarget_1.SourceObjectID) AS NumTargets
							FROM        QMS_Assessment AS QMS_Assessment_1 
							INNER JOIN	QMS_AssessmentTarget AS QMS_AssessmentTarget_1 ON QMS_Assessment_1.AssessmentID = QMS_AssessmentTarget_1.AssessmentID
							WHERE   (QMS_Assessment_1.RecordStatus < 201) 
							AND		(QMS_Assessment_1.AssessmentTargetTypeID = 1) 
							AND		(QMS_AssessmentTarget_1.RecordStatus < 201)
							AND		(QMS_Assessment_1.TempSpec = 0)
							GROUP BY QMS_Assessment_1.AssessmentID
							HAVING      (COUNT(QMS_AssessmentTarget_1.SourceObjectID) IS NOT NULL)
			) AS AssessmentTargets ON QMS_Assessment.AssessmentID = AssessmentTargets.AssessmentID
		WHERE   (APP_AssessmentStatus.Active = 1) 
		AND		(QMS_Assessment.RecordStatus < 201) 
		AND		(QMS_Assessment.AssessmentTargetTypeID = 1) 
		AND		(AssessmentTargets.NumTargets IS NULL)
		AND		(QMS_Assessment.TempSpec = 0)
                       
	RETURN @AssessmentID

END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_Assessment_IsSignedOffByUserLevel')
	EXEC('CREATE FUNCTION [dbo].[FNC_Assessment_IsSignedOffByUserLevel]() RETURNS bit AS BEGIN RETURN 0 END')
GO
-- =============================================
-- Author:		Jonathan Toolan
-- Create date: 12/01/2015
-- Description:	Has the assessment been signed off by that UserLevel since the last rejection (or ever if no rejections)?
-- Version:		2
-- =============================================
-- JT TFS 13-11815: Created
-- JT TFS 13-22563: Unarchiving an assessment also resets the sign-off process.
-- =============================================
ALTER FUNCTION dbo.FNC_Assessment_IsSignedOffByUserLevel(@AssessmentID int, @UserLevelID int)
RETURNS bit
AS
BEGIN
	DECLARE @tmpRecordLock TABLE (
		AssessmentID int,
		UserLevel int,
		LockStatus int,
		LockStart datetime
	)

	INSERT @tmpRecordLock (AssessmentID, UserLevel, LockStatus, LockStart)
	SELECT assessment.AssessmentID, dbo.FNC_GetUserLevel(lock.UserID) UserLevel, LockStatus, LockStart 
	FROM QMS_RecordLock lock
	INNER JOIN QMS_Assessment assessment ON assessment.GUID = lock.RowGUID
	WHERE lock.ObjectName = 'QMS_Assessment'
	AND assessment.AssessmentID = @AssessmentID
	AND LockStatus IN (111, 112, 120, 121)

	-- Need to consider what happens if it's rejected or unarchived. It'll need a new sign-off from everyone.
	IF EXISTS (SELECT * FROM @tmpRecordLock WHERE LockStatus IN (111, 112))
		DELETE @tmpRecordLock
		WHERE LockStart <= (SELECT max(LockStart) FROM @tmpRecordLock WHERE LockStatus IN (111, 112))

	RETURN CASE WHEN EXISTS (SELECT * FROM @tmpRecordLock WHERE UserLevel = @UserLevelID) THEN 1 ELSE 0 END
END
GO



PRINT 'Sorting Stored Procedures...'

PRINT 'Sorting sync procedures...'

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_AssessmentSubType_Combo')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_AssessmentSubType_Combo] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author		:	Markp
-- Create date	:	18/08/2009
-- Description	:	Gets sub types for specified assessment type
-- Version		:	2
-- =============================================
-- PH TFS 13-24526 : 24/07/2015 Return all assessment sub types if no assessment type provided
-- =============================================
ALTER PROCEDURE [dbo].[QRY_AssessmentSubType_Combo] 
	@AssessmentTypeID int = null
AS

SELECT     AssessmentSubTypeID, AssessmentSubType
FROM         APP_AssessmentSubType
WHERE     (Hide = 0) AND (AssessmentTypeID = @AssessmentTypeID OR @AssessmentTypeID IS NULL) 
ORDER BY ListIndex

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_AssessmentQuestion_AdminList')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_AssessmentQuestion_AdminList] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 09/07/2008
-- Description:	Gets list of questions for specified assessment
-- Version:		10
-- =============================================
-- j.toolan 10 Apr 2012: [TFS: 3087] Added IsLocked flag which restricts users from editing the question in assesssment question admin screen.
--jf 26/10/12 - added hide from reports option
-- JE TFS 5456: added List Samples On Report col
-- JE TFS 5067: added Lookup_DataLookupDataTypeID col in order to access a lookup's data type
-- JT TFS 6621		- Added code to specify AssessmentQuestionID
-- JT TFS 6960		- Added code to include lookup for LookupAssessmentQuestionID (just list of questions in this assessment)
-- JE TFS 7951		- added MultiSampleSummary col
-- JT TFS 7785		- Added GuidanceRequired column.
-- PH TFS 13-18551	- Added IsQuestionGroup column, so that DataType controls can be hidden if group heading
-- =============================================
ALTER PROCEDURE [dbo].[QRY_AssessmentQuestion_AdminList] 
	@AssessmentID int,
	@AssessmentQuestionID int = NULL
AS

SET NOCOUNT ON
	IF @AssessmentQuestionID IS NOT NULL AND NULLIF(@AssessmentID, 0) IS NULL
		SELECT @AssessmentID = AssessmentID 
		FROM QMS_AssessmentQuestion 
		WHERE AssessmentQuestionID = @AssessmentQuestionID 


	IF NULLIF(object_id('tempdb..#tmpLookupQuestion'), 0) IS NOT NULL DROP TABLE #tmpLookupQuestion
	SELECT sorted.AssessmentQuestionID, convert(nvarchar(30), sorted.QuestionNumber) + N' ' + QMS_QuestionPool.Question 'Question'
	INTO #tmpLookupQuestion
	FROM dbo.FNC_AssessmentQuestion_Sort(@AssessmentID) sorted
	INNER JOIN QMS_AssessmentQuestion ON QMS_AssessmentQuestion.AssessmentQuestionID = sorted.AssessmentQuestionID 
	INNER JOIN QMS_QuestionPool ON QMS_AssessmentQuestion.QuestionID = QMS_QuestionPool.QuestionID 


SET NOCOUNT OFF

SELECT	QMS_AssessmentQuestion.AssessmentQuestionID,
		QMS_AssessmentQuestion.AssessmentID,
		FNC_AssessmentQuestion_Sort.QuestionNumber AS SortNumber, 
        QMS_AssessmentQuestion.QuestionNo,
		QMS_AssessmentQuestion.QuestionID,
		QMS_QuestionPool.Question AS Lookup_QuestionID, 
        QMS_AssessmentQuestion.ParentQuestionID,
		AssessmentQuestion_Parent.QuestionText AS Lookup_ParentQuestionID,
		QMS_AssessmentQuestion.DataTypeID,
        (SELECT     DataType
			FROM          APP_DataType
			WHERE      (DataTypeID = QMS_AssessmentQuestion.DataTypeID)) AS Lookup_DataTypeID,
		QMS_AssessmentQuestion.DataLookupID,
        (SELECT     DataLookup
			FROM          APP_DataLookup
			WHERE      (DataLookupID = QMS_AssessmentQuestion.DataLookupID)) AS Lookup_DataLookupID,
		QMS_AssessmentQuestion.CustomLookupID,
        (SELECT		DataTypeID --TFS 5067
			FROM		APP_DataLookup
			WHERE	(DataLookupID = QMS_AssessmentQuestion.DataLookupID)) AS Lookup_DataLookupDataTypeID,
        (SELECT     LookupName
			FROM          QMS_CustomLookup
			WHERE      (CustomLookupID = QMS_AssessmentQuestion.CustomLookupID)) AS Lookup_CustomLookupID,
		QMS_AssessmentQuestion.QuestionLevelID,
        (SELECT     QuestionLevel
			FROM          APP_QuestionLevel
			WHERE      (QuestionLevelID = QMS_AssessmentQuestion.QuestionLevelID)) AS Lookup_QuestionLevelID,
		QMS_AssessmentQuestion.GuidanceID, 
		QMS_Guidance.GuidanceDesc AS Lookup_GuidanceID,
		QMS_AssessmentQuestion.ShortGuidance,
		QMS_AssessmentQuestion.ResponseFormula, 
		QMS_AssessmentQuestion.ScoreCalculationID,
        (SELECT		ScoreCalculation
			FROM	APP_QuestionScoreCalculation
			WHERE	(ScoreCalculationID = QMS_AssessmentQuestion.ScoreCalculationID)) AS Lookup_ScoreCalculationID,
		QMS_AssessmentQuestion.ScoreFormula, 
		QMS_AssessmentQuestion.HasSamples, QMS_AssessmentQuestion.SampleSummaryID,
        (SELECT		SampleSummary
			FROM	APP_SampleSummary
			WHERE	(SampleSummaryID = QMS_AssessmentQuestion.SampleSummaryID)) AS Lookup_SampleSummaryID,
		QMS_AssessmentQuestion.Sample_DataTypeID,
		QMS_AssessmentQuestion.Multi_SampleSummaryID,
		(SELECT		SampleSummary
			FROM	APP_SampleSummary
			WHERE	(SampleSummaryID = QMS_AssessmentQuestion.Multi_SampleSummaryID)) AS Lookup_Multi_SampleSummaryID,
        (SELECT		DataType
			FROM	APP_DataType AS APP_DataType_1
			WHERE	(DataTypeID = QMS_AssessmentQuestion.Sample_DataTypeID)) AS Lookup_Sample_DataTypeID,
		QMS_AssessmentQuestion.MinSampleCount, 
		QMS_AssessmentQuestion.LookupAssessmentQuestionID, 
		(SELECT		Question
			FROM	#tmpLookupQuestion
			WHERE	AssessmentQuestionID = QMS_AssessmentQuestion.LookupAssessmentQuestionID) as Lookup_LookupAssessmentQuestionID,
		QMS_AssessmentQuestion.LookupSampleSet, 
		QMS_AssessmentQuestion.InstrumentID,
        (SELECT		InstrumentName
			FROM	APP_Instrument
			WHERE	(InstrumentID = QMS_AssessmentQuestion.InstrumentID)) AS Lookup_InstrumentID,
		QMS_AssessmentQuestion.Acceptable_Numeric_Min, 
		QMS_AssessmentQuestion.Acceptable_Numeric_Max,
		QMS_AssessmentQuestion.BackfillResponse,
		QMS_AssessmentQuestion.UpdateOverallQAFlag, 
		QMS_AssessmentQuestion.ResponseRequired,
		QMS_AssessmentQuestion.ChartQuestion,
		QMS_AssessmentQuestion.Hide,
		QMS_QuestionPool.Sys_Question,
		QMS_QuestionPool.Hide AS [Lookup_IsLocked],
		QMS_AssessmentQuestion.HideFromReports,
		QMS_AssessmentQuestion.ListSamplesOnReport,
		QMS_AssessmentQuestion.GuidanceRequired,
		QMS_QuestionPool.IsQuestionGroup 
FROM         QMS_AssessmentQuestion INNER JOIN
                      dbo.FNC_AssessmentQuestion_Sort(@AssessmentID) AS FNC_AssessmentQuestion_Sort ON 
                      QMS_AssessmentQuestion.AssessmentQuestionID = FNC_AssessmentQuestion_Sort.AssessmentQuestionID INNER JOIN
                      QMS_QuestionPool ON QMS_AssessmentQuestion.QuestionID = QMS_QuestionPool.QuestionID LEFT OUTER JOIN
                      QMS_Guidance ON QMS_AssessmentQuestion.GuidanceID = QMS_Guidance.GuidanceID LEFT OUTER JOIN
                          (SELECT     TOP (100) PERCENT QMS_AssessmentQuestion_Parent.AssessmentQuestionID, CAST(AssessmentQuestion_Sort_Parent.QuestionNumber AS nvarchar) 
                                                   + N' ' + QMS_QuestionPool_Parent.Question AS QuestionText
                            FROM          QMS_AssessmentQuestion AS QMS_AssessmentQuestion_Parent INNER JOIN
                                                   QMS_QuestionPool AS QMS_QuestionPool_Parent ON 
                                                   QMS_AssessmentQuestion_Parent.QuestionID = QMS_QuestionPool_Parent.QuestionID INNER JOIN
                                                   dbo.FNC_AssessmentQuestion_Sort(@AssessmentID) AS AssessmentQuestion_Sort_Parent ON 
                                                   QMS_AssessmentQuestion_Parent.AssessmentQuestionID = AssessmentQuestion_Sort_Parent.AssessmentQuestionID
                            WHERE      (QMS_AssessmentQuestion_Parent.AssessmentID = @AssessmentID) AND (QMS_AssessmentQuestion_Parent.RecordStatus < 201)
                            ORDER BY AssessmentQuestion_Sort_Parent.QuestionNumber, QMS_QuestionPool_Parent.Question) AS AssessmentQuestion_Parent ON 
                      QMS_AssessmentQuestion.ParentQuestionID = AssessmentQuestion_Parent.AssessmentQuestionID
WHERE     (QMS_AssessmentQuestion.AssessmentID = @AssessmentID) 
AND		(@AssessmentQuestionID IS NULL OR QMS_AssessmentQuestion.AssessmentQuestionID = @AssessmentQuestionID)
AND		(QMS_AssessmentQuestion.RecordStatus < 201) 
AND (QMS_QuestionPool.RecordStatus < 201)
ORDER BY FNC_AssessmentQuestion_Sort.SortIndex1, FNC_AssessmentQuestion_Sort.SortIndex2, FNC_AssessmentQuestion_Sort.SortIndex3, 
                      FNC_AssessmentQuestion_Sort.SortIndex4

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_Assessment_History')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_Assessment_History] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 28/10/2008
-- Description:	Gets version history for selected assessment
-- Version:		4
-- =============================================
-- JT TFS 4642 - Added default status (NULL) to FNC_AssessmentQuestion_VersionChanges
-- MP TFS 4642 - Added @UserID parameter to reference new FNC_AssessmentQuestion_VersionChanges_Selected instead
-- JT TFS 13-22563 - Return Assessment Status so that front end can avoid restoring non-archived assessments.
-- =============================================
ALTER PROCEDURE [dbo].[QRY_Assessment_History] 
	@Sys_AssessmentID INT,
	@UserID INT = NULL
AS

BEGIN
	SET NOCOUNT ON

	IF @UserID IS NULL
		SET @UserID = 1

	--Fill temp table with all assessments in view
	DELETE FROM dbo.TEMP_Selected
	WHERE UserID=@UserID
	
    INSERT  INTO dbo.TEMP_Selected
            ( UserID ,
              SelectedID 
            )
            SELECT  @UserID ,
                    AssessmentID
            FROM    QMS_Assessment
            WHERE   ( Sys_AssessmentID = @Sys_AssessmentID )
	
	SET NOCOUNT OFF

	SELECT     QMS_Assessment.AssessmentID, QMS_Assessment.AssessmentName, QMS_Assessment.AssessmentDesc, QMS_Assessment.Version, 
						  QMS_Assessment.ValidFrom, QMS_Assessment.ValidTo, APP_AssessmentStatus.AssessmentStatus, dbo.FNC_ConvertUTCDate(ISNULL(Modified.MaxLockFinish, 
						  ISNULL(QMS_Assessment.Updated, QMS_Assessment.Inserted))) AS Modified, Modified.FullName AS ModifiedBy, CurrentCheckOut.FullName AS CheckedOutBy, 
						  AssessmentQuestion_VersionChanges.QuestionText AS [Question changes],
						  QMS_Assessment.AssessmentStatusID
	FROM         QMS_Assessment INNER JOIN
						  APP_AssessmentStatus ON QMS_Assessment.AssessmentStatusID = APP_AssessmentStatus.AssessmentStatusID LEFT OUTER JOIN
						  dbo.FNC_AssessmentQuestion_VersionChanges_Selected(@UserID) AS AssessmentQuestion_VersionChanges ON 
						  QMS_Assessment.AssessmentID = AssessmentQuestion_VersionChanges.AssessmentID LEFT OUTER JOIN
							  (SELECT     VW_RecordLock_Current.RowGUID, VW_RecordLock_Current.UserID, QMS_User_1.FullName
								FROM          VW_RecordLock_Current INNER JOIN
													   QMS_User AS QMS_User_1 ON VW_RecordLock_Current.UserID = QMS_User_1.UserID
								WHERE      (VW_RecordLock_Current.ObjectName = N'QMS_Assessment')) AS CurrentCheckOut ON 
						  QMS_Assessment.GUID = CurrentCheckOut.RowGUID LEFT OUTER JOIN
							  (SELECT     MaxRecordLock.RowGUID, MaxRecordLock.MaxLockFinish, User_Modify.FullName
								FROM          QMS_RecordLock INNER JOIN
													   QMS_User AS User_Modify ON QMS_RecordLock.UserID = User_Modify.UserID INNER JOIN
														   (SELECT     RowGUID, MAX(LockFinish) AS MaxLockFinish
															 FROM          QMS_RecordLock AS RecordLock_Modified
															 WHERE      (RecordStatus < 201) AND (ObjectName = N'QMS_Assessment') AND (RecordStatus < 201)
															 GROUP BY RowGUID
															 HAVING      (MAX(LockFinish) IS NOT NULL)) AS MaxRecordLock ON QMS_RecordLock.RowGUID = MaxRecordLock.RowGUID AND 
													   QMS_RecordLock.LockFinish = MaxRecordLock.MaxLockFinish) AS Modified ON QMS_Assessment.GUID = Modified.RowGUID
	WHERE     (QMS_Assessment.RecordStatus < 201) AND (QMS_Assessment.Sys_AssessmentID = @Sys_AssessmentID)
	ORDER BY QMS_Assessment.[Version]
END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_Assessment_ActiveComboUser')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_Assessment_ActiveComboUser] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 22/07/2008
-- Description:	Gets active assessments
-- Version:		6
-- =============================================
-- Jonathan Toolan, 25-Aug-2011: Change so that we can use multiple types for an assessment.
-- Jonathan Toolan, 12-Sep-2011: Don't include deleted types from AssessmentTypeMulti.
-- JE TFS 2671 - added fill for assessment targets that have SourceObjectText instead of SourceObjectID
-- JE TFS 3485 - changed @CheckDate conversion to use standard SQL datetime format
-- JT TFS 13-22155: Final query changed so that it can only pick up masters (TempSpec = 0).
-- =============================================
ALTER PROCEDURE QRY_Assessment_ActiveComboUser 
	@AssessmentTypeID int,
	@UserID int
AS

DECLARE @CheckDate datetime

IF @AssessmentTypeID = 4 --label check
BEGIN --lookup depot date
	DECLARE @DepotDateString nvarchar(20)

	SELECT	@DepotDateString = TEMP_ResponseHeader.Header_Text
	FROM		TEMP_ResponseHeader 
	INNER JOIN	QMS_AssessmentHeader ON TEMP_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
	WHERE   (TEMP_ResponseHeader.UserID = @UserID) 
	AND		(QMS_AssessmentHeader.DataLookupID = 12)

	IF @DepotDateString IS NOT NULL
		SELECT @CheckDate = CONVERT(datetime, @DepotDateString)
END

IF @CheckDate IS NULL
	SET @CheckDate = GETDATE()
	
-- declare @AssessmentTypeID int, @UserID int SELECT @AssessmentTypeID = 2, @UserID = 1
DECLARE @Table_Assessment TABLE (AssessmentID int)

SET NOCOUNT ON

--Fill assessments with specific target / temp spec
INSERT INTO @Table_Assessment (AssessmentID)
SELECT DISTINCT QMS_Assessment.AssessmentID
FROM         APP_AssessmentStatus 
INNER JOIN	QMS_Assessment ON APP_AssessmentStatus.AssessmentStatusID = QMS_Assessment.AssessmentStatusID 
INNER JOIN	(	SELECT DISTINCT ai.AssessmentID
                FROM          TEMP_ResponseHeader AS rh 
				INNER JOIN	QMS_AssessmentTarget AS at ON rh.SourceObjectID = at.SourceObjectID 
				INNER JOIN	QMS_Assessment AS ai ON at.AssessmentID = ai.AssessmentID 
												AND rh.AssessmentTargetTypeID = ai.AssessmentTargetTypeID
                WHERE   (at.Hide = 0) 
				AND		(at.RecordStatus < 201) 
				AND		(rh.UserID = @UserID)

	) AS SelectedTargets ON QMS_Assessment.AssessmentID = SelectedTargets.AssessmentID 
INNER JOIN	(	SELECT     ash.AssessmentID, ash.Spec_SourceID AS SupplierID, ash.Spec_Text AS SupplierName
                FROM          QMS_AssessmentSpecHeader AS ash 
				INNER JOIN	QMS_SpecificationHeader AS sh ON ash.SpecificationHeaderID = sh.SpecificationHeaderID
                WHERE   (sh.RecordStatus < 201) 
				AND		(ash.RecordStatus < 201) 
				AND		(sh.Sys_SpecHeader = N'TempSpec_Supplier')

	) AS TempSpec_Supplier ON QMS_Assessment.AssessmentID = TempSpec_Supplier.AssessmentID 
INNER JOIN	(	SELECT     rh.Header_SourceID AS SupplierID, rh.Header_Text AS SupplierName
                FROM        TEMP_ResponseHeader AS rh 
				INNER JOIN	QMS_AssessmentHeader AS ah ON rh.AssessmentHeaderID = ah.AssessmentHeaderID
                WHERE   (ah.DataTypeID = 9) 
				AND		(ah.DataLookupID = 1) 
				AND		(rh.UserID = @UserID)

	) AS Selected_Supplier ON TempSpec_Supplier.SupplierID = Selected_Supplier.SupplierID
WHERE	(APP_AssessmentStatus.Active = 1) 
AND		(QMS_Assessment.RecordStatus < 201) 
AND		(QMS_Assessment.TempSpec = 1) 
AND		(QMS_Assessment.ValidFrom IS NULL OR QMS_Assessment.ValidFrom <= @CheckDate) 
AND		(QMS_Assessment.ValidTo IS NULL OR QMS_Assessment.ValidTo >= @CheckDate) 
AND		(-- assessment type matches
		QMS_Assessment.AssessmentTypeID = @AssessmentTypeID
		OR EXISTS	(	SELECT  AssessmentTypeMultiID
						FROM    QMS_AssessmentTypeMulti AS atm
						WHERE   (AssessmentID = QMS_Assessment.AssessmentID) 
						AND		(RecordStatus < 201) 
						AND		(AssessmentTypeID = @AssessmentTypeID)
		)
) 
	
IF (SELECT COUNT(AssessmentID) FROM @Table_Assessment) = 0
BEGIN
	--Fill assessments with specific target using SourceObjectID (e.g. StockCode)
	INSERT INTO @Table_Assessment (AssessmentID)
	SELECT     QMS_Assessment.AssessmentID
	FROM         APP_AssessmentStatus 
	INNER JOIN	QMS_Assessment ON APP_AssessmentStatus.AssessmentStatusID = QMS_Assessment.AssessmentStatusID 
	INNER JOIN	(	SELECT DISTINCT a.AssessmentID
					FROM        TEMP_ResponseHeader AS rh 
					INNER JOIN	QMS_AssessmentTarget AS at ON rh.SourceObjectID = at.SourceObjectID 
					INNER JOIN	QMS_Assessment AS a ON at.AssessmentID = a.AssessmentID 
													AND rh.AssessmentTargetTypeID = a.AssessmentTargetTypeID
					WHERE   (at.Hide = 0) 
					AND		(at.RecordStatus < 201) 
					AND		(rh.UserID = @UserID)

		) AS SelectedTargets ON QMS_Assessment.AssessmentID = SelectedTargets.AssessmentID 
	LEFT OUTER JOIN	(	SELECT  QMS_AssessmentSpecHeader.AssessmentID, 
								QMS_AssessmentSpecHeader.Spec_SourceID AS SupplierID,
								QMS_AssessmentSpecHeader.Spec_Text AS SupplierName
                        FROM        QMS_AssessmentSpecHeader 
						INNER JOIN	QMS_SpecificationHeader ON QMS_AssessmentSpecHeader.SpecificationHeaderID = QMS_SpecificationHeader.SpecificationHeaderID
                        WHERE	(QMS_SpecificationHeader.RecordStatus < 201) 
						AND		(QMS_AssessmentSpecHeader.RecordStatus < 201) 
						AND		(QMS_SpecificationHeader.Sys_SpecHeader = N'TempSpec_Supplier')

		) AS TempSpec_Supplier ON QMS_Assessment.AssessmentID = TempSpec_Supplier.AssessmentID
	WHERE     (APP_AssessmentStatus.Active = 1) 
	AND		(QMS_Assessment.RecordStatus < 201) 
	AND	(QMS_Assessment.ValidFrom IS NULL OR QMS_Assessment.ValidFrom <= @CheckDate) 
	AND (QMS_Assessment.ValidTo IS NULL OR QMS_Assessment.ValidTo >= @CheckDate) 
	AND (-- supplierID matches for temp specs.
			QMS_Assessment.TempSpec = 0
			OR 
			(QMS_Assessment.TempSpec = 1 AND TempSpec_Supplier.SupplierID IS NULL)
	) 
	AND (-- assessment type matches
			QMS_Assessment.AssessmentTypeID = @AssessmentTypeID
			OR EXISTS	(	SELECT     AssessmentTypeMultiID
							FROM    QMS_AssessmentTypeMulti AS atm
							WHERE   (AssessmentID = QMS_Assessment.AssessmentID) 
							AND		(RecordStatus < 201) 
							AND		(AssessmentTypeID = @AssessmentTypeID)
			)
	)
	
END

-- TFS 2671 --joins on SourceObjectText instead of SourceObjectID
IF (SELECT COUNT(AssessmentID) FROM @Table_Assessment) = 0
BEGIN
	--Fill assessments with target using SourceText (e.g. BusinessGroup)
	INSERT INTO @Table_Assessment (AssessmentID)
	SELECT     QMS_Assessment.AssessmentID
	FROM        APP_AssessmentStatus 
	INNER JOIN	QMS_Assessment ON APP_AssessmentStatus.AssessmentStatusID = QMS_Assessment.AssessmentStatusID 
	INNER JOIN	(	SELECT DISTINCT a.AssessmentID
                    FROM        TEMP_ResponseHeader AS rh 
					INNER JOIN	QMS_AssessmentTarget AS at ON rh.SourceObjectText = at.SourceObjectText 
					INNER JOIN	QMS_Assessment AS a ON at.AssessmentID = a.AssessmentID
                    WHERE   (at.Hide = 0) 
					AND		(at.RecordStatus < 201) 
					AND		(rh.UserID = @UserID)

	) AS SelectedTargets ON QMS_Assessment.AssessmentID = SelectedTargets.AssessmentID
	WHERE   (APP_AssessmentStatus.Active = 1) 
	AND		(QMS_Assessment.RecordStatus < 201) 
	AND		(QMS_Assessment.TempSpec = 0) 
	AND		(QMS_Assessment.ValidFrom IS NULL OR QMS_Assessment.ValidFrom <= @CheckDate) 
	AND		(QMS_Assessment.ValidTo IS NULL OR QMS_Assessment.ValidTo >= @CheckDate) 
	AND		(-- assessment type matches
				QMS_Assessment.AssessmentTypeID = @AssessmentTypeID
				OR EXISTS	(	SELECT     AssessmentTypeMultiID
								FROM          QMS_AssessmentTypeMulti AS atm
								WHERE   (AssessmentID = QMS_Assessment.AssessmentID) 
								AND		(RecordStatus < 201) 
								AND		(AssessmentTypeID = @AssessmentTypeID)
				)
	)
END

IF (SELECT COUNT(AssessmentID) FROM @Table_Assessment) = 0
BEGIN
	--Fill assessments with no target                      
	INSERT INTO @Table_Assessment (AssessmentID)
	SELECT     QMS_Assessment.AssessmentID
	FROM         APP_AssessmentStatus 
	INNER JOIN	QMS_Assessment ON APP_AssessmentStatus.AssessmentStatusID = QMS_Assessment.AssessmentStatusID 
	LEFT OUTER JOIN	(	SELECT     AssessmentID, COUNT(AssessmentTargetID) AS NrTargets
                        FROM          QMS_AssessmentTarget AS at
                        WHERE   (Hide = 0) 
						AND		(RecordStatus < 201)
                        GROUP BY AssessmentID

		) AS TargetCount ON QMS_Assessment.AssessmentID = TargetCount.AssessmentID
	WHERE	(APP_AssessmentStatus.Active = 1) 
	AND		(QMS_Assessment.RecordStatus < 201) 
	AND		(QMS_Assessment.ValidFrom IS NULL OR QMS_Assessment.ValidFrom <= @CheckDate) 
	AND		(QMS_Assessment.ValidTo IS NULL OR QMS_Assessment.ValidTo >= @CheckDate)
	AND		(QMS_Assessment.TempSpec = 0) -- when we get to here, we should have already eliminated temp specs as a possibility.
	AND ( -- no targets
			QMS_Assessment.AssessmentTargetTypeID IS NULL
		OR	TargetCount.NrTargets IS NULL 
		OR	TargetCount.NrTargets = 0
	)
	AND ( -- assessment type matches
			QMS_Assessment.AssessmentTypeID = @AssessmentTypeID
			OR EXISTS (	SELECT     AssessmentTypeMultiID
						FROM          QMS_AssessmentTypeMulti AS atm
						WHERE      (AssessmentID = QMS_Assessment.AssessmentID) 
						AND (RecordStatus < 201) 
						AND (AssessmentTypeID = @AssessmentTypeID)
			)
	)




END

SET NOCOUNT OFF

SELECT		a.AssessmentID, a.AssessmentName + CASE WHEN a.Version IS NULL THEN '' ELSE ' v' + a.Version END AS AssessmentText
FROM		QMS_Assessment a
INNER JOIN	@Table_Assessment AS SelectedAssessment ON a.AssessmentID = SelectedAssessment.AssessmentID
ORDER BY AssessmentText
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QAS_QRY_AssessmentSpecHeader_GrowerAdminList')
	EXEC ('CREATE PROCEDURE [dbo].[QAS_QRY_AssessmentSpecHeader_GrowerAdminList] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 21/07/2010
-- Description:	Gets the growers assigned to selected assessment
-- Version: 2
-- =============================================
-- Martin P TFS 13: Removed hardcoded WHERE	(QMS_SpecificationHeader.AssessmentTypeID = 2) and joined on QMS_SpecificationHeader where Hide = 0
-- =============================================
ALTER PROCEDURE [dbo].[QAS_QRY_AssessmentSpecHeader_GrowerAdminList] 
	@AssessmentID int,
	@UserID int
AS

SELECT     QMS_AssessmentSpecHeaderMulti.AssessmentSpecHeaderMultiID, PM_VW_UserGroup.GroupName, PM_VW_UserGroup.GroupCode, PM_VW_Supplier.SupplierName
FROM         QMS_SpecificationHeader 
					INNER JOIN QMS_AssessmentSpecHeader ON QMS_SpecificationHeader.SpecificationHeaderID = QMS_AssessmentSpecHeader.SpecificationHeaderID 
					INNER JOIN QMS_AssessmentSpecHeaderMulti ON  QMS_AssessmentSpecHeader.AssessmentSpecHeaderID = QMS_AssessmentSpecHeaderMulti.AssessmentSpecHeaderID 
					INNER JOIN PM_VW_UserGroup ON QMS_AssessmentSpecHeaderMulti.Multi_SourceID = PM_VW_UserGroup.UserGroupID 
					INNER JOIN PM_VW_Supplier ON PM_VW_UserGroup.SupplierID = PM_VW_Supplier.SupplierID 
					INNER JOIN dbo.FNC_User_SupplierList(@UserID) AS User_SupplierList ON PM_VW_Supplier.SupplierID = User_SupplierList.SupplierID
					INNER JOIN dbo.QAS_AssessmentType ON QAS_AssessmentType.AssessmentTypeID = QMS_SpecificationHeader.AssessmentTypeID
WHERE     	(QAS_AssessmentType.Hide = 0) 
			AND (QMS_SpecificationHeader.RecordStatus < 201) 
			AND (QMS_SpecificationHeader.Sys_SpecHeader = N'QAS_Growers') 
			AND (QMS_AssessmentSpecHeaderMulti.RecordStatus < 201) 
			AND (QMS_AssessmentSpecHeader.AssessmentID = @AssessmentID)
ORDER BY PM_VW_UserGroup.GroupName
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QAS_QRY_AssessmentSpecHeader_AdminList')
	EXEC ('CREATE PROCEDURE [dbo].[QAS_QRY_AssessmentSpecHeader_AdminList] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 15/04/2010
-- Description:	Gets the specification header for admin
-- Version:		8
-- =============================================
-- J.Toolan 12/1/12: Added defaults for n/a and OverrideTitle
-- JT TFS 6006: Added number of images column
-- JT TFS 6591:	Gets multi-select data for AssessmentSpecHeader, restricting growers to those allowed to be seen by the current user
-- JT TFS 7625: Duplicate rows no longer returned where deleted QMS_AssessmentTargets exist
-- JT TFS 13-8403: Duplicate links through to QAS_HeaderConfig table are removed.
-- Martin P TFS 13: Removed hardcoded WHERE	(QMS_SpecificationHeader.AssessmentTypeID = 2) and joined on QMS_SpecificationHeader where Hide = 0
-- =============================================
ALTER PROCEDURE [dbo].[QAS_QRY_AssessmentSpecHeader_AdminList] 
	@AssessmentID int,
	@UserID int = 1
AS
--DECLARE @AssessmentID int SELECT @AssessmentID = 47 --present with attachments
--DECLARE @AssessmentID int SELECT @AssessmentID = 2302 --present
--DECLARE @AssessmentID int SELECT @AssessmentID = 57 -- missing one.

SET NOCOUNT ON

DECLARE @QAS_HeaderConfig TABLE (
	SpecificationHeaderID int,
	OverrideTitle nvarchar(255), 
	NotApplicable bit,
	LineNumber int
)

-- Extract the logic to gather the OverrideTitle and NotApplicable from the QAS_HeaderConfig table. 
-- It was causing duplicates to appear in the code when two targets existed that pointed to different configs.
INSERT @QAS_HeaderConfig(SpecificationHeaderID, OverrideTitle, NotApplicable, LineNumber)
SELECT QAS_HeaderConfig.SpecificationHeaderID, 
		QAS_HeaderConfig.OverrideTitle, 
		QAS_HeaderConfig.NotApplicable,
		ROW_NUMBER() OVER (PARTITION BY QAS_HeaderConfig.SpecificationHeaderID ORDER BY QMS_AssessmentTarget.AssessmentTargetID) -- First entered target takes precedence.
FROM QMS_AssessmentTarget
LEFT OUTER JOIN PM_VW_StockCode ON QMS_AssessmentTarget.SourceObjectID = PM_VW_StockCode.StockCodeID
								--AND AssessmentTargetTypeID = 1 -- This *SHOULD* exist, but doesn't.
INNER JOIN QAS_HeaderConfig ON (StockCodeSubGroup_List IS NULL AND QAS_HeaderConfig.BusinessGroup = ISNULL(QMS_AssessmentTarget.SourceObjectText, PM_VW_StockCode.BusinessGroup))
							OR EXISTS (SELECT * FROM FNC_SplitString(StockCodeSubGroup_List, ',') WHERE Token = dbo.FNC_ExtractLeadingInteger(PM_VW_StockCode.StockCodeSubGroup))
WHERE QMS_AssessmentTarget.AssessmentID = @AssessmentID
AND QMS_AssessmentTarget.RecordStatus < 201
					
-- Remove any extra header configs (have at most one). To change which one is selected, alter the row_number function in the query above
DELETE @QAS_HeaderConfig WHERE LineNumber > 1

SET NOCOUNT OFF

SELECT  QMS_SpecificationHeader.SpecificationHeaderID, 
		ISNULL(QAS_HeaderConfig.OverrideTitle, QMS_SpecificationHeader.HeaderTitle) 'HeaderTitle', 
		QMS_SpecificationHeader.Sys_SpecHeader, 
		QMS_SpecificationHeader.AllowMultiSelect, 
		QMS_SpecificationHeader.DataTypeID, 
		AssessmentSpecHeader.AssessmentSpecHeaderID, 
		AssessmentSpecHeader.Spec_Text, 
		AssessmentSpecHeader.Spec_Boolean, 
		AssessmentSpecHeader.Spec_Date, 
		AssessmentSpecHeader.Spec_Numeric, 
		AssessmentSpecHeader.Spec_SourceID, 
		ISNULL(AssessmentSpecHeader.NotApplicable, QAS_HeaderConfig.NotApplicable) 'NotApplicable', 
		AssessmentSpecHeaderMulti.MultiSelectID, 
		AssessmentSpecHeaderMulti.MultiSelectText,
		NumberOfImages = (	SELECT count(AttachmentID) FROM QMS_Attachment
							WHERE   (QMS_Attachment.RecordStatus < 201) 
							AND		(QMS_Attachment.AttachmentSourceID = 4)
							AND		(QMS_Attachment.SourceObjectID = AssessmentSpecHeader.AssessmentSpecHeaderID))
FROM			QMS_SpecificationHeader
-- defaults for not applicable and override title.
LEFT OUTER JOIN @QAS_HeaderConfig as QAS_HeaderConfig ON QAS_HeaderConfig.SpecificationHeaderID = QMS_SpecificationHeader.SpecificationHeaderID 
LEFT OUTER JOIN (	SELECT  AssessmentSpecHeaderID, SpecificationHeaderID, Spec_Text, Spec_Boolean, Spec_Date, Spec_Numeric, Spec_SourceID, NotApplicable
					FROM    QMS_AssessmentSpecHeader
					WHERE   (RecordStatus < 201) 
					AND		(AssessmentID = @AssessmentID)) AS AssessmentSpecHeader ON AssessmentSpecHeader.SpecificationHeaderID = QMS_SpecificationHeader.SpecificationHeaderID
LEFT OUTER JOIN	dbo.FNC_AssessmentSpecHeaderMulti_Summary_ByUser(@AssessmentID, @UserID) AS AssessmentSpecHeaderMulti ON AssessmentSpecHeaderMulti.AssessmentSpecHeaderID = AssessmentSpecHeader.AssessmentSpecHeaderID 
INNER JOIN dbo.QAS_AssessmentType ON QAS_AssessmentType.AssessmentTypeID = QMS_SpecificationHeader.AssessmentTypeID
WHERE	(QAS_AssessmentType.Hide = 0)
AND		(QMS_SpecificationHeader.RecordStatus < 201)
AND (dbo.QMS_SpecificationHeader.Sys_SpecHeader LIKE 'QAS_%')
ORDER BY QMS_SpecificationHeader.ListIndex
GO



PRINT 'Data migration tasks...'

PRINT 'Database updated successfully!'
GO
