-- =============================================
-- Author:		Jon Edwards
-- Create date: 07/01/2015
-- Version:		0.1
-- Description:	Updates GreenlightQC_Mobile database for Sprint 56
-- =============================================

SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO

PRINT 'Looking up ProduceManager db...'
GO

IF nullif(object_id('tempdb..#tmpPMDBName'), 0) IS NOT NULL DROP TABLE #tmpPMDBName

CREATE TABLE #tmpPMDBName (PMDBName NVARCHAR(255))

DECLARE @DBName nvarchar(50), @PMDBName NVARCHAR(50)
SET @DBName = DB_NAME() 

IF LEN(@DBName)>12 
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,13,LEN(@DBName)-11)
ELSE
	SELECT @PMDBName = name FROM sys.databases WHERE name = 'ProduceManager' + SUBSTRING(@DBName,5,LEN(@DBName)-3)

IF @PMDBName IS NOT NULL
BEGIN
	INSERT INTO #tmpPMDBName ( PMDBName ) VALUES  ( @PMDBName )
	--PRINT '***Need to replace ProduceManager references in views to db: ' + @PMDBName
	--INSERT INTO dbo.QMS_DeploymentErrorLog (ScriptName, UserName,DeploymentDate, ObjectName)
	--SELECT ScriptName, USER_NAME(), GETUTCDATE(), 'Need to replace ProduceManager references in views to db: ' + @PMDBName
	--FROM #tmpScriptName
END
GO

--PRINT 'Inserting tables...'

IF  NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[PM_StockBatchItems]') AND type in (N'U'))
BEGIN
CREATE TABLE [dbo].[PM_StockBatchItems](
	[StockItemID] [int] IDENTITY(1,1) NOT FOR REPLICATION NOT NULL,
	[StockBatchID] [int] NULL,
	[StockCodeID] [int] NULL,
	[SupplierID] [int] NULL,
	[UseByDate] [datetime] NULL,
	[RecordStatus] [tinyint] NULL,
	[Source] [tinyint] NULL,
	[Updated] [datetime] NULL,
	[GUID] [uniqueidentifier] ROWGUIDCOL  NOT NULL,
	[Created] [bigint] NULL,
	[RowVersion] [timestamp] NOT NULL,
 CONSTRAINT [PK_PM_StockBatchItems] PRIMARY KEY CLUSTERED 
(
	[StockItemID] ASC
)WITH (PAD_INDEX = OFF, STATISTICS_NORECOMPUTE = OFF, IGNORE_DUP_KEY = OFF, ALLOW_ROW_LOCKS = ON, ALLOW_PAGE_LOCKS = ON) ON [PRIMARY]
) ON [PRIMARY]

ALTER TABLE [dbo].[PM_StockBatchItems] ADD  CONSTRAINT [DF_PM_StockBatchItems_RecordStatus]  DEFAULT ((2)) FOR [RecordStatus]

ALTER TABLE [dbo].[PM_StockBatchItems] ADD  CONSTRAINT [DF_PM_StockBatchItems_Source]  DEFAULT ((5)) FOR [Source]

ALTER TABLE [dbo].[PM_StockBatchItems] ADD  CONSTRAINT [DF_PM_StockBatchItems_Updated]  DEFAULT (getutcdate()) FOR [Updated]

ALTER TABLE [dbo].[PM_StockBatchItems] ADD  CONSTRAINT [DF_PM_StockBatchItems_GUID]  DEFAULT (newid()) FOR [GUID]

ALTER TABLE [dbo].[PM_StockBatchItems] ADD  CONSTRAINT [DF_PM_StockBatchItems_Created]  DEFAULT (@@dbts+(1)) FOR [Created]

ALTER TABLE [dbo].[PM_StockBatchItems]  WITH NOCHECK ADD  CONSTRAINT [FK_PM_StockBatchItems_PM_StockBatch] FOREIGN KEY([StockBatchID])
REFERENCES [dbo].[PM_StockBatch] ([StockBatchID])
ON UPDATE CASCADE
ON DELETE CASCADE
NOT FOR REPLICATION 

ALTER TABLE [dbo].[PM_StockBatchItems] CHECK CONSTRAINT [FK_PM_StockBatchItems_PM_StockBatch]

ALTER TABLE [dbo].[PM_StockBatchItems]  WITH NOCHECK ADD  CONSTRAINT [FK_PM_StockBatchItems_PM_StockCode] FOREIGN KEY([StockCodeID])
REFERENCES [dbo].[PM_StockCode] ([StockCodeID])
ON UPDATE CASCADE
ON DELETE CASCADE
NOT FOR REPLICATION 

ALTER TABLE [dbo].[PM_StockBatchItems] CHECK CONSTRAINT [FK_PM_StockBatchItems_PM_StockCode]

ALTER TABLE [dbo].[PM_StockBatchItems]  WITH NOCHECK ADD  CONSTRAINT [FK_PM_StockBatchItems_PM_Supplier] FOREIGN KEY([SupplierID])
REFERENCES [dbo].[PM_Supplier] ([SupplierID])
NOT FOR REPLICATION 

ALTER TABLE [dbo].[PM_StockBatchItems] NOCHECK CONSTRAINT [FK_PM_StockBatchItems_PM_Supplier]

END
GO


IF NOT EXISTS(SELECT sys.columns.object_id FROM sys.columns INNER JOIN sys.objects ON  sys.objects.object_id = sys.columns.object_id  where sys.objects.name ='PM_StockBatchItems'  and  sys.columns.name  = 'TotalQuantity')
alter table dbo.PM_StockBatchItems add
  TotalQuantity decimal(18,2)
GO


--PRINT 'Updating tables...'



--PRINT 'Creating indexes...'

--PRINT 'Updating Views...'

IF EXISTS (SELECT * FROM dbo.sysobjects WHERE id = OBJECT_ID(N'[dbo].[PM_VW_StockBatchItems]') AND xtype = 'V')
	DROP VIEW [dbo].[PM_VW_StockBatchItems]
GO

DECLARE @sql nvarchar(max)
SELECT @sql = '-- =============================================
-- Author:		Unknown
-- Create date: 29/06/2010
-- Description:	Gets header informaton for PM_GoodsIntake
-- Version:		5
-- =============================================
-- JT TFS 4286: 9-Aug-2012: Added PalletID
-- JF: Added ArrivalTemp and Certified
-- MP TFS 2798: Added Updated column
-- =============================================
create VIEW PM_VW_StockBatchItems
	AS
	SELECT        StockItemID, StockBatchID, StockCodeID, SupplierID, UseByDate, TotalQuantity
	FROM            dbo.PM_StockBatchItems
	WHERE        (RecordStatus < 201)'
	EXEC sp_executesql @sql
GO


--PRINT 'Dropping Views...'

PRINT 'Updating User Defined Functions...'
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_AssessmentType_IsWebCheck')
	EXEC('CREATE FUNCTION [dbo].[FNC_AssessmentType_IsWebCheck]() RETURNS bit AS BEGIN RETURN 0 END')
GO
--=============================================
-- Author:		Phil.Harding
-- Create date: 03/03/2015
-- Version:		1
-- Description: Indicates whether an AssessmentType is authorised for use in WebChecks front end (defined in APP_Menu - ParentID = 233)
--=============================================
-- PH TFS 13-15140: First iteration (Helps hide/show "Edit Check" icon)
--=============================================
ALTER FUNCTION [dbo].[FNC_AssessmentType_IsWebCheck] ( @AssessmentTypeID int )
RETURNS bit

AS
BEGIN

DECLARE @result bit = 0
DECLARE @recordCount int

SELECT @recordCount = COUNT(*)
FROM [dbo].[APP_Menu]
WHERE (Visible=1) AND (AppID = 3) AND (ParentID = 233) AND (MenuAction LIKE ('%/' + CONVERT(nvarchar(10),@AssessmentTypeID) + '/%'))

IF (@recordCount > 0) BEGIN SET @result = 1 END

RETURN @result

END
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'FNC_User_StockCodeList')
	EXEC('CREATE FUNCTION [dbo].[FNC_User_StockCodeList]() RETURNS @x TABLE (y bit) AS BEGIN RETURN END')
GO
-- =============================================
-- Author:		Markp
-- Create date: 02/12/2009
-- Description:	Returns list of stockcodes that user can access (for admin)
-- Version:		4
-- =============================================
-- JE updated 10/01/2012 - added filtering by Business Group
-- TFS 3655: j.toolan: 27/6/2012: Stop multiple StockcodeIDs from returning.
-- TFS 5028: remove primary supplier filter for supplier users
-- JT TFS 15423: Either have stockcode permissions OR supplier stockcode pack permissions. The two do not complement each other.
-- =============================================
ALTER FUNCTION [dbo].[FNC_User_StockCodeList] 
(
	@UserID int
)
RETURNS 
@StockCode TABLE 
(
	StockCodeID int
)
AS
BEGIN
	--Get table of available Business Groups
	DECLARE @Table_BusinessGroup TABLE (StockCodeBusinessGroup nvarchar(100))
	INSERT INTO @Table_BusinessGroup (StockCodeBusinessGroup)
	SELECT		PermissionTitle
	FROM		QMS_UserPermissions
	WHERE		(PermissionType = 106) AND (PermissionAllow = 1) AND (UserID = @UserID)

	--check for permissions in three levels of hierarchy: Business Group, Sub Group, Stock Code
	IF (SELECT COUNT(StockCodeBusinessGroup) FROM @Table_BusinessGroup)>0 --Business group restricted

		INSERT INTO @StockCode (StockCodeID)
		SELECT		StockCodeID
		FROM        PM_VW_StockCode
		WHERE		BusinessGroup IN (SELECT StockCodeBusinessGroup FROM @Table_BusinessGroup)

	ELSE IF (SELECT COUNT(StockCodeBusinessGroup) FROM @Table_BusinessGroup)=0 --no Business group restrictions
	BEGIN		
		--Get table of available sub-groups
		DECLARE @Table_SubGroup TABLE (StockCodeSubGroup nvarchar(100))
		INSERT INTO @Table_SubGroup (StockCodeSubGroup)
		SELECT     PermissionTitle
		FROM         QMS_UserPermissions
		WHERE     (PermissionType = 105) AND (PermissionAllow = 1) AND (UserID = @UserID)
		
		IF	(SELECT COUNT(StockCodeSubGroup) FROM @Table_SubGroup)>0  --Sub-group restricted
			INSERT INTO @StockCode (StockCodeID)
			SELECT     StockCodeID
			FROM         PM_VW_StockCode
			WHERE StockCodeSubGroup IN (SELECT StockCodeSubGroup FROM @Table_SubGroup)
		
		ELSE
		BEGIN

			-- if they have stockcode permissions, then it should only use stockcode permissions.
			-- if they have supplier permissions, then it will use supplier admin data to determine what stockcodes they can see.

			--Get table of available suppliers
			DECLARE @Table_Supplier TABLE (SupplierID int)
			INSERT INTO @Table_Supplier (SupplierID)
			SELECT     CAST(PermissionTitle AS int) AS SupplierID
			FROM         QMS_UserPermissions
			WHERE     (PermissionType = 101) AND (PermissionAllow = 1) AND (UserID = @UserID)
			
			--Get table with available stock codes
			DECLARE @Table_StockCode TABLE (StockCodeID int)
			INSERT INTO @Table_StockCode (StockCodeID)
			SELECT     CAST(PermissionTitle AS int) AS StockCodeID
			FROM         QMS_UserPermissions
			WHERE     (PermissionType = 104) AND (PermissionAllow = 1) AND (UserID = @UserID)
			

			IF (SELECT COUNT(StockCodeID) FROM @Table_StockCode)>0 --Has restrictions
				INSERT INTO @StockCode (StockCodeID)
				SELECT     StockCodeID
				FROM         @Table_StockCode
				GROUP BY StockCodeID

			--Add supplier stock codes to stock code table
			ELSE IF (SELECT COUNT(SupplierID) FROM @Table_Supplier)>0 
				INSERT INTO @StockCode (StockCodeID)
				SELECT  DISTINCT   PM_VW_StockCodePack.StockCodeID
				FROM         @Table_Supplier AS Supplier INNER JOIN
									  PM_VW_StockCodePack ON Supplier.SupplierID = PM_VW_StockCodePack.SupplierID			

			ELSE
				INSERT INTO @StockCode (StockCodeID)
				SELECT     StockCodeID
				FROM         PM_VW_StockCode



		END
	END
	RETURN 
END
GO


IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_ResponseHeader_StockCodeID]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
DROP FUNCTION [dbo].[FNC_ResponseHeader_StockCodeID]
GO

/****** Object:  UserDefinedFunction [dbo].[FNC_ResponseHeader_StockCodeID]    Script Date: 16/01/2015 11:23:37 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[FNC_ResponseHeader_StockCodeID]') AND type in (N'FN', N'IF', N'TF', N'FS', N'FT'))
BEGIN
execute dbo.sp_executesql @statement = N'-- =============================================
-- Author:		Markp
-- Create date: 25/03/2009
-- Description:	Gets stock code id for responses
-- Version:		5
-- =============================================
-- JT TFS 7830 - Also link through raw stockcode datalookup
-- JE TFS 8684 - added filter so deleted Header items are ignored
-- MP TFS 12720: Added check on ResponseID to mitigate duplicates
-- JF TFS 15314 - timeout issues - changed duplicate check to an outer join (Greencell had 100k records in @ResponseHeader)
-- =============================================
CREATE FUNCTION [dbo].[FNC_ResponseHeader_StockCodeID] 
(
)
RETURNS 
@ResponseHeader TABLE 
(
	ResponseID int,
	StockCodeID int
)
AS
BEGIN
	--Add header where stock code selected
	INSERT INTO @ResponseHeader (ResponseID, StockCodeID)
	SELECT     QMS_Response.ResponseID, QMS_ResponseHeader.Header_SourceID
	FROM         QMS_Response INNER JOIN
                      QMS_ResponseHeader ON QMS_Response.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
                      QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID
	WHERE     (QMS_Response.RecordStatus < 201) AND (QMS_ResponseHeader.Header_SourceID IS NOT NULL) AND (QMS_AssessmentHeader.DataTypeID = 9) 
                      AND (QMS_AssessmentHeader.DataLookupID IN (2, 35)) AND (QMS_AssessmentHeader.RecordStatus < 201)

	--Add consignment item headers
	INSERT INTO @ResponseHeader (ResponseID, StockCodeID)
	SELECT     QMS_Response.ResponseID, PM_VW_ConsignmentItems.StockCodeID
	FROM         QMS_Response INNER JOIN
                      QMS_ResponseHeader ON QMS_Response.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
                      QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
                      PM_VW_ConsignmentItems ON QMS_ResponseHeader.Header_SourceID = PM_VW_ConsignmentItems.ConsignmentItemID
					  LEFT OUTER JOIN @ResponseHeader as ResponseHeader ON ResponseHeader.ResponseID = QMS_Response.ResponseID
	WHERE     (QMS_Response.RecordStatus < 201) AND (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 6) AND
				(QMS_AssessmentHeader.RecordStatus < 201) 
				--AND NOT (QMS_Response.ResponseID IN (SELECT ResponseID FROM @ResponseHeader))
				AND ResponseHeader.ResponseID IS NULL
	
	--Add delivery item headers
	INSERT INTO @ResponseHeader (ResponseID, StockCodeID)
	SELECT     QMS_Response.ResponseID, PM_VW_ConsignmentItems.StockCodeID
	FROM         QMS_Response INNER JOIN
                      QMS_ResponseHeader ON QMS_Response.ResponseID = QMS_ResponseHeader.ResponseID INNER JOIN
                      QMS_AssessmentHeader ON QMS_ResponseHeader.AssessmentHeaderID = QMS_AssessmentHeader.AssessmentHeaderID INNER JOIN
                      PM_VW_GoodsIntake ON QMS_ResponseHeader.Header_SourceID = PM_VW_GoodsIntake.GoodsIntakeID INNER JOIN
                      PM_VW_ConsignmentItems ON PM_VW_GoodsIntake.ConsignmentItemID = PM_VW_ConsignmentItems.ConsignmentItemID
					  LEFT OUTER JOIN @ResponseHeader as ResponseHeader ON ResponseHeader.ResponseID = QMS_Response.ResponseID
	WHERE     (QMS_Response.RecordStatus < 201) AND (QMS_AssessmentHeader.DataTypeID = 9) AND (QMS_AssessmentHeader.DataLookupID = 8) AND
				(QMS_AssessmentHeader.RecordStatus < 201) 
				--AND NOT (QMS_Response.ResponseID IN (SELECT ResponseID FROM @ResponseHeader))
				AND ResponseHeader.ResponseID IS NULL

	RETURN 
END


' 
END

GO



PRINT 'Updating Stored Procedures...'
GO


IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[RPT_QuestionResponse_RemoveSelected]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[RPT_QuestionResponse_RemoveSelected]
GO


IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[RPT_QuestionResponse_RemoveSelected]') AND type in (N'P', N'PC') )

BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[RPT_QuestionResponse_RemoveSelected] AS' 
END
GO

-- =============================================
-- Author:		Adam C
-- Create date: 26/02/15
-- Description:	Further filters down the TEMP_Selected for the
--				question response report so the max allowed count 
--				check is more accurate
-- Version:		1
-- =============================================
alter PROCEDURE RPT_QuestionResponse_RemoveSelected
(
	@UserID INTEGER,
    @ShowGreen BIT,
	@ShowAmber BIT,
	@ShowRed BIT,
	@AssessmentType INTEGER
) 
AS
BEGIN

	DECLARE @ValidSelectedID TABLE (SelectedID INTEGER)
	INSERT INTO @ValidSelectedID
	SELECT DISTINCT SelectedID FROM TEMP_Selected INNER JOIN QMS_Response ON
	TEMP_Selected.SelectedID = QMS_Response.ResponseID INNER JOIN QMS_Assessment ON
	QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN QMS_ResponseQuestion ON
	QMS_ResponseQuestion.ResponseID = QMS_Response.ResponseID
	WHERE TEMP_Selected.UserID = @UserID
	AND ((@ShowGreen = 1 AND QMS_ResponseQuestion.Recommended_QAFlagID = 1)
	OR (@ShowAmber = 1 AND QMS_ResponseQuestion.Recommended_QAFlagID = 2)
	OR (@ShowRed = 1 AND QMS_ResponseQuestion.Recommended_QAFlagID = 3))
	AND QMS_Assessment.AssessmentTypeId = @AssessmentType


	DELETE FROM TEMP_Selected
	WHERE UserID = @UserID
	AND NOT EXISTS (SELECT SelectedID FROM @ValidSelectedID VS WHERE VS.SelectedID = TEMP_Selected.SelectedID)

END
GO


/****** Object:  StoredProcedure [dbo].[SAVE_CorrectiveActionUser]    Script Date: 21/01/2015 13:47:27 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SAVE_CorrectiveActionUser]') AND type in (N'P', N'PC'))
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[SAVE_CorrectiveActionUser] AS' 
END
GO
-- =============================================
-- Author:		Jon Edwards
-- Create date: 06/01/2015
-- Description:	Saves CA user data to QMS_CorrectiveActionUser
-- Version:		3
-- =============================================
-- JE TFS 13-12635: created
-- JE TFS 13-13404: amended INSERT for unassigned users into QMS_CorrectiveActionUser
-- JT TFS 13-14013: Simplified code, and remove corrective actions that you no longer have permission to see.
-- =============================================
alter PROCEDURE SAVE_CorrectiveActionUser
	@CorrectiveActionID int
AS
BEGIN
	SET NOCOUNT ON

	IF nullif(object_id('tempdb..#tmpCorrectiveActionUser'), 0) IS NOT NULL DROP TABLE #tmpCorrectiveActionUser
	CREATE TABLE #tmpCorrectiveActionUser (
		UserID int,
		Assigned bit
	)

	-- We want #tmpCorrectiveActionUser to contain exactly what QMS_CorrectiveAction should eventually contain.
	INSERT INTO #tmpCorrectiveActionUser(UserID, Assigned)
	SELECT UserID, 1
	FROM dbo.FNC_GENERATE_CorrectiveAction_AssignUsers()
	WHERE CorrectiveActionID = @CorrectiveActionID 

	INSERT INTO #tmpCorrectiveActionUser (UserID, Assigned)
	SELECT UserID, 0
	FROM dbo.FNC_GENERATE_CorrectiveAction_UserList(@CorrectiveActionID) as userlist
	WHERE UserID NOT IN (SELECT UserID FROM #tmpCorrectiveActionUser)


	-- UPDATE EXISTING
	UPDATE QMS_CorrectiveActionUser
	SET Assigned = CAUser.Assigned,
		Updated = GETUTCDATE()
	FROM QMS_CorrectiveActionUser
	INNER JOIN #tmpCorrectiveActionUser as CAUser ON CAUser.UserID = QMS_CorrectiveActionUser.UserID
													AND QMS_CorrectiveActionUser.CorrectiveActionID = @CorrectiveActionID 
	WHERE QMS_CorrectiveActionUser.Assigned <> CAUser.Assigned


	-- INSERT MISSING
	INSERT QMS_CorrectiveActionUser (CorrectiveActionID, UserID, Assigned)
	SELECT @CorrectiveActionID, CAUser.UserID, CAUser.Assigned 
	FROM #tmpCorrectiveActionUser AS CAUser
	WHERE NOT EXISTS (	SELECT 1 FROM QMS_CorrectiveActionUser
						WHERE CorrectiveActionID = @CorrectiveActionID 
						AND UserID = CAUser.UserID)


	-- DELETE REMOVED ONES
	DELETE QMS_CorrectiveActionUser
	WHERE CorrectiveActionID = @CorrectiveActionID
	AND UserID NOT IN (SELECT UserID FROM #tmpCorrectiveActionUser)


	DROP TABLE #tmpCorrectiveActionUser

	SET NOCOUNT OFF
END
GO


GO


IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SAVE_UserPermissions]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[SAVE_UserPermissions]
GO


IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'SAVE_UserPermissions')
	EXEC ('CREATE PROCEDURE [dbo].[SAVE_UserPermissions] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 02/07/2008
-- Description:	Saves or deletes user permissions
-- Version:		8
-- =============================================
-- JT TFS 13-1791: Lookup UserPermissionID if not specified, and convert AppID to PermissionType.
-- JT TFS 13-4134: Added permissions for add/edit/delete for menu items added for mobile (AppID = 2).
-- JT TFS 13-4012: Added code to remove favourites of children as well as current user.
-- JE TFS 13-12638: update QMS_CorrectiveActionUser when permissions have been changed
-- JE TFS 13-13177: update Assigned column in QMS_CorrectiveActionUser
-- JT TFS 13-15105: Only update corrective action user list when datapermissions are changed, not favourites or menu permissions.
-- =============================================
ALTER PROCEDURE dbo.SAVE_UserPermissions
	@UserID int,
	@UserPermissionID int,
	@PermissionType int,
	@PermissionTitle nvarchar(255),
	@PermissionAllow bit,
	@ChangedBy int = NULL -- UserID of user who is changing this entry.
AS
BEGIN
	SET NOCOUNT ON
	IF @ChangedBy IS NULL SET @ChangedBy = @UserID

	-- If PermissionType is an AppID, then change it to be a PermissionType
	IF @PermissionType IN (0,1,2,3)
	BEGIN
		SELECT @PermissionType = 
			CASE	WHEN EXISTS (	SELECT * FROM APP_Menu 
									WHERE MenuID = convert(int, @PermissionTitle)
									AND ParentID IS NULL) 
					THEN 0 ELSE 1 END
		
		-- Remove any other permissions which link to this userpermission (favourites, homepage, reportfavourites)
		IF @PermissionAllow = 0
		BEGIN
			UPDATE QMS_UserPermissions
			SET PermissionAllow = 0,
				Updated = getutcdate(),
				Deleted = getutcdate(),
				Changed_UserID = @ChangedBy
			WHERE PermissionType IN (4, 20, 21)
			AND PermissionTitle = @PermissionTitle
			AND (	UserID = @UserID 
					OR 
					UserID IN (	SELECT UserID FROM QMS_User children
								WHERE ParentUserID = @UserID)
			)
			AND PermissionAllow <> 0 -- only update if it would change from 1 to 0.
		END
		
		-- Do a menu item check for mobile permissions (only on leaf items)
		IF EXISTS (SELECT * FROM APP_Menu 
					WHERE MenuID = convert(int, @PermissionTitle)
					AND AppID = 2)
		AND NOT EXISTS (SELECT * FROM APP_Menu
						WHERE ParentID = convert(int, @PermissionTitle)
						AND AppID = 2)
		BEGIN
			-- Ensure that three child permissions exist: add/edit/delete
			UPDATE QMS_UserPermissions
			SET PermissionAllow = @PermissionAllow,
				Updated = getutcdate(),
				Deleted = CASE WHEN @PermissionAllow = 0 THEN getutcdate() ELSE Deleted END,
				Changed_UserID = @ChangedBy
			WHERE PermissionType IN (11,12, 13)
			AND PermissionTitle = @PermissionTitle
			AND UserID = @UserID
			AND PermissionAllow <> @PermissionAllow
			
			INSERT QMS_UserPermissions (UserID, PermissionType, PermissionTitle, PermissionAllow, Changed_UserID)
			SELECT @UserID, PermissionType, @PermissionTitle, @PermissionAllow, @ChangedBy 
			FROM (SELECT 11 PermissionType UNION SELECT 12 UNION SELECT 13) as t
			WHERE NOT EXISTS (
				SELECT * FROM QMS_UserPermissions
				WHERE PermissionType = t.PermissionType
				AND PermissionTitle = @PermissionTitle
				AND UserID = @UserID
			)
		END
	END

	-- Modify this actual permission.
	IF @UserPermissionID = 0
	BEGIN
		-- Treat zero and one permissiontypes as identical.
		DECLARE @SearchType int
		SELECT @SearchType = CASE WHEN @PermissionType IN (0,1) THEN 0 ELSE @PermissionType END

		SELECT @UserPermissionID = UserPermissionID
		FROM QMS_UserPermissions
		WHERE UserID = @UserID 
		AND CASE WHEN PermissionType IN (0,1) THEN 0 ELSE PermissionType END = @SearchType 
		AND PermissionTitle = @PermissionTitle;
	END

	IF ISNULL(@UserPermissionID,0)=0 --New permission
	BEGIN
		DECLARE @UserPermissions TABLE( UserPermissionID int);

		INSERT INTO QMS_UserPermissions
					(UserID, PermissionType, PermissionTitle, PermissionAllow, Changed_UserID)
		OUTPUT INSERTED.UserPermissionID 
			INTO @UserPermissions 
		VALUES     (@UserID,@PermissionType,@PermissionTitle,@PermissionAllow, @ChangedBy)
		
		--Get new UserPermissionID
		SELECT     @UserPermissionID = MAX(UserPermissionID)
		FROM         @UserPermissions
	END
	ELSE
	BEGIN
		UPDATE  QMS_UserPermissions
		SET     PermissionType = @PermissionType, 
				PermissionTitle = @PermissionTitle, 
				PermissionAllow = @PermissionAllow, 
				Updated = GETUTCDATE(),
				-- only change deleted if allow is set to false.
				Deleted = CASE WHEN @PermissionAllow = 0 THEN getutcdate() ELSE Deleted END,
				Changed_UserID = @ChangedBy
		WHERE     (UserPermissionID = @UserPermissionID)
	END

	-- Only data permissions will change corrective action visibilities.
	IF @PermissionType >= 100
	BEGIN

		--TFS 13-12638: update QMS_CorrectiveActionUser to reflect changes to user permissions
		--clear table of all CAs for the user whose permissions are being updated
		DELETE FROM QMS_CorrectiveActionUser
		WHERE UserID = @UserID

		--insert active CAs for UserID
		INSERT INTO QMS_CorrectiveActionUser(CorrectiveActionID, UserID)
		SELECT	CorrectiveActionID, @UserID
		FROM	dbo.FNC_GENERATE_User_CorrectiveActionList(@UserID, 1)

		--insert inactive CAs for UserID
		INSERT INTO QMS_CorrectiveActionUser (CorrectiveActionID, UserID)
		SELECT	CorrectiveActionID, @UserID
		FROM	dbo.FNC_GENERATE_User_CorrectiveActionList(@UserID, 0)

		--insert records for assignd users (results of function will have been updated by above insert/update actions)
		INSERT INTO QMS_CorrectiveActionUser (CorrectiveActionID, UserID, Assigned)
		SELECT	CorrectiveActionID, UserID, 1
		FROM	dbo.FNC_GENERATE_CorrectiveAction_AssignUsers()
		WHERE	(UserID = @UserID)
	END

	SET NOCOUNT OFF

	SELECT @UserPermissionID
END
GO







IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SYNC_ArchiveData]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[SYNC_ArchiveData]
GO


IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SYNC_ArchiveData]') AND type in (N'P', N'PC') )

BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[SYNC_ArchiveData] AS' 
END
GO

-- =============================================
-- Author:		Markp
-- Create date: 25/06/2010
-- Description:	Archives old data to reduce size of Mobile database
-- version: 4
-- jf tfs 14982 - Update SYNC_ArchiveData to include PM archiving based on DateRequired
-- jf tfs 15149 - return if archive days=0
-- jf tfs 15149 - use <= for date comparrisons as if there are more than 1000 consignments with same date it doesn't work
-- =============================================
ALTER PROCEDURE [dbo].[SYNC_ArchiveData] 
--exec SYNC_ArchiveData

AS

DECLARE @Days int
SET @Days = ISNULL(dbo.FNC_GetOptionValue(20),0)

IF @Days = 0
	RETURN 0
	
DECLARE @DateFrom datetime
SET @DateFrom = DATEADD(day,-@Days,GETDATE())
select @DateFrom

DECLARE @LastSent bigint

PRINT 'Delete QMS_ResponseQuestionSample...'

SELECT     @LastSent = ISNULL(LastSent,0)
FROM         SYNC_ClientAnchor
WHERE     (TableName = N'QMS_ResponseQuestionSample')

DELETE FROM QMS_ResponseQuestionSample
WHERE     (Created < @LastSent) AND (Inserted < @DateFrom)

PRINT 'Delete QMS_ResponseQuestion...'

SELECT     @LastSent = ISNULL(LastSent,0)
FROM         SYNC_ClientAnchor
WHERE     (TableName = N'QMS_ResponseQuestion')

DELETE FROM QMS_ResponseQuestion
WHERE     (Created < @LastSent) AND (Inserted < @DateFrom) AND (NOT (ResponseQuestionID IN
                          (SELECT     ResponseQuestionID
                            FROM          QMS_ResponseQuestionSample)))
                            
PRINT 'Delete QMS_ResponseHeader...'

SELECT     @LastSent = ISNULL(LastSent,0)
FROM         SYNC_ClientAnchor
WHERE     (TableName = N'QMS_ResponseHeader')

DELETE FROM QMS_ResponseHeader
WHERE     (Created < @LastSent) AND (Inserted < @DateFrom)

PRINT 'Delete QMS_Response...'

SELECT     @LastSent = ISNULL(LastSent,0)
FROM         SYNC_ClientAnchor
WHERE     (TableName = N'QMS_Response')

DELETE FROM QMS_Response
WHERE     (Created < @LastSent) AND (Inserted < @DateFrom) AND (NOT (ResponseID IN
                          (SELECT     ResponseID
                            FROM          QMS_ResponseQuestion)))
                            
PRINT 'Delete QMS_Attachment...'

SELECT     @LastSent = ISNULL(LastSent,0)
FROM         SYNC_ClientAnchor
WHERE     (TableName = N'QMS_Attachment')

DELETE FROM QMS_Attachment
WHERE     (Created < @LastSent) AND (Inserted < @DateFrom) AND (AttachmentSourceID = 1)

DELETE FROM QMS_Attachment
WHERE     (RecordStatus > 200)

--Release space on attachment table
DECLARE @DBName nvarchar(255)
SET @DBName = DB_NAME()
DBCC CLEANTABLE(@DBName,'QMS_Attachment')

PRINT 'Delete QMS_CorrectiveAction...'

SELECT     @LastSent = ISNULL(LastSent,0)
FROM         SYNC_ClientAnchor
WHERE     (TableName = N'QMS_CorrectiveAction')

DELETE FROM QMS_CorrectiveAction
WHERE     (Created < @LastSent) AND (Inserted < @DateFrom)
                            
PRINT 'Delete QMS_Notification...'

SELECT     @LastSent = ISNULL(LastSent,0)
FROM         SYNC_ClientAnchor
WHERE     (TableName = N'QMS_Notification')

DELETE FROM QMS_Notification
WHERE     (Created < @LastSent)

PRINT 'Delete QMS_ErrorLog...'

DELETE FROM QMS_ErrorLog
WHERE     (Occured < @DateFrom)



-- limit the amount of records deleted by getting the oldest 1000 consignment records - will take a few itterations to catch up
DECLARE @MinDateRequired DATETIME
SELECT @MinDateRequired = max(DateRequired) FROM  (SELECT TOP 1000 DateRequired FROM PM_Consignment WHERE DateRequired <=@DateFrom ORDER BY DateRequired ASC) tmp


PRINT 'Delete PM_StockBatchItems...'
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[PM_StockBatchItems]') AND type in (N'U'))
	DELETE FROM PM_StockBatchItems  
	WHERE StockBatchID IN 
		(SELECT StockBatchID FROM PM_StockBatchSource INNER JOIN PM_GoodsIntake ON PM_GoodsIntake.GoodsIntakeID = PM_StockBatchSource.GoodsIntakeID 
			INNER JOIN PM_ConsignmentItems ON PM_ConsignmentItems.ConsignmentID = PM_GoodsIntake.ConsignmentItemID  
			INNER JOIN PM_Consignment ON PM_ConsignmentItems.ConsignmentID  =PM_Consignment.ConsignmentID  WHERE PM_Consignment.DateRequired <= @MinDateRequired
		)


PRINT 'Delete PM_StockBatch...'
DELETE FROM PM_StockBatch 
WHERE StockBatchID IN 
		(SELECT StockBatchID FROM PM_StockBatchSource INNER JOIN PM_GoodsIntake ON PM_GoodsIntake.GoodsIntakeID = PM_StockBatchSource.GoodsIntakeID 
			INNER JOIN PM_ConsignmentItems ON PM_ConsignmentItems.ConsignmentID = PM_GoodsIntake.ConsignmentItemID  
			INNER JOIN PM_Consignment ON PM_ConsignmentItems.ConsignmentID  =PM_Consignment.ConsignmentID  WHERE PM_Consignment.DateRequired <= @MinDateRequired
		)


PRINT 'Delete PM_StockBatchSource...'
DELETE FROM PM_StockBatchSource
WHERE GoodsintakeID in	
		(SELECT GoodsIntakeID  FROM PM_GoodsIntake 
			INNER JOIN PM_ConsignmentItems ON PM_ConsignmentItems.ConsignmentID = PM_GoodsIntake.ConsignmentItemID  
			INNER JOIN PM_Consignment ON PM_ConsignmentItems.ConsignmentID  =PM_Consignment.ConsignmentID  WHERE PM_Consignment.DateRequired <= @MinDateRequired
		)




PRINT 'Delete PM_GoodsIntake...'
DELETE FROM PM_GoodsIntake 
WHERE ConsignmentItemID in 
		(SELECT ConsignmentItemID FROM PM_ConsignmentItems 
			INNER JOIN PM_Consignment ON PM_ConsignmentItems.ConsignmentID  =PM_Consignment.ConsignmentID  WHERE PM_Consignment.DateRequired <= @MinDateRequired
		)


PRINT 'Delete PM_ConsignmentItems...'
DELETE FROM PM_ConsignmentItems 
WHERE ConsignmentID  IN 
		(SELECT ConsignmentID FROM PM_Consignment WHERE DateRequired <= @MinDateRequired)


PRINT 'Delete PM_Consignment...'
DELETE FROM PM_Consignment WHERE DateRequired <= @MinDateRequired




SELECT     @LastSent = ISNULL(LastSent,0)
FROM         SYNC_ClientAnchor
WHERE     (TableName = N'QMS_Response')

DELETE FROM QMS_Response
WHERE     (Created < @LastSent) AND (Inserted < @DateFrom) AND (NOT (ResponseID IN
                          (SELECT     ResponseID
                            FROM          QMS_ResponseQuestion)))

GO




/****** Object:  StoredProcedure [dbo].[QRY_QuestionPool_AdminList]    Script Date: 06/02/2015 11:28:31 ******/
IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SYNC_Import_PM_StockBatchItems]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[SYNC_Import_PM_StockBatchItems]
GO


/****** Object:  StoredProcedure [dbo].[QRY_QuestionPool_AdminList]    Script Date: 06/02/2015 11:28:31 ******/
SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SYNC_Import_PM_StockBatchItems]') AND type in (N'P', N'PC'))
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[SYNC_Import_PM_StockBatchItems] AS' 
END
GO
-- =============================================
-- Author:		jf
-- Create date: 11/2/15
-- Description:	Imort StockBatchItems
-- Version:		1
-- =============================================
-- jf 11/2/15 - added @TotalQuantity
-- =============================================
ALTER PROCEDURE [dbo].[SYNC_Import_PM_StockBatchItems] 
	@StockItemID int, 
	@StockBatchID INt, 
	@StockCodeID int, 
	@SupplierID int, 
	@UseByDate datetime, 
	@RecordStatus tinyint, 
	@Source tinyint, 
	@Updated datetime,  
	@GUID uniqueidentifier,
	@Created bigint,
	@RowVersion timestamp,
	@TotalQuantity decimal(18,2),

	@InsertCount int = 0 OUTPUT,
	@UpdateCount int = 0 OUTPUT,
	@ReceivedAnchor bigint = 0 OUTPUT
AS

IF NOT EXISTS(SELECT * FROM PM_StockBatchItems WHERE (StockItemID = @StockItemID))
BEGIN --Append record
	IF @RecordStatus < 201
	BEGIN
		INSERT INTO PM_StockBatchItems
						(StockBatchID, StockCodeID, SupplierID, UseByDate, RecordStatus, Source, Updated, GUID, TotalQuantity )
                      --(StockBatchID, BatchType, LocationID, Barcode, BatchRef, QAStatus, DateAdded, DateClosed, ExpiryDate, Closed, RecordStatus, GUID)
		VALUES     (@StockBatchID,@StockCodeID,@SupplierID,@UseByDate,@RecordStatus,@Source,@Updated,@GUID,@TotalQuantity)
		SET @InsertCount = @@rowcount
	END
END

ELSE
BEGIN --Update record
	UPDATE    PM_StockBatchItems
	SET              StockBatchID=@StockBatchID, StockCodeID=@StockCodeID, SupplierID=@SupplierID, UseByDate=@UseByDate, RecordStatus=@RecordStatus, Source=@Source, Updated=@Updated, TotalQuantity=@TotalQuantity
	WHERE     (StockItemID = @StockItemID)
	SET @UpdateCount = @@rowcount
END

IF CAST(@RowVersion AS bigint) > @Created
	SET @ReceivedAnchor = CAST(@RowVersion AS bigint)
ELSE
	SET @ReceivedAnchor = @Created

GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[RPT_ResponseSummary_Attachments]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[RPT_ResponseSummary_Attachments]
GO

IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[RPT_ResponseSummary_Attachments]') AND type in (N'P', N'PC') )
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[RPT_ResponseSummary_Attachments] AS' 
END
GO

-- =============================================
-- Author:		Adam C
-- Create date: 20/02/15
-- Description:	Gets attachment data for QC summary report
-- Version:		1
-- =============================================
alter PROCEDURE RPT_ResponseSummary_Attachments
(
	@UserID INTEGER
) 
AS
BEGIN

	DECLARE @AttachmentCount INTEGER 
	SELECT @AttachmentCount = (	SELECT COUNT(AttachmentID) FROM QMS_Attachment 
								WHERE SourceObjectID IN (SELECT SelectedID FROM TEMP_Selected 
														WHERE UserID = @UserID) 
								AND QMS_Attachment.RecordStatus < 100)

	DECLARE @Attachments TABLE (
		RowNumber INTEGER, 
		SourceObjectID INTEGER, 
		AttachmentTitle NVARCHAR(MAX), 
		AttachmentFileName NVARCHAR(MAX), 
		Attachment IMAGE, 
		AttachmentID INTEGER, 
		OriginalRow BIT
	)
	INSERT INTO @Attachments
	SELECT TOP 20 ROW_NUMBER() OVER (ORDER BY APP_AssessmentType.ListIndex, QMS_Assessment.AssessmentName, QMS_Response.EndDate) AS RowNumber,
				  SourceObjectID, 
				  AttachmentTitle, 
				  AttachmentFileName, 
				  Attachment,
				  AttachmentID,
				  1
	FROM QMS_Attachment
	INNER JOIN QMS_Response ON QMS_Response.ResponseID = QMS_Attachment.SourceObjectID
	INNER JOIN QMS_Assessment ON QMS_Assessment.AssessmentID = QMS_Response.AssessmentID 
	INNER JOIN APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Assessment.AssessmentTypeID
	WHERE SourceObjectID IN (	SELECT SelectedID FROM TEMP_Selected 
								WHERE UserID = @UserID)
								AND QMS_Attachment.RecordStatus < 100
	ORDER BY APP_AssessmentType.ListIndex, QMS_Assessment.AssessmentName, QMS_Response.EndDate

	-- So that two attachments are next to each other.
	UPDATE @Attachments
	SET RowNumber = RowNumber - 1, OriginalRow = 0
	WHERE RowNumber % 2 = 0

	SELECT A1.SourceObjectID AS A1SourceObjectID, 
		   A1.AttachmentTitle AS A1AttachmentTitle, 
		   A1.AttachmentFileName AS A1AttachmentFileName, 
		   A1.Attachment AS A1Attachment,
		   CASE WHEN A1.AttachmentID IS NULL THEN NULL
		   ELSE 'Check No: ' + CAST(A1.SourceObjectID AS NVARCHAR(255)) + ' - ' + A1.AttachmentFileName
		   END AS A1Title,
		   A2.SourceObjectID AS A2SourceObjectID, 
		   A2.AttachmentTitle AS A2AttachmentTitle, 
		   A2.AttachmentFileName AS A2AttachmentFileName, 
		   A2.Attachment AS A2Attachment,
		   CASE WHEN A2.AttachmentID IS NULL THEN NULL
		   ELSE 'Check No: ' + CAST(A2.SourceObjectID AS NVARCHAR(255)) + ' - ' + A2.AttachmentFileName
		   END AS A2Title,
		   @AttachmentCount AS FullAttachmentCount 
	FROM @Attachments A1 
	LEFT JOIN @Attachments A2 ON A1.RowNumber = A2.RowNumber AND A1.AttachmentID <> A2.AttachmentID
	WHERE A1.OriginalRow = 1

END
GO


IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[RPT_ResponseSummary]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[RPT_ResponseSummary]
GO

IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[RPT_ResponseSummary]') AND type in (N'P', N'PC') )
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[RPT_ResponseSummary] AS' 
END
GO



-- =============================================
-- Author:		Markp
-- Create date: 14/10/08
-- Description:	Gets data for QC summary report
-- Version:		6
-- =============================================
-- Jonathan Toolan: 25-Aug-2011 AssessmentTypeID linked from Response, not Assessment.
-- JE TFS 4853: added QA Flag override config check and QA Flag cell colouring
-- JT TFS 6689: Added DataWarehouse code.
-- JE TFS 13-11439: added HideQAScore col
-- AC TFS 10156: Added ShowAttachments parameter and return value to show or hide the attachments from the report
-- =============================================
ALTER PROCEDURE RPT_ResponseSummary 
(
	@DateFrom datetime,
	@DateTo datetime,
	@UserID int,
	@ShowAttachments BIT = 0
)
AS 
BEGIN 

	DECLARE @UseDataWarehouse bit
	SET @UseDataWarehouse = dbo.FNC_CanUseDataWarehouse(@UserID)

	--get config value for Override QAFlag
	DECLARE @OverrideFlag bit
	SET @OverrideFlag = CAST(ISNULL(dbo.FNC_GetOptionValue(74),0) AS bit)

	if NULLIF(object_id('tempdb..#tmpQAFlag'), 0) IS NOT NULL DROP TABLE #tmpQAFlag
	SELECT	QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
	INTO #tmpQAFlag
	FROM	APP_QAFlag

	IF @UseDataWarehouse = 1 
	BEGIN
		SELECT     @DateFrom AS DateFrom, @DateTo AS DateTo, APP_AssessmentType.AssessmentType, QMS_Assessment.AssessmentName, response.ResponseID, 
							  response.StartDate, response.EndDate, response.Score as Actual_Score, QAFlag.QAFlagID, QAFlag.QAFlag, QMS_User.FullName, 
							  response.HeaderText AS SourceText, response.Comments, 
							  response.GreenResponses, response.GreenComments, 
							  response.AmberResponses, response.AmberComments, 
							  response.RedResponses, response.RedComments, 
							  response.QuantityReceived, response.QuantityChecked, response.QuantityAccepted, 
							  response.QuantityRejected, Temperature.Response_Temperature AS Temperature_Response, Temperature.QAFlagID AS Temperature_QAFlagID,
							  APP_QAFlag.QAFlagBackColour, APP_QAFlag.QAFlagForeColour,
							  ISNULL(dbo.FNC_GetOptionValue(15), N'0') AS HideQAScore,
							  @ShowAttachments AS ShowAttachments
		FROM        RPT_Warehouse_Response response
		INNER JOIN	APP_QAFlag ON response.QAFlagID = APP_QAFlag.QAFlagID 
		INNER JOIN	QMS_Assessment ON response.AssessmentID = QMS_Assessment.AssessmentID 
		INNER JOIN	APP_AssessmentType ON response.AssessmentTypeID = APP_AssessmentType.AssessmentTypeID 
		INNER JOIN	TEMP_Selected ON response.ResponseID = TEMP_Selected.SelectedID 
		LEFT OUTER JOIN	(	SELECT     InnerResponse.ResponseID, MAX(QMS_ResponseQuestion.Response_Numeric) AS Response_Temperature, 
														   MAX(QMS_ResponseQuestion.Recommended_QAFlagID) AS QAFlagID
							FROM		QMS_QuestionPool 
							INNER JOIN	QMS_AssessmentQuestion ON QMS_QuestionPool.QuestionID = QMS_AssessmentQuestion.QuestionID 
							INNER JOIN	QMS_ResponseQuestion ON QMS_AssessmentQuestion.AssessmentQuestionID = QMS_ResponseQuestion.AssessmentQuestionID 
							INNER JOIN	RPT_Warehouse_Response AS InnerResponse ON QMS_ResponseQuestion.ResponseID = InnerResponse.ResponseID 
							INNER JOIN	TEMP_Selected AS Selected ON InnerResponse.ResponseID = Selected.SelectedID
							WHERE      (QMS_QuestionPool.Question LIKE '%temp%') AND (Selected.UserID = @UserID)
							GROUP BY InnerResponse.ResponseID) AS Temperature ON response.ResponseID = Temperature.ResponseID 
		LEFT OUTER JOIN	QMS_User ON response.UserID = QMS_User.UserID 
		LEFT OUTER JOIN	#tmpQAFlag AS QAFlag ON APP_QAFlag.QAFlagID = QAFlag.QAFlagID
		WHERE     (TEMP_Selected.UserID = @UserID)
		ORDER BY APP_AssessmentType.ListIndex, QMS_Assessment.AssessmentName, response.EndDate
	END
	ELSE
	BEGIN
		SELECT     @DateFrom AS DateFrom, @DateTo AS DateTo, APP_AssessmentType.AssessmentType, QMS_Assessment.AssessmentName, QMS_Response.ResponseID, 
							  QMS_Response.StartDate, QMS_Response.EndDate, QMS_Response.Actual_Score, QAFlag.QAFlagID, QAFlag.QAFlag, QMS_User.FullName, 
							  AssessmentHeader.DisplayText AS SourceText, QMS_Response.Comments, FNC_ResponseQuestion.GreenResponses, FNC_ResponseQuestion.GreenComments, 
							  FNC_ResponseQuestion.AmberResponses, FNC_ResponseQuestion.AmberComments, FNC_ResponseQuestion.RedResponses, 
							  FNC_ResponseQuestion.RedComments, QMS_Response.QuantityReceived, QMS_Response.QuantityChecked, QMS_Response.QuantityAccepted, 
							  QMS_Response.QuantityRejected, Temperature.Response_Temperature AS Temperature_Response, Temperature.QAFlagID AS Temperature_QAFlagID,
							  APP_QAFlag.QAFlagBackColour, APP_QAFlag.QAFlagForeColour,
							  ISNULL(dbo.FNC_GetOptionValue(15), N'0') AS HideQAScore,
							  @ShowAttachments AS ShowAttachments
		FROM         QMS_Response INNER JOIN
							  APP_QAFlag ON
							  --check config value to join on Actual or Override QAFlagID
							  (CASE WHEN @OverrideFlag = 0 THEN Actual_QAFlagID ELSE ISNULL(Override_QAFlagID, Actual_QAFlagID) END)
									 = APP_QAFlag.QAFlagID INNER JOIN
							  QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
							  APP_AssessmentType ON QMS_Response.AssessmentTypeID = APP_AssessmentType.AssessmentTypeID INNER JOIN
							  TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID LEFT OUTER JOIN
							  dbo.FNC_ResponseQuestion_Selected(@UserID, 0) AS FNC_ResponseQuestion ON 
							  QMS_Response.ResponseID = FNC_ResponseQuestion.ResponseID LEFT OUTER JOIN
								  (SELECT     Response.ResponseID, MAX(QMS_ResponseQuestion.Response_Numeric) AS Response_Temperature, 
														   MAX(QMS_ResponseQuestion.Recommended_QAFlagID) AS QAFlagID
									FROM          QMS_QuestionPool INNER JOIN
														   QMS_AssessmentQuestion ON QMS_QuestionPool.QuestionID = QMS_AssessmentQuestion.QuestionID INNER JOIN
														   QMS_ResponseQuestion ON QMS_AssessmentQuestion.AssessmentQuestionID = QMS_ResponseQuestion.AssessmentQuestionID INNER JOIN
														   QMS_Response AS Response ON QMS_ResponseQuestion.ResponseID = Response.ResponseID INNER JOIN
														   TEMP_Selected AS Selected ON Response.ResponseID = Selected.SelectedID
									WHERE      (QMS_QuestionPool.Question LIKE '%temp%') AND (Selected.UserID = @UserID)
									GROUP BY Response.ResponseID) AS Temperature ON QMS_Response.ResponseID = Temperature.ResponseID LEFT OUTER JOIN
							  dbo.FNC_AssessmentHeader_Selected(@UserID) AS AssessmentHeader ON QMS_Response.ResponseID = AssessmentHeader.ResponseID LEFT OUTER JOIN
							  QMS_User ON QMS_Response.UserID = QMS_User.UserID 
							  LEFT OUTER JOIN #tmpQAFlag AS QAFlag ON APP_QAFlag.QAFlagID = QAFlag.QAFlagID
		WHERE     (TEMP_Selected.UserID = @UserID)
		ORDER BY APP_AssessmentType.ListIndex, QMS_Assessment.AssessmentName, QMS_Response.EndDate
	END

END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SAVE_CorrectiveAction]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[SAVE_CorrectiveAction]
GO

IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SAVE_CorrectiveAction]') AND type in (N'P', N'PC') )
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[SAVE_CorrectiveAction] AS' 
END
GO

-- =============================================
-- Author:		Markp
-- Create date: 01/08/2008
-- Description:	Saves corrective action
-- Version:		11
-- =============================================
-- j.toolan 21 Feb 2012: Added ReminderDate
-- j.toolan  7 Mar 2012: Reset NotifcationSent on 6 if AssignSupplierID set (only if status changes)
-- MP TFS 5873 - Made sure SourceObjectID updated so that correct links to response created
-- JE TFS 7087 - added CorrectiveActionTypeID to UPDATE statement
-- JE TFS 7877 - corrected resetting of NotifcationSent for Completed alerts
-- JE TFS 7836 - updated to save LocationID CA is assigned
-- JT TFS 13-7950 - Make sure that if CorrectiveActionTypeID is specified, and CorrectiveActionEscalationID is NOT, then set it.
-- JE TFS 13-12635: call new SAVE_CorrectiveActionUser SP to save permissions data to QMS_CorrectiveActionUser table
-- MP TFS 13-15377: excluded call to new SAVE_CorrectiveActionUser SP for mobile
-- =============================================
ALTER PROCEDURE [dbo].[SAVE_CorrectiveAction]
	@CorrectiveActionID int,
	@CASourceID int,
	@SourceObjectID int,
	@CATitle nvarchar(50),
	@CASubTitle nvarchar(50),
	@CAReason nvarchar(1000),
	@CARequired nvarchar(1000),
	@CAStatusID int,
	@CorrectiveActionTypeID int,
	@CorrectiveActionEscalationID int,
	@NotificationGroupID int,
	@AssignUserID int,
	@DateOccurred datetime,
	@PriorityLevelID int,
	@DateLogged datetime,
	@DateDeadline datetime,
	@ActionTaken nvarchar(1000),
	@Comments nvarchar(1000),
	@DateCompleted datetime,
	@Completed_UserID int,
	@Completed_MachineName nvarchar(50),
	@DateSignedOff datetime,
	@SignOff_UserID int,
	@SignOff_MachineName nvarchar(50),
	@UserID int,
	@AssignSupplierID int,
	@ReminderDate datetime,
	@ReminderSent bit = NULL, -- This is not saved currently.
	@LocationID int
AS
BEGIN
	SET NOCOUNT ON

	-- BUG 7950 SP FIX
	SELECT @LocationID = NULLIF(@LocationID, 0) -- Do not allow zero for LocationID. It stops first new CA alert going out.

	DECLARE @DefaultCorrectiveActionEscalationID int
	SELECT @DefaultCorrectiveActionEscalationID = CorrectiveActionEscalationID 
	FROM QMS_CorrectiveActionEscalation 
	WHERE CorrectiveActionTypeID = @CorrectiveActionTypeID
	AND ListIndex = (	SELECT min(ListIndex) FROM QMS_CorrectiveActionEscalation 
						WHERE CorrectiveActionTypeID = @CorrectiveActionTypeID)
	-- END BUG 7950 fix

	IF ISNULL(@CorrectiveActionID,0)=0 --New record
	BEGIN
		INSERT INTO QMS_CorrectiveAction
						  (CASourceID, SourceObjectID, CATitle, CASubTitle, CAReason, CARequired, CAStatusID, CorrectiveActionTypeID, CorrectiveActionEscalationID, NotificationGroupID, 
						  AssignUserID, DateOccurred, PriorityLevelID, DateLogged, DateDeadline, Logged_UserID, Logged_MachineName, ActionTaken, Comments, DateCompleted, 
						  Completed_UserID, Completed_MachineName, AssignSupplierID, ReminderDate, LocationID)
		VALUES     (@CASourceID,@SourceObjectID,@CATitle,@CASubTitle,@CAReason,@CARequired,@CAStatusID,@CorrectiveActionTypeID,
					ISNULL(@CorrectiveActionEscalationID, @DefaultCorrectiveActionEscalationID),
					@NotificationGroupID,@AssignUserID,@DateOccurred,@PriorityLevelID,@DateLogged,@DateDeadline,@UserID,
						   HOST_NAME(),@ActionTaken,@Comments,@DateCompleted,@Completed_UserID,@Completed_MachineName,@AssignSupplierID, @ReminderDate, @LocationID)
	                      
		--Get new ID
		SELECT     @CorrectiveActionID = MAX(CorrectiveActionID)
		FROM         QMS_CorrectiveAction
	END
	ELSE --Update record
	BEGIN
		DECLARE @originalCAStatusID int
		SELECT	@originalCAStatusID = CAStatusID 
		FROM	QMS_CorrectiveAction
		WHERE	CorrectiveActionID = @CorrectiveActionID

		UPDATE    QMS_CorrectiveAction
		SET              SourceObjectID = @SourceObjectID, CATitle = @CATitle, CASubTitle = @CASubTitle, CAReason = @CAReason, CARequired = @CARequired, 
							  CAStatusID = @CAStatusID, CorrectiveActionTypeID = @CorrectiveActionTypeID, NotificationGroupID = @NotificationGroupID, AssignUserID = @AssignUserID, DateOccurred = @DateOccurred, 
							  PriorityLevelID = @PriorityLevelID, DateDeadline = @DateDeadline, ActionTaken = @ActionTaken, Comments = @Comments, DateCompleted = @DateCompleted, 
							  Completed_UserID = @Completed_UserID, Completed_MachineName = @Completed_MachineName, DateSignedOff = @DateSignedOff, 
							  SignOff_UserID = @SignOff_UserID, SignOff_MachineName = @SignOff_MachineName, 
							  RecordStatus = CASE WHEN RecordStatus = 30 THEN 20 ELSE RecordStatus END, Updated = GETUTCDATE(), AssignSupplierID = @AssignSupplierID, 
							  ReminderDate = @ReminderDate, LocationID = @LocationID
		WHERE     (CorrectiveActionID = @CorrectiveActionID)

		-- Only update notification flag if the status has changed.
		IF (@originalCAStatusID <> @CAStatusID)
		BEGIN
				--get Failure_NotificationGroupID for current CA escalation
			DECLARE @Failure_NotificationGroupID int 
			SELECT     @Failure_NotificationGroupID = Failure_NotificationGroupID
			FROM         QMS_CorrectiveActionEscalation
			WHERE     (CorrectiveActionEscalationID = @CorrectiveActionEscalationID)

			--if CA being set to Complete and a Failure_NotificationGroup has been set, reset NotificationGroup flag
			IF @CAStatusID = 4 --AND @Failure_NotificationGroupID IS NOT NULL --removed for TFS 7877 as this stops Completed alerts firing when no Failure_NotificationGroupID set
			BEGIN
				UPDATE	QMS_CorrectiveAction
				SET		NotificationSent = 0
				WHERE	(CorrectiveActionID = @CorrectiveActionID)
			END
			
				--if CA being set to Signed-off and AssignSupplierID has been set, reset NotificationGroup flag
			IF @CAStatusID IN (5, 6) AND @AssignSupplierID IS NOT NULL
			BEGIN
				UPDATE	QMS_CorrectiveAction
				SET		NotificationSent = 0
				WHERE	(CorrectiveActionID = @CorrectiveActionID)
			END
		END
	END
	SET NOCOUNT OFF

	--save data for CA to QMS_CorrectiveActionUser (should handle both new and updated CAs as @CorrectiveActionID is set either way)
    IF DB_NAME() LIKE 'GreenlightQC_Mobile%'
        BEGIN --Add current user to current corrective action
            IF NOT EXISTS ( SELECT  CorrectiveActionID
                            FROM    dbo.QMS_CorrectiveActionUser
                            WHERE   UserID = @UserID
                                    AND CorrectiveActionID = @CorrectiveActionID )
                INSERT  INTO dbo.QMS_CorrectiveActionUser
                        ( CorrectiveActionID, UserID )
                VALUES  ( @CorrectiveActionID, -- CorrectiveActionID - int
                          @UserID -- UserID - int
                          )
        END
    ELSE
        EXEC SAVE_CorrectiveActionUser @CorrectiveActionID
	

	SELECT @CorrectiveActionID
END
GO

IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SYNC_Client_UpdateMachineName]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[SYNC_Client_UpdateMachineName]
GO

IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[SYNC_Client_UpdateMachineName]') AND type in (N'P', N'PC') )
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[SYNC_Client_UpdateMachineName] AS' 
END
GO

-- =============================================
-- Author:		Jonathan Toolan
-- Create date: 19/02/2015
-- Description:	Gets sync client data
-- Version:		1
-- ==============================================
-- JT TFS 14911: Update machine name on each Sync
-- ==============================================
ALTER PROCEDURE SYNC_Client_UpdateMachineName
	@ClientGuid as uniqueidentifier,
	@MachineName nvarchar(50)
AS
BEGIN
	DECLARE @ClientID int, @ClientType int
	SELECT TOP 1 @ClientID = ClientID, @ClientType = ClientType
	FROM SYNC_Client 
	WHERE @ClientGuid = ClientGuid
	ORDER BY [Enabled] DESC, LastSync DESC, Inserted DESC -- Most recently synched and enabled one first (in case there are more than one)

	UPDATE SYNC_Client
	SET MachineName = @MachineName
	WHERE ClientID = @ClientID 
	AND MachineName <> @MachineName -- Avoid a write unless we have to.

	UPDATE SYNC_Client
	SET Enabled = 0 
	WHERE MachineName = @MachineName
	AND ClientType = @ClientType
	AND LastSync < dateadd(day, -7, getutcdate()) -- Has *NOT* been synchronised in the last 7 days.
	AND @ClientID <> ClientID -- Not the one we just updated above.
	AND Enabled <> 0 -- Avoid a write unless we have to.

END
GO


IF  EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[QRY_CorrectiveAction_ResponseList]') AND type in (N'P', N'PC'))
DROP PROCEDURE [dbo].[QRY_CorrectiveAction_ResponseList]
GO


SET ANSI_NULLS ON
GO
SET QUOTED_IDENTIFIER ON
GO
IF NOT EXISTS (SELECT * FROM sys.objects WHERE object_id = OBJECT_ID(N'[dbo].[QRY_CorrectiveAction_ResponseList]') AND type in (N'P', N'PC'))
BEGIN
EXEC dbo.sp_executesql @statement = N'CREATE PROCEDURE [dbo].[QRY_CorrectiveAction_ResponseList] AS' 
END
GO



-- =============================================
-- Author:		Markp
-- Create date: 31/07/2008
-- Description:	Gets the corrective actions for specified response
-- Version:		5
-- =============================================
-- JE TFS 6272 - returns Child Check CAs if using Multi Checks
-- JE TFS 7836 - updated to return LocationID CA is assigned
-- JE TFS 13-6317: only build up temp table if a ResponseID is provided
-- JE TFS 13-6317: corrected error caused by previous change
-- JF TFS 15310 - Perf issues resolved (when responseid=0)
-- =============================================
ALTER PROCEDURE [dbo].[QRY_CorrectiveAction_ResponseList] 
	@ResponseID int
AS

--DBCC DROPCLEANBUFFERS
--exec QRY_CorrectiveAction_ResponseList 0

--temp table to hold Child Check ResponseIDs
IF NULLIF(object_id('tempdb..#tmpChildChecks'), 0) IS NOT NULL DROP TABLE #tmpChildChecks
CREATE TABLE #tmpChildChecks (ResponseID int)

IF @ResponseID > 0
BEGIN
	--get ParentResponseGUID for ResponseID passed in
	DECLARE @ParentResponseGUID uniqueidentifier
	SELECT	@ParentResponseGUID = GUID
	FROM	QMS_Response
	WHERE	ResponseID = @ResponseID

	--get all Child Checks for above GUID
	INSERT	#tmpChildChecks (ResponseID)
	SELECT	ResponseID
	FROM	QMS_Response
	WHERE	ParentResponseGUID = @ParentResponseGUID

	--get Child Check count
	DECLARE @ChildCheckCount int
	SELECT @ChildCheckCount = COUNT(ResponseID) FROM #tmpChildChecks
END
ELSE
	INSERT	#tmpChildChecks (ResponseID) VALUES (@ResponseID)



SELECT     CorrectiveActionID, CASourceID, SourceObjectID, CATitle, CAReason, CARequired, CAStatusID, CorrectiveActionTypeID, CorrectiveActionEscalationID, 
                      NotificationGroupID, PriorityLevelID, DateLogged, DateDeadline, ActionTaken, Comments, DateCompleted, Completed_UserID, Completed_MachineName, 
                      DateSignedOff, SignOff_UserID, SignOff_MachineName, CASubTitle, AssignUserID, DateOccurred, AssignSupplierID, ReminderDate, ReminderSent, LocationID
FROM         QMS_CorrectiveAction
WHERE     
(RecordStatus < 201) AND 
(CASourceID = 1) AND 
(SourceObjectID = @ResponseID) 
OR
(RecordStatus < 201) AND 
(CASourceID = 2) AND 
(SourceObjectID IN
				(SELECT     ResponseQuestionID
					FROM          QMS_ResponseQuestion
					WHERE      
						(RecordStatus < 201)  AND
						(
							--jf put @ResponseID into #tmpChildChecks when @ChildCheckCount = 0 to improve performance
							--(@ChildCheckCount = 0 AND ResponseID = @ResponseID) 
							--OR --just match one ResponseID if no Child Checks
							--(@ChildCheckCount > 0 AND ResponseID IN (SELECT ResponseID FROM #tmpChildChecks))
							ResponseID IN (SELECT ResponseID FROM #tmpChildChecks)
						)
				)
) --if there are Child Checks, get CAs for every check

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_Response_PreviousList')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_Response_PreviousList] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 28/07/2008
-- Description:	Gets list of previous responses
-- Version:		21
-- =============================================
-- Jonathan Toolan: 25-Aug-2011 AssessmentTypeID linked from Response, not Assessment.
-- Jonathan Toolan: 26-Mar-2012 Location Added.
-- Jonathan Toolan: TFS 3211 Multiple AssessmentTypeID_LIST
-- JE TFS 6272 - added Child Check/Top Level Check functionality
-- JT TFS 7346 - Added AssessmentID filter parameter.
-- JT TFS 7989 - Added ExportSetupTable to this SP as an example, so that custom SPs can copy and modify more easily.
-- JT TFS 8130 - ShowOutstanding to switch between "outstanding" mode and "completed" mode.
-- JT TFS 8578 - Added generic filtering if applied.
-- JE TFS 13-466: added provision for AQC Multi assessment format
-- JT TFS 13-612: Added DisplayMode parameter to show multi-check children (default) or parents.
-- JE TFS 13-1621: amended setting of @multiCheckDisplay so parent checks only are shown when isViewQCChecksScreen
-- JE TFS 13-2537: amended so that 3:isQCActionsScreen shows Parent checks only
-- JF TFS 2880 - Ensure that Child checks are available in Office Web - View QC Checks. 
				--Basically if the Multicheck contains a Summary column, then I would expect the Parent check to be referenced/displayed in ALL views
				--If no Summary Column (majority of multicheck customers) - then the Child check should be referenced/displayed in ALL views
-- TFS 2955		Navigate to Tasks > QC Actions for an assessment type of Active Quality Chain,Expected outcome: I am expecting to see the Parent checks here, Actual outcome: I am seeing the Child Checks
-- JE TFS 13-9699: only show non-Approved child checks for AQC assessment type
-- JE TFS 13-11892: output column indicating what sort of multi check response  is (i.e. parent or child) for AQC
-- JE TFS 13-12540: AQC Multi Check shows parent checks only for Previous tab
-- MP TFS 13-12735: Added logic to check for AQC approval config setting
-- jf 13042 - AQC may not be multicheck so allow for it
-- PH TFS 13-15140:  Added 'IsWebChecks' column to determine whether the Response is authorised to be edited using WebChecks front end (permission derived from APP_Menu with ParentID = 233)
-- =============================================
ALTER PROCEDURE [dbo].[QRY_Response_PreviousList]
    @AssessmentTypeID INT = NULL ,
    @AssessmentTypeID_LIST NVARCHAR(255) = NULL ,
    @DateFrom DATETIME ,
    @DateTo DATETIME ,
    @UserID INT ,
    @AssessmentID INT = NULL ,
    @ShowOutstanding BIT = 0 , -- 0: show completed checks only, 1: show outstanding checks only, NULL: show all checks regardless of ResponseStatus.
    @IsFilterApplied BIT = 0 ,
    @DisplayMode INT = NULL	
		-- 0 or NULL: not specified.
		-- 1:isPreviousChecksScreen
		-- 2:isViewQCChecksScreen
		-- 3:isQCActionsScreen
		-- 4:isQCCheckReport
AS 
    BEGIN
        SET NOCOUNT ON
    
-- If (isPreviousChecksScreen), then show parent checks (1), otherwise show children (2).
-- @multiCheckDisplay should match return value from dbo.FNC_Response_IsMultiCheck.

--FNC_Response_IsMultiCheck results
--0 -- Isn't multi-check at all.
--1 -- Is a parent multi-check
--2 -- Is a child multi-check
																				
        

--get Assessment Format
        DECLARE @AssessmentFormat INT
        SELECT  @AssessmentFormat = AssessmentFormat
        FROM    APP_AssessmentType
        WHERE   AssessmentTypeID = @AssessmentTypeID


		DECLARE @multiCheckDisplay INT;
        SELECT  @multiCheckDisplay = CASE WHEN ISNULL(@DisplayMode, 0) IN ( 1 ) 
										THEN  1    --show parent checks only for previouschecks screen
										WHEN @AssessmentFormat = 6 THEN 1
										ELSE 2
									END
									
		DECLARE @AQCApproval bit
		SELECT @AQCApproval = ISNULL(dbo.FNC_GetOptionValue(120),0)

        IF ( @AssessmentTypeID_LIST IS NULL ) 
            SET @AssessmentTypeID_LIST = CONVERT(NVARCHAR, @AssessmentTypeID)

--Clear temp table
        DELETE  FROM TEMP_Selected
        WHERE   ( UserID = @UserID )

--Fill temp table
        IF @IsFilterApplied = 0 
            BEGIN
                INSERT INTO TEMP_Selected
                      (SelectedID, UserID)
				SELECT     QMS_Response.ResponseID, @UserID AS UserID
				FROM         QMS_Response INNER JOIN
                      dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
                      QMS_Response.ResponseID = User_ResponseList.ResponseID INNER JOIN
                      dbo.FNC_SplitString(@AssessmentTypeID_LIST, ',') AS SplitString ON QMS_Response.AssessmentTypeID = SplitString.Token

            END
        ELSE 
            BEGIN
                INSERT INTO TEMP_Selected
                      (SelectedID, UserID)
				SELECT     ResponseID, @UserID AS UserID
				FROM         TEMP_ResponseFilter
				WHERE     (UserID = @UserID)

            END

        SET NOCOUNT OFF

        IF @AssessmentTypeID_LIST = '2' --Arrival check
            BEGIN
                SELECT     QMS_Response.ResponseID, Attachments.NumAttachments, QMS_Assessment.AssessmentName, ResponseHeader.Supplier, ResponseHeader.StockCode, 
                      ResponseHeader.Location, QMS_Response.StartDate, QMS_Response.EndDate, QMS_Response.Actual_Score, QMS_Response.Actual_QAFlagID, QAFlag.QAFlag, 
                      QMS_Response.Override_QAFlagID, QAFlag_Override.QAFlag AS Override_QAFlag, QMS_Response.Offline_ResponseID, 
                      CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete, dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, QMS_User.FullName
				FROM         QMS_Response INNER JOIN
                      TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID INNER JOIN
                      --APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID INNER JOIN 
                      QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
                      QMS_User ON QMS_Response.UserID = QMS_User.UserID INNER JOIN
                      dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
                      QMS_Response.ResponseID = User_ResponseList.ResponseID LEFT OUTER JOIN
                      dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID LEFT OUTER JOIN
                          (SELECT     SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
                            FROM          QMS_Attachment
                            WHERE      (AttachmentSourceID = 1) AND (RecordStatus < 201)
                            GROUP BY SourceObjectID) AS Attachments ON QMS_Response.ResponseID = Attachments.ResponseID LEFT OUTER JOIN
                          (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
                            FROM          APP_QAFlag) AS QAFlag_Override ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID LEFT OUTER JOIN
                          (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
                            FROM          APP_QAFlag AS APP_QAFlag_1) AS QAFlag ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
				WHERE     
				
					(QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND 
					(@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL) AND 
					(dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND 
					(@ShowOutstanding IS NULL) 
					OR                      
					(QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND 
					(@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL) AND 
					(dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND 
					(@ShowOutstanding = 1) AND 
                    (QMS_Response.ResponseStatus = 0) 
                    OR
                    (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND 
                    (@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL) AND 
                    (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND 
                    (@ShowOutstanding = 0) AND 
                    (QMS_Response.ResponseStatus <> 0)
                    
				ORDER BY QMS_Response.EndDate DESC
            END
        ELSE 
            IF @AssessmentTypeID_LIST IN ( '3', '4', '5' ) --Online/Label/Dispatch
                BEGIN
                    SELECT     QMS_Response.ResponseID, Attachments.NumAttachments, ResponseHeader.Customer, ResponseHeader.StockCode, ResponseHeader.Location, 
                      ResponseHeader.DepotDate AS [Depot date], QMS_Assessment.AssessmentName, QMS_Response.StartDate, QMS_Response.EndDate, 
                      QMS_Response.Actual_QAFlagID, QAFlag.QAFlag, QMS_Response.Override_QAFlagID, QAFlag_Override.QAFlag AS Override_QAFlag, 
                      QMS_Response.Offline_ResponseID, CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete, dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, 
                      QMS_User.FullName
					FROM         QMS_Response INNER JOIN
                      TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID INNER JOIN
                      --APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID INNER JOIN 
                      QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
                      QMS_User ON QMS_Response.UserID = QMS_User.UserID INNER JOIN
                      dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
                      QMS_Response.ResponseID = User_ResponseList.ResponseID LEFT OUTER JOIN
                      dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID LEFT OUTER JOIN
                          (SELECT     SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
                            FROM          QMS_Attachment
                            WHERE      (AttachmentSourceID = 1) AND (RecordStatus < 201)
                            GROUP BY SourceObjectID) AS Attachments ON QMS_Response.ResponseID = Attachments.ResponseID LEFT OUTER JOIN
                          (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
                            FROM          APP_QAFlag) AS QAFlag_Override ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID LEFT OUTER JOIN
                          (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
                            FROM          APP_QAFlag) AS QAFlag ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
					WHERE     (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
                      @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding IS NULL) OR
                      (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
                      @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 1) AND 
                      (QMS_Response.ResponseStatus = 0) OR
                      (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
                      @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 0) AND 
                      (QMS_Response.ResponseStatus <> 0)
					ORDER BY QMS_Response.EndDate DESC
                END
            ELSE 
                IF @AssessmentTypeID_LIST = '6' --Shelf life
                    BEGIN
                        SELECT     QMS_Response.ResponseID, Attachments.NumAttachments, ResponseHeader.Customer, ResponseHeader.StockCode, ResponseHeader.Location, 
							  ResponseHeader.AssessmentSubType AS [Test type], ResponseHeader.DepotDate AS [Depot date], QMS_Assessment.AssessmentName, QMS_Response.StartDate, 
							  QMS_Response.EndDate, QMS_Response.Actual_QAFlagID, QAFlag.QAFlag, QMS_Response.Override_QAFlagID, QAFlag_Override.QAFlag AS Override_QAFlag, 
							  QMS_Response.Offline_ResponseID, CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete, dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, 
							  QMS_User.FullName
						FROM         QMS_Response INNER JOIN
							  TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID INNER JOIN
							  --APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID INNER JOIN 
							  QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
							  QMS_User ON QMS_Response.UserID = QMS_User.UserID INNER JOIN
							  dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
							  QMS_Response.ResponseID = User_ResponseList.ResponseID LEFT OUTER JOIN
							  dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID LEFT OUTER JOIN
								  (SELECT     SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
									FROM          QMS_Attachment
									WHERE      (AttachmentSourceID = 1) AND (RecordStatus < 201)
									GROUP BY SourceObjectID) AS Attachments ON QMS_Response.ResponseID = Attachments.ResponseID LEFT OUTER JOIN
								  (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
									FROM          APP_QAFlag) AS QAFlag_Override ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID LEFT OUTER JOIN
								  (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
									FROM          APP_QAFlag) AS QAFlag ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
						WHERE     (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding IS NULL) OR
							  (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 1) AND 
							  (QMS_Response.ResponseStatus = 0) OR
							  (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST)) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 0) AND 
							  (QMS_Response.ResponseStatus <> 0)
						ORDER BY QMS_Response.EndDate DESC
                    END
            ELSE 
                IF @AssessmentTypeID_LIST = '16' --Active Quality Chain
                    BEGIN
                        SELECT     QMS_Response.ResponseID, Attachments.NumAttachments, ResponseHeader.Customer, ResponseHeader.StockCode, ResponseHeader.Location, 
							  ResponseHeader.AssessmentSubType AS [Test type], ResponseHeader.DepotDate AS [Depot date], QMS_Assessment.AssessmentName, QMS_Response.StartDate, 
							  QMS_Response.EndDate, QMS_Response.Actual_QAFlagID, QAFlag.QAFlag, QMS_Response.Override_QAFlagID, QAFlag_Override.QAFlag AS Override_QAFlag, 
							  QMS_Response.Offline_ResponseID, CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) = 0 THEN 0 ELSE 1 END AS BIT) AS Complete, dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, 
							  QMS_User.FullName, CAST(CASE WHEN @AQCApproval = 1 THEN ISNULL(dbo.FNC_Response_IsApproved(QMS_Response.ResponseID), 0) ELSE 1 END AS BIT) AS IsApproved,
							  ISNULL(dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID), 0) AS MultiCheckLevel
						FROM         QMS_Response INNER JOIN
							  TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID INNER JOIN
							  QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
							  QMS_User ON QMS_Response.UserID = QMS_User.UserID INNER JOIN
							  dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
							  QMS_Response.ResponseID = User_ResponseList.ResponseID LEFT OUTER JOIN
							  dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID LEFT OUTER JOIN
								  (SELECT     SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
									FROM          QMS_Attachment
									WHERE      (AttachmentSourceID = 1) AND (RecordStatus < 201)
									GROUP BY SourceObjectID) AS Attachments ON QMS_Response.ResponseID = Attachments.ResponseID LEFT OUTER JOIN
								  (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
									FROM          APP_QAFlag) AS QAFlag_Override ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID LEFT OUTER JOIN
								  (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
									FROM          APP_QAFlag) AS QAFlag ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID
						WHERE (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST))
								AND ((@multiCheckDisplay = 1 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 1) OR (@multiCheckDisplay = 2 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (1,2)) OR dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 0)
								AND (@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL)
								AND (@ShowOutstanding IS NULL) OR
							  (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST))
								AND ((@multiCheckDisplay = 1 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 1) OR (@multiCheckDisplay = 2 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (1,2)) OR dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 0)
								AND (@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL)
								AND (@ShowOutstanding = 1)
								AND (QMS_Response.ResponseStatus = 0) OR
							  (QMS_Response.AssessmentTypeID = CONVERT(INT, @AssessmentTypeID_LIST))
								AND ((@multiCheckDisplay = 1 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 1) OR (@multiCheckDisplay = 2 AND dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (1,2)) OR dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) = 0)
								AND (@AssessmentID = QMS_Response.AssessmentID OR @AssessmentID IS NULL)
								AND (@ShowOutstanding = 0)
								AND (QMS_Response.ResponseStatus <> 0)
						ORDER BY QMS_Response.EndDate DESC
                    END
                ELSE 
                    BEGIN
                        SELECT     QMS_Response.ResponseID, Attachments.NumAttachments, QMS_Assessment.AssessmentName, ResponseHeader.Location, QMS_Response.StartDate, 
							  QMS_Response.EndDate, QMS_Response.Actual_Score, QMS_Response.Actual_QAFlagID, QAFlag.QAFlag, QMS_Response.Override_QAFlagID, 
							  QAFlag_Override.QAFlag AS Override_QAFlag, QMS_Response.Offline_ResponseID, CAST(CASE WHEN ISNULL(QMS_Response.ResponseStatus, 0) 
							  = 0 THEN 0 ELSE 1 END AS BIT) AS Complete, dbo.FNC_AssessmentType_IsWebCheck(QMS_Response.AssessmentTypeID) AS IsWebCheck, QMS_User.FullName
						FROM         QMS_Response INNER JOIN
							  TEMP_Selected ON QMS_Response.ResponseID = TEMP_Selected.SelectedID AND TEMP_Selected.UserID = @UserID INNER JOIN
							  --APP_AssessmentType ON APP_AssessmentType.AssessmentTypeID = QMS_Response.AssessmentTypeID INNER JOIN 
							  QMS_Assessment ON QMS_Response.AssessmentID = QMS_Assessment.AssessmentID INNER JOIN
							  QMS_User ON QMS_Response.UserID = QMS_User.UserID INNER JOIN
							  dbo.FNC_User_ResponseList(@UserID, @DateFrom, @DateTo) AS User_ResponseList ON 
							  QMS_Response.ResponseID = User_ResponseList.ResponseID LEFT OUTER JOIN
								  (SELECT     SourceObjectID AS ResponseID, COUNT(AttachmentID) AS NumAttachments
									FROM          QMS_Attachment
									WHERE      (AttachmentSourceID = 1) AND (RecordStatus < 201)
									GROUP BY SourceObjectID) AS Attachments ON QMS_Response.ResponseID = Attachments.ResponseID LEFT OUTER JOIN
								  (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
									FROM          APP_QAFlag) AS QAFlag_Override ON QMS_Response.Override_QAFlagID = QAFlag_Override.QAFlagID LEFT OUTER JOIN
								  (SELECT     QAFlagID, ISNULL(dbo.FNC_ResourceString_User(ResourceName, @UserID), QAFlag) AS QAFlag
									FROM          APP_QAFlag) AS QAFlag ON QMS_Response.Actual_QAFlagID = QAFlag.QAFlagID LEFT OUTER JOIN
							  dbo.FNC_ResponseHeader() AS ResponseHeader ON QMS_Response.ResponseID = ResponseHeader.ResponseID
						WHERE     (QMS_Response.AssessmentTypeID IN
								  (SELECT     CONVERT(INT, Token) AS Expr1
									FROM          dbo.FNC_SplitString(@AssessmentTypeID_LIST, ','))) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding IS NULL) OR
							  (QMS_Response.AssessmentTypeID IN
								  (SELECT     CONVERT(INT, Token) AS Expr1
									FROM          dbo.FNC_SplitString(@AssessmentTypeID_LIST, ','))) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 1) AND 
							  (QMS_Response.ResponseStatus = 0) OR
							  (QMS_Response.AssessmentTypeID IN
								  (SELECT     CONVERT(INT, Token) AS Expr1
									FROM          dbo.FNC_SplitString(@AssessmentTypeID_LIST, ','))) AND (@AssessmentID = QMS_Response.AssessmentID OR
							  @AssessmentID IS NULL) AND (dbo.FNC_Response_IsMultiCheck(QMS_Response.ResponseID) IN (0, @multiCheckDisplay)) AND (@ShowOutstanding = 0) AND 
							  (QMS_Response.ResponseStatus <> 0)
						ORDER BY QMS_Response.EndDate DESC
                    END

--get list of columns that need to be hidden in Excel export
--NOTE:		tableindex is important, and zero based. This export table is table 1 because it's the second table returned in the dataset
--		FixedColumns: These are returned in the main body. These should be fixed. Leave this as NULL to use default ones specified by front end.
--		StaticColumns: These are displayed underneath the compact text in the grid, and do not move, but are specific for each customer.
--		CyclicColumns: If this contains the word "REMAINING_COLUMNS", then it will also include any columns not specified in Fixed or Static.
-- NOTE:	leabing static and cyclic columns NULL will use first two columns not in the default fixed columns as "static", and other columns not in either fixed or static as "cyclic"
        SELECT  1 AS TableIndex ,
                CONVERT(BIT, 1) AS ExportSetupTable ,
                'Portrait/Landscape' AS Orientation ,
                'n/a' AS SheetName ,
                'n/a' AS HiddenColumns ,
                'n/a' AS FixedColumns ,
                'n/a' AS StaticColumns ,
                'n/a' AS CyclicColumns
        UNION
        SELECT  0 ,
                0 ,
                'Portrait' ,
                'View Checks' ,
                NULL ,
                NULL ,
                'Location,FullName' ,
                'REMAINING_COLUMNS'
        ORDER BY TableIndex ASC

    END
GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_CorrectiveAction_OutstandingAll')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_CorrectiveAction_OutstandingAll] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 13/08/2008
-- Description:	Lists all oustanding corrective actions
-- Version: 7
-- =============================================
-- JE TFS 2758 - added filtering params
-- JE TFS 3003 - added join on FNC_User_CorrectiveActionList when filtering by Supplier
-- JE TFS 5642 - added join to FNC_User_CorrectiveActionList when filtering
-- JT TFS 6689: FNC_ReportFilter_Response takes UserID parameter.
-- JE TFS 7839 - added Location column
-- MP TFS 13-14225 - switched to use FNC_GENERATE_User_CorrectiveActionList
-- =============================================
ALTER PROCEDURE [dbo].[QRY_CorrectiveAction_OutstandingAll]

	@UserID int,
	@ReportFilterID int = NULL,
	@FilterID int = NULL,
	@FilterText nvarchar(255) = NULL

AS

--Lookup earliest CA date to pass to FNC_ReportFilter_Response
DECLARE @DateFrom datetime
SET @DateFrom = (SELECT	DATEADD(month, -1, MIN(DateOccurred))
					FROM	QMS_CorrectiveAction INNER JOIN
							APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID
					WHERE	(QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3))

--No CA filter
IF ISNULL(@ReportFilterID,0) = 0
	SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
                      CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, QMS_CorrectiveAction.CARequired, 
                      QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), 
                      APP_CorrectiveActionStatus.CAStatus) AS Status, APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, QMS_CorrectiveAction.DateLogged, 
                      QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level],
                      PM_VW_Location.LocationName AS Location, 
                      ISNULL(QMS_NotificationGroup.NotificationGroup,ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
                      QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.CAStatusID
	FROM         QMS_CorrectiveActionType RIGHT OUTER JOIN
                      QMS_CorrectiveAction INNER JOIN
                      dbo.FNC_GENERATE_User_CorrectiveActionList(@UserID, 1) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN
                      dbo.FNC_CorrectiveAction_Response() AS CAResponseList ON UserCAList.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
                      APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID INNER JOIN
                      QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
                      QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID ON 
                      QMS_CorrectiveActionType.CorrectiveActionTypeID = QMS_CorrectiveAction.CorrectiveActionTypeID LEFT OUTER JOIN
                      QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
                      QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
                      APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
                      PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
	WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3) AND (QMS_CorrectiveAction.DateOccurred >= @DateFrom)
	ORDER BY APP_PriorityLevel.ListIndex DESC, QMS_CorrectiveAction.DateDeadline

--CA filter has been set
IF @ReportFilterID IS NOT NULL
	IF @ReportFilterID = 1 --filtering by Supplier
		SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
							  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
							  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
							  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
							  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
							  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
							  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
							  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
							  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.CAStatusID
		FROM         QMS_CorrectiveActionType RIGHT OUTER JOIN
							  QMS_CorrectiveAction INNER JOIN
							  FNC_GENERATE_User_CorrectiveActionList(@UserID, 1) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN		
							  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
							  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID INNER JOIN
							  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
							  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID ON 
							  QMS_CorrectiveActionType.CorrectiveActionTypeID = QMS_CorrectiveAction.CorrectiveActionTypeID LEFT OUTER JOIN
							  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
							  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID				  
		WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3) AND (AssignSupplierID = @FilterID) AND (DateOccurred >= @DateFrom)
		ORDER BY APP_PriorityLevel.ListIndex DESC, QMS_CorrectiveAction.DateDeadline
	
	IF @ReportFilterID > 1 --not filtering by Supplier
		SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
							  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
							  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
							  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
							  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
							  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
							  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
							  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
							  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.CAStatusID
		FROM         QMS_CorrectiveActionType RIGHT OUTER JOIN
							  QMS_CorrectiveAction INNER JOIN
							  FNC_GENERATE_User_CorrectiveActionList(@UserID, 1) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN	
							  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
							  FNC_ReportFilter_Response(@ReportFilterID, @FilterID, @FilterText, @DateFrom, GETDATE(), @UserID) AS Filter ON
									CAResponseList.ResponseID = Filter.ResponseID INNER JOIN
							  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID INNER JOIN
							  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
							  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID ON 
							  QMS_CorrectiveActionType.CorrectiveActionTypeID = QMS_CorrectiveAction.CorrectiveActionTypeID LEFT OUTER JOIN
							  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
							  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID	 LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID			  
		WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3) AND (DateOccurred >= @DateFrom)
		ORDER BY APP_PriorityLevel.ListIndex DESC, QMS_CorrectiveAction.DateDeadline

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_CorrectiveAction_CompletedAll')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_CorrectiveAction_CompletedAll] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 13/08/2008
-- Description:	Lists all completed corrective actions
-- Version: 10
-- =============================================
-- JE TFS 2758 - added Filtering params
-- JE TFS 2798 - only returns CA's that are Completed (pending sign off)
-- JE TFS 2799 - added Date filtering
-- JE TFS 3003 - added join on FNC_User_CorrectiveActionList when filtering by Supplier
-- JE TFS 5642 - added join to FNC_User_CorrectiveActionList when filtering and
--		looked up FilterDateFrom so that CAs outside of response date range aren't excluded
-- JE TFS 5646 - added Reviewed column
-- JE TFS 5887 - removed duplicate Action Taken and Comments columns
-- JT TFS 6689: FNC_ReportFilter_Response takes UserID parameter.
-- JE TFS 7839 - added Location column
-- MP TFS 13-14225 - switched to use FNC_GENERATE_User_CorrectiveActionList
-- =============================================
ALTER PROCEDURE [dbo].[QRY_CorrectiveAction_CompletedAll]

	@UserID int,
	@ReportFilterID int = NULL,
	@FilterID int = NULL,
	@FilterText nvarchar(255) = NULL,
	@DateFrom datetime,
	@DateTo datetime

AS

--populate temp table with CA refs and their Reviewers' names
IF nullif(object_id('tempdb..#tmpCAReviewers'), 0) IS NOT NULL DROP TABLE #tmpCAReviewers
CREATE TABLE #tmpCAReviewers (
	CorrectiveActionID int,
	Reviewers nvarchar(500)
)
INSERT INTO #tmpCAReviewers
SELECT	CorrectiveActionID, dbo.FNC_CorrectiveAction_ReviewList(QMS_CorrectiveAction.CorrectiveActionID) AS Reviewers
FROM	QMS_CorrectiveAction

--No CA filter
IF ISNULL(@ReportFilterID,0) = 0
	SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
						  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
						  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
						  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
						  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
						  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
						  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
						  ISNULL(QMS_NotificationGroup.NotificationGroup,ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
						  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.DateCompleted, 
						  QMS_User_Completed.FullName AS [Completed by], QMS_CorrectiveAction.CAStatusID, CAReviewers.Reviewers AS [Reviewed by]
	FROM         QMS_CorrectiveAction INNER JOIN
						  FNC_GENERATE_User_CorrectiveActionList(@UserID, 0) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN
						  #tmpCAReviewers AS CAReviewers ON UserCAList.CorrectiveActionID = CAReviewers.CorrectiveActionID INNER JOIN
						  FNC_CorrectiveAction_Response() AS CAResponseList ON UserCAList.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN						  
						  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
						  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
						  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
						  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
						  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
						  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
						  QMS_User AS QMS_User_Completed ON QMS_CorrectiveAction.Completed_UserID = QMS_User_Completed.UserID LEFT OUTER JOIN
						  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
	WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID = 4) AND
					(QMS_CorrectiveAction.DateCompleted BETWEEN @DateFrom AND @DateTo)
	ORDER BY QMS_CorrectiveAction.DateCompleted DESC, APP_PriorityLevel.ListIndex DESC

--CA filter has been set
IF @ReportFilterID IS NOT NULL
	IF @ReportFilterID = 1 --filtering by Supplier
		SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
							  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
							  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
							  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
							  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
							  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
							  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
							  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
							  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.DateCompleted, 
							  QMS_User_Completed.FullName AS [Completed by], QMS_CorrectiveAction.CAStatusID, CAReviewers.Reviewers AS [Reviewed by]
		FROM         QMS_CorrectiveAction INNER JOIN
							  FNC_GENERATE_User_CorrectiveActionList(@UserID, 0) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN		
							  #tmpCAReviewers AS CAReviewers ON UserCAList.CorrectiveActionID = CAReviewers.CorrectiveActionID INNER JOIN
							  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
							  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
							  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
							  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
							  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Completed ON QMS_CorrectiveAction.Completed_UserID = QMS_User_Completed.UserID LEFT OUTER JOIN
							  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
							  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
		WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID = 4) AND (AssignSupplierID = @FilterID) AND
					(QMS_CorrectiveAction.DateCompleted BETWEEN @DateFrom AND @DateTo)
		ORDER BY QMS_CorrectiveAction.DateCompleted DESC, APP_PriorityLevel.ListIndex DESC	

	IF @ReportFilterID > 1 --not filtering by Supplier
	BEGIN
		--get earliest EndDate to use as DateFrom to pass to filter function
		DECLARE @FilterDateFrom as datetime
		SELECT	@FilterDateFrom = MIN(QMS_Response.EndDate)
		FROM	FNC_GENERATE_User_CorrectiveActionList(@UserID, 0) AS UserCAList INNER JOIN	
					FNC_CorrectiveAction_Response() AS CAResponseList ON UserCAList.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
					QMS_Response ON CAResponseList.ResponseID = QMS_Response.ResponseID
				
		SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
							  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
							  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
							  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
							  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
							  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
							  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
							  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
							  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.DateCompleted, 
							  QMS_User_Completed.FullName AS [Completed by], QMS_CorrectiveAction.CAStatusID, CAReviewers.Reviewers AS [Reviewed by]
		FROM         QMS_CorrectiveAction INNER JOIN
							  FNC_GENERATE_User_CorrectiveActionList(@UserID, 0) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN	
							  #tmpCAReviewers AS CAReviewers ON UserCAList.CorrectiveActionID = CAReviewers.CorrectiveActionID INNER JOIN
							  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
							  FNC_ReportFilter_Response(@ReportFilterID, @FilterID, @FilterText, @FilterDateFrom, @DateTo, @UserID) AS Filter ON
									CAResponseList.ResponseID = Filter.ResponseID INNER JOIN						  
							  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
							  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
							  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
							  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Completed ON QMS_CorrectiveAction.Completed_UserID = QMS_User_Completed.UserID LEFT OUTER JOIN
							  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
							  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
		WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID = 4) AND
					(QMS_CorrectiveAction.DateCompleted BETWEEN @DateFrom AND @DateTo)
		ORDER BY QMS_CorrectiveAction.DateCompleted DESC, APP_PriorityLevel.ListIndex DESC
	END

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_CorrectiveAction_FailedAll')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_CorrectiveAction_FailedAll] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 19/06/2009
-- Description:	Lists all failed & rejected corrective actions
-- Version: 13
-- =============================================
-- JE TFS 2758 - added filtering params
-- JE TFS 2799 - added Date filtering
-- JE TFS 5614 - changed filter for Rejected (StatusID 6) CA's so they don't need DateFailed set
-- JE TFS 5462 - added join to FNC_User_CorrectiveActionList when filtering and
--		looked up FilterDateFrom so that CAs outside of response date range aren't excluded
-- JE TFS 6039 - removed duplicate ActionTaken and Comments cols
-- JT TFS 6689 - FNC_ReportFilter_Response takes UserID parameter.
-- JE TFS 6046 - added filter on DateCompleted for rejected CAs as these won't have DateFailed set
-- JE TFS 7839 - added Location column
-- JE TFS 7586 - updated to take into account IsActive flag on FNC_User_CorrectiveActionList now returns Failed and Rejected when True
-- JE TFS 8188 - added local variables to get round performance issues associated with parameter sniffing
-- JE TFS 8188 - undid above change and added temp tables to hold results of functions
-- MP TFS 13-14225 - switched to use FNC_GENERATE_User_CorrectiveActionList
-- =============================================
ALTER PROCEDURE [dbo].[QRY_CorrectiveAction_FailedAll]

	@UserID int,
	@ReportFilterID int = NULL,
	@FilterID int = NULL,
	@FilterText nvarchar(255) = NULL,
	@DateFrom datetime,
	@DateTo datetime

AS

--test variables
--declare @UserID int, @ReportFilterID int, @FilterID int, @FilterText nvarchar(255), @DateFrom datetime, @DateTo datetime
--select @UserID = 1, @ReportFilterID = 6, @FilterID = 39, @FilterText = 'Ellesmere', @DateFrom = '25 Oct 2013 00:00:00', @DateTo = '25 Nov 2013 23:59:59'

--temp tables containing results from functions
IF nullif(object_id('tempdb..#tmpCAResponse'), 0) IS NOT NULL DROP TABLE #tmpCAResponse
SELECT	CorrectiveActionID, ResponseID
INTO	#tmpCAResponse
FROM	dbo.FNC_CorrectiveAction_Response()

IF nullif(object_id('tempdb..#tmpReportFilter'), 0) IS NOT NULL DROP TABLE #tmpReportFilter
SELECT	ResponseID
INTO	#tmpReportFilter
FROM	FNC_ReportFilter_Response(@ReportFilterID, @FilterID, @FilterText, @DateFrom, @DateTo, @UserID)

--temp table containing all CAs (i.e. all statuses) user has permission to see
IF nullif(object_id('tempdb..#tmpUserCAs'), 0) IS NOT NULL DROP TABLE #tmpUserCAs
SELECT	CorrectiveActionID
INTO	#tmpUserCAs
FROM	dbo.FNC_GENERATE_User_CorrectiveActionList(@UserID, 1)

INSERT INTO	#tmpUserCAs (CorrectiveActionID)
SELECT	CorrectiveActionID
FROM	dbo.FNC_GENERATE_User_CorrectiveActionList(@UserID, 0)

--No CA filter
IF ISNULL(@ReportFilterID,0) = 0
	SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
						  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
						  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
						  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
						  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
						  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
						  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
						  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
						  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.DateFailed, 
						  QMS_CorrectiveAction.DateCompleted, QMS_User_Completed.FullName AS [Completed by], QMS_CorrectiveAction.CAStatusID
	FROM         QMS_CorrectiveAction INNER JOIN
						  #tmpUserCAs AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN
						  FNC_CorrectiveAction_Response() AS CAResponseList ON UserCAList.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN						  
						  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
						  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
						  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
						  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
						  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
						  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
						  QMS_User AS QMS_User_Completed ON QMS_CorrectiveAction.Completed_UserID = QMS_User_Completed.UserID LEFT OUTER JOIN
						  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
	WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (((APP_CorrectiveActionStatus.CAStatusID = 3 OR
								APP_CorrectiveActionStatus.CAStatusID = 7) AND (QMS_CorrectiveAction.DateFailed BETWEEN @DateFrom AND @DateTo)) OR
								(APP_CorrectiveActionStatus.CAStatusID = 6) AND (QMS_CorrectiveAction.DateCompleted BETWEEN @DateFrom AND @DateTo))
	ORDER BY QMS_CorrectiveAction.DateFailed DESC, APP_PriorityLevel.ListIndex DESC

--CA filter has been set
IF @ReportFilterID IS NOT NULL
	IF @ReportFilterID = 1 --filtering by Supplier
		SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
							  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
							  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
							  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
							  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
							  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
							  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
							  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
							  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, QMS_CorrectiveAction.DateFailed, 
							  QMS_CorrectiveAction.DateCompleted, QMS_User_Completed.FullName AS [Completed by], QMS_CorrectiveAction.CAStatusID
		FROM         QMS_CorrectiveAction INNER JOIN
							  #tmpUserCAs AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN
							  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
							  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
							  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
							  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
							  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Completed ON QMS_CorrectiveAction.Completed_UserID = QMS_User_Completed.UserID LEFT OUTER JOIN
							  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
		WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (((APP_CorrectiveActionStatus.CAStatusID = 3 OR
								APP_CorrectiveActionStatus.CAStatusID = 7) AND (QMS_CorrectiveAction.DateFailed BETWEEN @DateFrom AND @DateTo)) OR
								(APP_CorrectiveActionStatus.CAStatusID = 6) AND (QMS_CorrectiveAction.DateCompleted BETWEEN @DateFrom AND @DateTo)) AND
							(AssignSupplierID = @FilterID)
		ORDER BY QMS_CorrectiveAction.DateFailed DESC, APP_PriorityLevel.ListIndex DESC

	IF @ReportFilterID > 1 --not filtering by Supplier
	BEGIN
		--get earliest EndDate to use as DateFrom to pass to filter function
		DECLARE @FilterDateFrom AS datetime
		SELECT	@FilterDateFrom = MIN(QMS_Response.EndDate)
		FROM	#tmpUserCAs AS UserCAList INNER JOIN	
					FNC_CorrectiveAction_Response() AS CAResponseList ON UserCAList.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
					QMS_Response ON CAResponseList.ResponseID = QMS_Response.ResponseID

		SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
							  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
							  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
							  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
							  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
							  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
							  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
							  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
							  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, QMS_CorrectiveAction.DateFailed, 
							  QMS_CorrectiveAction.DateCompleted, QMS_User_Completed.FullName AS [Completed by], QMS_CorrectiveAction.CAStatusID
		FROM         QMS_CorrectiveAction INNER JOIN
							  #tmpUserCAs AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN
							  #tmpCAResponse AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
							  #tmpReportFilter AS Filter ON CAResponseList.ResponseID = Filter.ResponseID INNER JOIN
							  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
							  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
							  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
							  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Completed ON QMS_CorrectiveAction.Completed_UserID = QMS_User_Completed.UserID LEFT OUTER JOIN
							  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
		WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (((APP_CorrectiveActionStatus.CAStatusID = 3 OR
								APP_CorrectiveActionStatus.CAStatusID = 7) AND (QMS_CorrectiveAction.DateFailed BETWEEN @DateFrom AND @DateTo)) OR
								(APP_CorrectiveActionStatus.CAStatusID = 6) AND (QMS_CorrectiveAction.DateCompleted BETWEEN @DateFrom AND @DateTo))
		ORDER BY QMS_CorrectiveAction.DateFailed DESC, APP_PriorityLevel.ListIndex DESC
	END

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_CorrectiveAction_OutstandingAssignedToMe')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_CorrectiveAction_OutstandingAssignedToMe] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 13/08/2008
-- Description:	Lists all oustanding corrective actions assigned to group that specified user is part of
-- Version: 7
-- =============================================
-- JE TFS 2757 - added supplier/location permission filtering using FNC_GENERATE_User_CorrectiveActionList
-- JE TFS 2758 - added filtering params
-- JE TFS 5642 - added join to FNC_User_CorrectiveActionList when filtering
-- JT TFS 6689: FNC_ReportFilter_Response takes UserID parameter.
-- JE TFS 7839 - added Location column
-- MP TFS 13-14225: switched to use FNC_GENERATE_User_CorrectiveActionList
-- =============================================
ALTER PROCEDURE [dbo].[QRY_CorrectiveAction_OutstandingAssignedToMe]
	@UserID int,
	@ReportFilterID int = NULL,
	@FilterID int = NULL,
	@FilterText nvarchar(255) = NULL
AS

--Lookup earliest CA date to pass to FNC_ReportFilter_Response
DECLARE @DateFrom datetime
SET @DateFrom = (SELECT DATEADD(month, -1, MIN(DateOccurred))
					FROM	QMS_CorrectiveAction INNER JOIN
							APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID INNER JOIN
						  dbo.FNC_CorrectiveAction_AssignUsers() AS AssignUser ON QMS_CorrectiveAction.CorrectiveActionID = AssignUser.CorrectiveActionID
					WHERE	(QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3)
								 AND (AssignUser.UserID = @UserID))

--No CA filter
IF ISNULL(@ReportFilterID,0) = 0
	SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
						  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
						  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
						  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
						  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
						  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
						  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
						  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
						  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.CAStatusID
	FROM        QMS_CorrectiveActionType RIGHT OUTER JOIN
						  QMS_CorrectiveAction INNER JOIN
						  FNC_GENERATE_User_CorrectiveActionList(@UserID, 1) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN
						  FNC_CorrectiveAction_Response() AS CAResponseList ON UserCAList.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN						  
						  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID INNER JOIN
						  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID INNER JOIN
						  dbo.FNC_CorrectiveAction_AssignUsers() AS AssignUser ON QMS_CorrectiveAction.CorrectiveActionID = AssignUser.CorrectiveActionID LEFT OUTER JOIN
						  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID ON 
						  QMS_CorrectiveActionType.CorrectiveActionTypeID = QMS_CorrectiveAction.CorrectiveActionTypeID LEFT OUTER JOIN
						  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
						  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
						  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
	WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3) AND (AssignUser.UserID = @UserID)
					AND (DateOccurred >= @DateFrom)
	ORDER BY APP_PriorityLevel.ListIndex DESC, QMS_CorrectiveAction.DateDeadline

--CA filter has been set
IF @ReportFilterID IS NOT NULL
BEGIN
	IF @ReportFilterID = 1 --filtering by Supplier
		SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
							  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
							  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
							  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
							  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
							  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
							  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName,
							  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
							  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.CAStatusID
		FROM         QMS_CorrectiveActionType RIGHT OUTER JOIN
							  QMS_CorrectiveAction INNER JOIN
							  FNC_GENERATE_User_CorrectiveActionList(@UserID, 1) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN
							  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
							  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID INNER JOIN
							  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
							  dbo.FNC_CorrectiveAction_AssignUsers() AS AssignUser ON QMS_CorrectiveAction.CorrectiveActionID = AssignUser.CorrectiveActionID LEFT OUTER JOIN
							  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID ON 
							  QMS_CorrectiveActionType.CorrectiveActionTypeID = QMS_CorrectiveAction.CorrectiveActionTypeID LEFT OUTER JOIN
							  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
							  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID	LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID 
		WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3) AND (AssignUser.UserID = @UserID) AND (AssignSupplierID = @FilterID)
					AND (DateOccurred >= @DateFrom)
		ORDER BY APP_PriorityLevel.ListIndex DESC, QMS_CorrectiveAction.DateDeadline
		
	IF @ReportFilterID > 1 --not filtering by Supplier
		SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
							  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
							  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
							  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
							  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
							  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
							  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName,
							  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
							  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.CAStatusID
		FROM         QMS_CorrectiveActionType RIGHT OUTER JOIN
							  QMS_CorrectiveAction INNER JOIN
							  FNC_GENERATE_User_CorrectiveActionList(@UserID, 1) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN
							  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
							  FNC_ReportFilter_Response(@ReportFilterID, @FilterID, @FilterText, @DateFrom, GETDATE(), @UserID) AS Filter ON
									CAResponseList.ResponseID = Filter.ResponseID INNER JOIN
							  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID INNER JOIN
							  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
							  dbo.FNC_CorrectiveAction_AssignUsers() AS AssignUser ON QMS_CorrectiveAction.CorrectiveActionID = AssignUser.CorrectiveActionID LEFT OUTER JOIN
							  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID ON 
							  QMS_CorrectiveActionType.CorrectiveActionTypeID = QMS_CorrectiveAction.CorrectiveActionTypeID LEFT OUTER JOIN
							  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
							  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID	LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
		WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3) AND (AssignUser.UserID = @UserID) AND (DateOccurred >= @DateFrom)
		ORDER BY APP_PriorityLevel.ListIndex DESC, QMS_CorrectiveAction.DateDeadline
END

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_CorrectiveAction_OutstandingLoggedByMe')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_CorrectiveAction_OutstandingLoggedByMe] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 13/08/2008
-- Description:	Lists all oustanding corrective actions logged by specified user
-- Version: 7
-- =============================================
-- JE TFS 2758 - added filtering params
-- JE TFS 3003 - added join on FNC_User_CorrectiveActionList when filtering by Supplier
-- JE TFS 5642 - added join to FNC_User_CorrectiveActionList when filtering
-- JT TFS 6689: FNC_ReportFilter_Response takes UserID parameter.
-- JE TFS 7839 - added Location column
-- MP TFS 13-14225 - switched to use FNC_GENERATE_User_CorrectiveActionList function
-- =============================================
ALTER PROCEDURE [dbo].[QRY_CorrectiveAction_OutstandingLoggedByMe]
	@UserID int,
	@ReportFilterID int = NULL,
	@FilterID int = NULL,
	@FilterText nvarchar(255) = NULL
AS

--Lookup earliest CA date to pass to FNC_ReportFilter_Response
DECLARE @DateFrom datetime
SET @DateFrom = (SELECT	DATEADD(month, -1, MIN(DateOccurred))
					FROM	QMS_CorrectiveAction INNER JOIN
							APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID
					WHERE	(QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3)
								 AND (QMS_CorrectiveAction.Logged_UserID = @UserID))

--No CA filter
IF ISNULL(@ReportFilterID,0) = 0
	SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
						  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
						  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
						  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
						  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
						  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
						  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
						  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], QMS_CorrectiveAction.CAStatusID
	FROM         QMS_CorrectiveAction INNER JOIN
						  FNC_GENERATE_User_CorrectiveActionList(@UserID, 1) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN
						  FNC_CorrectiveAction_Response() AS CAResponseList ON UserCAList.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
						  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
						  QMS_User ON QMS_CorrectiveAction.AssignUserID = QMS_User.UserID LEFT OUTER JOIN
						  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
						  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
						  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
						  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
	WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3) AND (QMS_CorrectiveAction.Logged_UserID = @UserID)
				AND (DateOccurred >= @DateFrom)
	ORDER BY APP_PriorityLevel.ListIndex DESC, QMS_CorrectiveAction.DateDeadline

--CA filter has been set
IF @ReportFilterID IS NOT NULL
	IF @ReportFilterID = 1 --filtering by Supplier
	SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
						  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
						  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
						  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
						  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
						  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
						  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
						  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], QMS_CorrectiveAction.CAStatusID
	FROM         QMS_CorrectiveAction INNER JOIN
						  FNC_GENERATE_User_CorrectiveActionList(@UserID, 1) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN		
						  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
						  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
						  QMS_User ON QMS_CorrectiveAction.AssignUserID = QMS_User.UserID LEFT OUTER JOIN
						  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
						  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
						  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
						  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
	WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3) AND (QMS_CorrectiveAction.Logged_UserID = @UserID) AND (AssignSupplierID = @FilterID)
				AND (DateOccurred >= @DateFrom)
	ORDER BY APP_PriorityLevel.ListIndex DESC, QMS_CorrectiveAction.DateDeadline


IF @ReportFilterID > 1 --not filtering by Supplier
	SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
						  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
						  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
						  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
						  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
						  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
						  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
						  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], QMS_CorrectiveAction.CAStatusID
	FROM         QMS_CorrectiveAction INNER JOIN
						  FNC_GENERATE_User_CorrectiveActionList(@UserID, 1) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN		
						  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
						  FNC_ReportFilter_Response(@ReportFilterID, @FilterID, @FilterText, @DateFrom, GETDATE(), @UserID) AS Filter ON
								CAResponseList.ResponseID = Filter.ResponseID INNER JOIN
						  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
						  QMS_User ON QMS_CorrectiveAction.AssignUserID = QMS_User.UserID LEFT OUTER JOIN
						  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
						  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
						  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
						  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
	WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID < 3) AND (QMS_CorrectiveAction.Logged_UserID = @UserID) AND
							(DateOccurred >= @DateFrom)
	ORDER BY APP_PriorityLevel.ListIndex DESC, QMS_CorrectiveAction.DateDeadline

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_CorrectiveAction_SignedOffAll')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_CorrectiveAction_SignedOffAll] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Jon Edwards
-- Create date: 16/02/2012
-- Description:	Lists all Signed Off corrective actions
-- Version: 8
-- =============================================
-- JE TFS 2799 - added Date filtering
-- JE TFS 3003 - added join on FNC_User_CorrectiveActionList when filtering by Supplier
-- JE TFS 5642 - added join to FNC_User_CorrectiveActionList when filtering and
--		looked up FilterDateFrom so that CAs outside of response date range aren't excluded
-- JE TFS 5646 - added Reviewed column
-- JE TFS 5872 - added Signed Off by and Signed Off date/time cols
-- JE TFS 5887 - removed duplicate Action Taken and Comments columns
-- JT TFS 6689: FNC_ReportFilter_Response takes UserID parameter.
-- JE TFS 7839 - added Location column
-- MP TFS 13-14225 - switched to use FNC_GENERATE_User_CorrectiveActionList
-- =============================================
ALTER PROCEDURE [dbo].[QRY_CorrectiveAction_SignedOffAll]

	@UserID int,
	@ReportFilterID int = NULL,
	@FilterID int = NULL,
	@FilterText nvarchar(255) = NULL,
	@DateFrom datetime,
	@DateTo datetime

AS

--populate temp table with CA refs and their Reviewers' names
IF nullif(object_id('tempdb..#tmpCAReviewers'), 0) IS NOT NULL DROP TABLE #tmpCAReviewers
CREATE TABLE #tmpCAReviewers (
	CorrectiveActionID int,
	Reviewers nvarchar(500)
)
INSERT INTO #tmpCAReviewers
SELECT	CorrectiveActionID, dbo.FNC_CorrectiveAction_ReviewList(QMS_CorrectiveAction.CorrectiveActionID) AS Reviewers
FROM	QMS_CorrectiveAction

--No CA filter
IF ISNULL(@ReportFilterID,0) = 0
	SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
						  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
						  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
						  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
						  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
						  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
						  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
						  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
						  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.DateCompleted, 
						  QMS_User_Completed.FullName AS [Completed by], QMS_CorrectiveAction.CAStatusID, CAReviewers.Reviewers AS [Reviewed by],
						  QMS_User_SignedOff.FullName AS [Signed off by], QMS_CorrectiveAction.DateSignedOff AS [Signed off date/time]
	FROM         QMS_CorrectiveAction INNER JOIN
						  FNC_GENERATE_User_CorrectiveActionList(@UserID, 0) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN
						  #tmpCAReviewers AS CAReviewers ON UserCAList.CorrectiveActionID = CAReviewers.CorrectiveActionID INNER JOIN
						  FNC_CorrectiveAction_Response() AS CAResponseList ON UserCAList.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN						  
						  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
						  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
						  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
						  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
						  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
						  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
						  QMS_User AS QMS_User_Completed ON QMS_CorrectiveAction.Completed_UserID = QMS_User_Completed.UserID LEFT OUTER JOIN
						  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
						  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
						  QMS_User AS QMS_User_SignedOff ON QMS_CorrectiveAction.SignOff_UserID = QMS_User_SignedOff.UserID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
	WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID = 5) AND
					(QMS_CorrectiveAction.DateCompleted BETWEEN @DateFrom AND @DateTo)
	ORDER BY QMS_CorrectiveAction.DateCompleted DESC, APP_PriorityLevel.ListIndex DESC

--CA filter has been set
IF @ReportFilterID IS NOT NULL
	IF @ReportFilterID = 1 --filtering by Supplier
		SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
							  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
							  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
							  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
							  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
							  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
							  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
							  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
							  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.DateCompleted, 
							  QMS_User_Completed.FullName AS [Completed by], QMS_CorrectiveAction.CAStatusID, CAReviewers.Reviewers AS [Reviewed by],
							  QMS_User_SignedOff.FullName AS [Signed off by], QMS_CorrectiveAction.DateSignedOff AS [Signed off date/time]
		FROM         QMS_CorrectiveAction INNER JOIN
							  FNC_GENERATE_User_CorrectiveActionList(@UserID, 0) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN		
							  #tmpCAReviewers AS CAReviewers ON UserCAList.CorrectiveActionID = CAReviewers.CorrectiveActionID INNER JOIN
							  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
							  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
							  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
							  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
							  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Completed ON QMS_CorrectiveAction.Completed_UserID = QMS_User_Completed.UserID LEFT OUTER JOIN
							  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
							  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
							  QMS_User AS QMS_User_SignedOff ON QMS_CorrectiveAction.SignOff_UserID = QMS_User_SignedOff.UserID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
		WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID = 5) AND (AssignSupplierID = @FilterID) AND
					(QMS_CorrectiveAction.DateCompleted BETWEEN @DateFrom AND @DateTo)
		ORDER BY QMS_CorrectiveAction.DateCompleted DESC, APP_PriorityLevel.ListIndex DESC	

	IF @ReportFilterID > 1 --not filtering by Supplier
	BEGIN
		--get earliest EndDate to use as DateFrom to pass to filter function
		DECLARE @FilterDateFrom AS datetime
		SELECT	@FilterDateFrom = MIN(QMS_Response.EndDate)
		FROM	FNC_GENERATE_User_CorrectiveActionList(@UserID, 0) AS UserCAList INNER JOIN	
					FNC_CorrectiveAction_Response() AS CAResponseList ON UserCAList.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
					QMS_Response ON CAResponseList.ResponseID = QMS_Response.ResponseID
					
		SELECT     CAST(0 AS bit) AS [Select], QMS_CorrectiveAction.CorrectiveActionID, dbo.FNC_GetCorrectiveActionIncrement(QMS_CorrectiveAction.CorrectiveActionID) AS CARef, 
							  CAResponseList.ResponseID, QMS_CorrectiveAction.CATitle, QMS_CorrectiveAction.CASubTitle, QMS_CorrectiveAction.CAReason, 
							  QMS_CorrectiveAction.CARequired, QMS_CorrectiveAction.ActionTaken, QMS_CorrectiveAction.Comments, 
							  ISNULL(dbo.FNC_ResourceString_User(APP_CorrectiveActionStatus.ResourceName, @UserID), APP_CorrectiveActionStatus.CAStatus) AS Status, 
							  APP_PriorityLevel.PriorityLevel, QMS_CorrectiveAction.DateOccurred, 
							  QMS_CorrectiveAction.DateLogged, QMS_CorrectiveAction.DateDeadline, QMS_CorrectiveActionType.CorrectiveActionType, 
							  QMS_CorrectiveActionEscalation.ListIndex AS [Escalation level], PM_VW_Location.LocationName AS Location,
							  ISNULL(QMS_NotificationGroup.NotificationGroup, ISNULL(QMS_User_Assign.FullName, Supplier_Assign.SupplierName)) AS [Assigned to], 
							  QMS_User_Logged.FullName AS [Logged by], QMS_CorrectiveAction.DateCompleted, 
							  QMS_User_Completed.FullName AS [Completed by], QMS_CorrectiveAction.CAStatusID, CAReviewers.Reviewers AS [Reviewed by],
							  QMS_User_SignedOff.FullName AS [Signed off by], QMS_CorrectiveAction.DateSignedOff AS [Signed off date/time]
		FROM         QMS_CorrectiveAction INNER JOIN
							  FNC_GENERATE_User_CorrectiveActionList(@UserID, 0) AS UserCAList ON QMS_CorrectiveAction.CorrectiveActionID = UserCAList.CorrectiveActionID INNER JOIN	
							  #tmpCAReviewers AS CAReviewers ON UserCAList.CorrectiveActionID = CAReviewers.CorrectiveActionID INNER JOIN
							  FNC_CorrectiveAction_Response() AS CAResponseList ON QMS_CorrectiveAction.CorrectiveActionID = CAResponseList.CorrectiveActionID INNER JOIN
							  FNC_ReportFilter_Response(@ReportFilterID, @FilterID, @FilterText, @FilterDateFrom, @DateTo, @UserID) AS Filter ON
									CAResponseList.ResponseID = Filter.ResponseID INNER JOIN						  
							  APP_CorrectiveActionStatus ON QMS_CorrectiveAction.CAStatusID = APP_CorrectiveActionStatus.CAStatusID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Assign ON QMS_CorrectiveAction.AssignUserID = QMS_User_Assign.UserID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Logged ON QMS_CorrectiveAction.Logged_UserID = QMS_User_Logged.UserID LEFT OUTER JOIN
							  QMS_NotificationGroup ON QMS_CorrectiveAction.NotificationGroupID = QMS_NotificationGroup.NotificationGroupID LEFT OUTER JOIN
							  QMS_CorrectiveActionEscalation ON QMS_CorrectiveAction.CorrectiveActionEscalationID = QMS_CorrectiveActionEscalation.CorrectiveActionEscalationID LEFT OUTER JOIN
							  QMS_CorrectiveActionType ON QMS_CorrectiveAction.CorrectiveActionTypeID = QMS_CorrectiveActionType.CorrectiveActionTypeID LEFT OUTER JOIN
							  QMS_User AS QMS_User_Completed ON QMS_CorrectiveAction.Completed_UserID = QMS_User_Completed.UserID LEFT OUTER JOIN
							  APP_PriorityLevel ON QMS_CorrectiveAction.PriorityLevelID = APP_PriorityLevel.PriorityLevelID LEFT OUTER JOIN
							  PM_VW_Supplier AS Supplier_Assign ON QMS_CorrectiveAction.AssignSupplierID = Supplier_Assign.SupplierID LEFT OUTER JOIN
							  QMS_User AS QMS_User_SignedOff ON QMS_CorrectiveAction.SignOff_UserID = QMS_User_SignedOff.UserID LEFT OUTER JOIN
						  PM_VW_Location ON QMS_CorrectiveAction.LocationID = PM_VW_Location.LocationID
		WHERE     (QMS_CorrectiveAction.RecordStatus < 201) AND (APP_CorrectiveActionStatus.CAStatusID = 5) AND
					(QMS_CorrectiveAction.DateCompleted BETWEEN @DateFrom AND @DateTo)
		ORDER BY QMS_CorrectiveAction.DateCompleted DESC, APP_PriorityLevel.ListIndex DESC
	END

GO

IF NOT EXISTS (SELECT * FROM INFORMATION_SCHEMA.ROUTINES WHERE ROUTINE_NAME = 'QRY_ResponseQuestionSample_ResponseList')
	EXEC ('CREATE PROCEDURE [dbo].[QRY_ResponseQuestionSample_ResponseList] AS SELECT ''stub version, to be replaced''')
GO
-- =============================================
-- Author:		Markp
-- Create date: 25/07/2008
-- Description:	Gets the samples for a full assessment response
-- Version:		4
-- =============================================
-- JE TFS 6272 - returns Child Check samples if using Multi Checks
-- JT TFS 13-15888 - speed increase by simplifying main query
-- jf hotfix - remove the ResponseID output
-- =============================================
ALTER PROCEDURE [dbo].[QRY_ResponseQuestionSample_ResponseList] 
	@ResponseID int
AS

--get ParentResponseGUID for ResponseID passed in
DECLARE @ParentResponseGUID uniqueidentifier
SELECT	@ParentResponseGUID = GUID
FROM	QMS_Response
WHERE	ResponseID = @ResponseID

--temp table to hold Child Check ResponseIDs
IF NULLIF(object_id('tempdb..#tmpChildChecks'), 0) IS NOT NULL DROP TABLE #tmpChildChecks
CREATE TABLE #tmpChildChecks (ResponseID int)

--get all Child Checks for above GUID
--if there are Child Checks, get samples for every check
INSERT	#tmpChildChecks (ResponseID)
SELECT	ResponseID
FROM	QMS_Response
WHERE	ParentResponseGUID = @ParentResponseGUID

-- If there are no children, then we want the one main ResponseID
IF NOT EXISTS (SELECT ResponseID FROM #tmpChildChecks)
	INSERT #tmpChildChecks (ResponseID) VALUES (@ResponseID)

--get Child Check count
DECLARE @ChildCheckCount int
SELECT @ChildCheckCount = COUNT(ResponseID) FROM #tmpChildChecks

SELECT     QMS_ResponseQuestion.ResponseQuestionID, QMS_ResponseQuestion.AssessmentQuestionID, QMS_ResponseQuestionSample.SampleNo, 
                      QMS_ResponseQuestionSample.ResponseSampleID, QMS_ResponseQuestionSample.Sample_Boolean, QMS_ResponseQuestionSample.Sample_Date, 
                      QMS_ResponseQuestionSample.Sample_Numeric, QMS_ResponseQuestionSample.Sample_SourceID, 
                      QMS_ResponseQuestionSample.Recommended_QAFlagID

					  --,QMS_Response.ResponseID
FROM         QMS_ResponseQuestionSample INNER JOIN
                      QMS_ResponseQuestion ON QMS_ResponseQuestionSample.ResponseQuestionID = QMS_ResponseQuestion.ResponseQuestionID INNER JOIN
                      QMS_Response ON QMS_ResponseQuestion.ResponseID = QMS_Response.ResponseID
WHERE   (QMS_ResponseQuestion.RecordStatus < 201) 
AND		(QMS_ResponseQuestionSample.RecordStatus < 201) 
AND		(QMS_Response.ResponseID IN (SELECT ResponseID FROM #tmpChildChecks)) 
ORDER BY QMS_ResponseQuestionSample.SampleNo
GO



--PRINT 'Updating sync procedures...'

--PRINT 'Data migration tasks...'

PRINT 'Database updated successfully!'
GO
